#!/usr/bin/python2
#
#********************************************************
#
# Name: docker-control.py
#
# (c) Copyright International Business Machines Corp 2017.
# US Government Users Restricted Rights - Use, duplication or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.
#
# This is a script to send a signal a job. The script is started when there is a signal to send to the job. 
# After the script is done, the signal is send to the other process of the job.
#********************************************************

import os 
import sys
import time
import subprocess
import logging
import glob

#append lsfdockerlib into sys.path
lsfserverdir = os.environ.get("LSF_SERVERDIR", None)

if lsfserverdir == None:
    sys.stderr.write("LSF_SERVERDIR is not defined, set LSF environment variable and try again\n")
    os._exit(254)

try:
    sys.path.append(lsfserverdir)
    from lsfdockerlib import *
except Exception as e:
    sys.stderr.write("Import error: %s, try lsfcontainerlib\n" % str(e))

    try:
        from lsfcontainerlib import *
    except Exception as e:
        sys.stderr.write("Import error: %s\n" % str(e))
        os._exit(253)

LSB_RELEASE_DATE = "2020-12-04 10:36:09"
LSB_BUILD_NUMBER= "564512"

logger = None

def sig_docker_container_processes(uuid, sig):
    
    logger.log("sig docker container process")
    cgdict = CGHelper.get_cgroup_path()
    
    if 'memory' not in cgdict.keys():
        logger.log("cgroup for memory is not enabled. exiting.")
        return None

    mempath = cgdict.get('memory')
    logger.log("cgroup memory path: %s" % mempath)

    cgparent = os.environ.get("LSB_JOB_CGROUP_PARENT", None)
    container_memcg_path = "%s/%s/%s/" % (mempath, cgparent, uuid)
    if not os.path.exists(container_memcg_path):
        container_memcg_path = "%s/system.slice/docker-%s.scope/" % (mempath, uuid)

        if os.path.exists(mempath):
            container_memcg_path = "%s/docker/%s/" % (mempath, uuid)
            if not os.path.exists(container_memcg_path):
                return None

    filename = "%s/cgroup.procs" % container_memcg_path
    logger.log("cgroup.procs: %s" % filename)
    
    try:
        fp = open(filename, 'r')
        content = fp.readlines()
        fp.close()
    except Exception as e:
        logger.log("open file: %s exception: %s " % (filename, str(e)))
        return None

    if len(content) < 1:
        logger.log("cgroup.procs content is empty!!!!")
        lsbexit(-1)
    
    strpids = "".join(content).replace("\n", " ") 
    logger.log("pids: %s" % strpids)

    if is_str_empty(strpids):
        logger.log("container pids is empty, exit -1")
        lsbexit(-1)

    cmd = "kill -%s %s" % (sig, strpids)
    logger.log("kill docker container cmd: %s" % cmd)

    try:
        out, err, rc = runcmd(cmd)
    except Exception as e:
        logger.log("out:\n%s\nerr:\n%s\nrc:%s\nException:\n%s" % ( "".join(out), "".join(err), rc, str(e)))
    finally:
        logger.log("out:\n%s\nerr:\n%s\nrc:%s" % ("".join(out), "".join(err), rc))

    lsbexit(rc)

def sig_podman_container_processes(uuid, sig):
    logger.log("in kill podman_container_processes, container uuid: %s" % uuid)

    if sig == 28: #ignore SIGWIN
        return 0

    if sig in [ '2', '3', '12', '15'  ]:
        sig = '9'

    #/sys/fs/cgroup/systemd/user.slice/user-33857.slice/user@33857.service/user.slice/podman-26313.scope/<cid>
    cgpath = CGHelper.get_root_cgroup()
    globptn = '%s/*/*/*/*/*/*/%s' % (cgpath, uuid)
    container_path_list = glob.glob(globptn)

    if len(container_path_list) < 1:
        logger.log("no container path found for container: %s" % uuid)
        return -1
    
    pidlist = []
    for container_path in container_path_list:

        pidfile = "%s/cgroup.procs" % container_path

        logger.log("pidfile: %s" % pidfile)

        fp = None
        content = []
        
        try:
            fp = open (pidfile, "r")
            content = fp.readlines();
            fp.close()
        except Exception as e:
            logger.log("Exception when opening file: %s, %s" % (pidfile, str(e)))
            return -1

        finally:
            fp.close()
    
        #content would like: ['795\n', '808\n']
        logger.log("cgroup.procs content: %s" % str(content))
        pidlist.extend(content)

    if len(pidlist) < 1:
        logger.log("cgroup.procs content is empty!!!!")
        return -1

    #unique the pid list
    pids = list(set(pidlist))

    strpids = "".join(content).replace("\n", " ") 
    logger.log("pids: %s" % strpids)

    if is_str_empty(pids):
        logger.log("container pids is empty, exit -1")
        return -1
    
    try:
        out, err, rc = runcmd("kill -%s %s" % (sig, strpids))
    except Exception as e:
        logger.log("out:\n%s\nerr:\n%s\nrc:%s\nException:%s" % ("".join(out), "".join(err), rc, str(e)))

    finally:
        logger.log("out:\n%s\nerr:\n%s\nrc:%s" % ("".join(out), "".join(err), rc))

    return rc

def spillbox_copylog():
    
    site = os.environ.get("SPILLBOX_SITE", "")
    cloud_user = os.environ.get("SPILLBOX_CLOUD_USER", "")
    project = os.environ.get("SPILLBOX_PROJECT", "")
    cache_dir = os.environ.get("SPILLBOX_CACHE_DIR", "")
    splapp_stdout = os.environ.get("SPLAPP_STDOUT", "")
    splapp_stderr = os.environ.get("SPLAPP_STDERR", "")
    jobid = os.environ.get('LSB_JOBID', None)
    jobindex = os.environ.get('LSB_JOBINDEX', None)
    workdir = os.environ.get("SPILLBOX_WORKDIR", "")
    remoteworkdir = os.environ.get("SPILLBOX_REMOTE_WORKDIR", "")
    home = os.environ.get("SPILLBOX_HOME", "")
    image = os.environ.get("SPILLBOX_DOCKER_IMAGE", "")
    postexec = "/usr/bin/docker run --rm --privileged --env=SPILLBOX_SITE=%s --env=SPILLBOX_PROJECT=%s --env=SPILLBOX_CLOUD_USER=%s --env=SPILLBOX_CACHE_DIR=%s --env=SPLAPP_STDOUT=%s --env=SPLAPP_STDERR=%s --env=LSB_JOBID=%s --env=LSB_JOBINDEX=%s --env=SPILLBOX_WORKDIR=%s --env=SPILLBOX_HOME=%s -v %s/.spillbox/sites/local/%s/terraform/bin:/spillbox -v %s:/spillbox_workdir --entrypoint=/spillbox/bapp_entrypoint.sh -i %s /spillbox/lsf_post_exec.sh" % (site, project, cloud_user, cache_dir, splapp_stdout, splapp_stderr, jobid, jobindex, workdir, home, remoteworkdir, site, remoteworkdir, image) 
    logger.log("postexec: %s" % (postexec))
    try:
        out, err, rc = runcmd(postexec)
    except Exception as e:
        logger.log("out:\n%s\nerr:\n%s\nrc:%s\nException:%s" % ("".join(out), "".join(err), rc, str(e)))

    finally:
        logger.log("out:\n%s\nerr:\n%s\nrc:%s" % ("".join(out), "".join(err), rc))
    return rc

def sig_container():
    
    sig = sys.argv[1]

    if sig is None:
        return -1

    uuidfile = os.environ.get('LSB_EXEC_DRIVER_ID', None)
    logger.log("uuidfile: %s" % uuidfile)
    try:
        fp = open(uuidfile, 'r')
    except Exception as e:
        logger.log("I/O error in opening file %s: %s" % (uuidfile, str(e)))
        return -1

    uuid = fp.read()
    fp.close()

    if is_str_empty(uuid):
        logger.log("uuid is not found, exit -1")
        return -1

    logger.log("uuid is %s, signal is: %s" % (uuid, sig))
    
    #podman related routine
    if ispodman():
        logger.log("is podman: True")
        ret = sig_podman_container_processes(uuid, sig)
        return ret
    
    #docker
    cmd = None

    sigdict = {
            '9'    : 'kill',
            '18'   : 'unpause',
            '19'   : 'pause',
            '10000': 'kill'
            }
    
    #if signal is in sigdict, use docker command, else signal process in container directly.
    if sig in sigdict.keys():
        cmd = "/usr/bin/docker %s %s" % (sigdict[sig], uuid)
    else:
        logger.log("signal: %s, send to container directly" % sig)
        ret = sig_docker_container_processes(uuid, sig)
        return ret

    logger.log("docker cmd is: %s" % cmd)

    out, err, ret = runcmd(cmd)

    if ret != 0:
        logger.log("cmd: %s, err: %s, out: %s" % (ret, err, out))

    return ret

if __name__ == "__main__":
    if len(sys.argv) == 1:
        ret = show_file_info(sys.argv[0])
        lsbexit(ret)

    uuidfile = os.environ.get('LSB_EXEC_DRIVER_ID', None)
    #uuidfile:/tmp/lsf.ulaworks004.job.131.1576083749
    if is_str_empty(uuidfile):
        lsbexit(-1)
    
    filename = uuidfile.split('/')[-1]
    
    logger = DriverLogger("controller", filename,  "debug")
    logger.init()
    spillbox_copylog()
    logger.log("my uid: %s, euid: %s" % (os.getuid(), os.geteuid()))
    if sig_container() < 0 :
        lsbexit(-1)
