#!/usr/bin/python2
#
#********************************************************
#
# Name: docker-monitor.py
#
# (c) Copyright International Business Machines Corp 2017.
# US Government Users Restricted Rights - Use, duplication or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.
#
# This is a script to monitor resources usage of a job. This script should be alive during the job execution. 
# When the monitor script is exited during job execution, LSF restarts the monitor script every 5 seconds. 
# The collected resources must have: pids, utime, stime, mem, swp, tmp, and nthreads.
#********************************************************

import sys
import os
import math
import logging
import subprocess
import signal
import string
import time
import sqlite3

#append lsfdockerlib into sys.path
lsfserverdir = os.environ.get("LSF_SERVERDIR", None)

if lsfserverdir == None:
    sys.stderr.write("LSF_SERVERDIR is not defined, set LSF environment variable and try again\n")
    os._exit(254)

try:
    sys.path.append(lsfserverdir)
    from lsfdockerlib import *
except Exception as e:
    sys.stderr.write("Import error: %s, try lsfcontainerlib\n" % str(e))

    try:
        from lsfcontainerlib import *
    except Exception as e:
        sys.stderr.write("Import error: %s\n" % str(e))
        os._exit(253)

LSB_RELEASE_DATE = "2020-12-04 10:36:09"
LSB_BUILD_NUMBER= "564512"

logger = None

#job related vars
driver_user = os.environ.get('LSB_EXEC_DRIVER_USER', None)
dbname      = os.environ.get("LSB_EXEC_DRIVER_DB", None)
driverid    = os.environ.get("LSB_EXEC_DRIVER_ID", None)
uuidfile    = driverid

jobid       = os.environ.get('LSB_JOBID', None)
jobindex    = os.environ.get('LSB_JOBINDEX', None)

taskindex   = os.environ.get('LSF_PM_TASKID', None)
taskts      = os.environ.get('LSB_CONTAINER_TSK_TIMESTAMP', None) #tasktimestamp

cgtype = None

#monitor poll interval
interval    = int(os.environ.get('LSB_DOCKER_MONITOR_INTERVAL', '1'))


def validate_initvars():

    if ispodman():
        logger.log("podman job, driver monitor ignored, exit" % cgtype)
        lsbexit(0)

    initvars = [driver_user, dbname, driverid, jobid, cgtype]
    if None in initvars:
        logger.log("init vars validate error: %s" % (str(initvars)))
        lsbexit(-1)

    if not os.path.exists(dbname) :
        logger.log("database %s doesn't exist" % dbname)
        lsbexit(-1)

#misc functions 
def check_parent():
    if (os.getppid() == 1) : # job exited, so monitor is ready to exit
        updatets("monitor exit")
        lsbexit(0)

#db functions
def initdb(tablename, resources):
    logger.log("connect to %s to init" % dbname)

    conn = sqlite3.connect(dbname)
    cursor = conn.cursor()
    values = []
    
    sql = 'SELECT * FROM %s WHERE jobid = "%s"' % (tablename, jobid)

    cursor.execute(sql)
    logger.log(sql)
    values = cursor.fetchall()

    if len(values) == 0 :
        sql = 'INSERT INTO %s VALUES ("%s", "%s", %d, %d, %d, %d, %d, %d);' % \
              (tablename, jobid, resources['pids'], \
               int(resources['utime']), int(resources['stime']), \
               int(resources['mem']), int(resources['swp']), int(resources['tmp']), \
               int(resources['nthreads']))
        try:
            cursor.execute("BEGIN TRANSACTION")
            cursor.execute(sql)
            conn.commit()
            logger.log(sql)
        except Exception as e:
            conn.rollback()
            logger.log("cannot execute sql: %s,: exception: %s" % (sql, str(e)))

    conn.close()

def updatedb(tablename, resources):
    logger.log("update db %s" % dbname)

    pids = resources['pids'].split()

    try:
        if (int)(pids[0]) > 0 :
            pidstr = '1 %s %d %s' % (pids[0], len(pids), resources['pids'])
        else :
            pidstr = '0'
    except IndexError:
        pidstr = '0'

    conn = sqlite3.connect(dbname)
    cursor = conn.cursor()

    sql = '''
            UPDATE %s 
            SET pids = "%s", utime = %d, stime = %d, mem = %d, swp = %d, tmp = %d, nthreads = %d 
            WHERE jobid = "%s"
          ''' % (tablename, pidstr, int(resources['utime']), int(resources['stime']), int(resources['mem']),
                int(resources['swp']), int(resources['tmp']), int(resources['nthreads']), jobid)

    logger.log(sql)

    try:
        cursor.execute("BEGIN TRANSACTION")
        cursor.execute(sql)
        conn.commit()
    except Exception as e:
        conn.rollback()
        logger.log("cannot execute sql: %s,: exception: %s" % (sql, str(e)))

    conn.close()

def get_container_cgpath(cid, initpath='/sys/fs/cgroup'):
    
    #find /sys/fs/cgroup/cpuacct -iname '*45ef7c516a1ab6c39d62c997d14d46b66ceaa49420ee607b9a7fc81a907a73c4*'
    cmd = "find %s/ -iname '*%s*'" % (initpath, cid)
    out, err, rc = runcmd(cmd)
    logger.log("cmd: %s\nout:\n%s\nerr:\n%s\nret: %s\n" % (cmd, "\n".join(out), "\n".join(err), rc))

    #remove empty lines
    out = [ line for line in out if  len(line.strip()) != 0 ] 

    if (rc != 0) or (len(out) != 1):
        logger.log("error when getting cgpath for container: %s" % cid)
        lsbexit(-1)

    return out[0]

class DriverMonitor:
    def __init__(self):
        self.jobdbid = jobid
        self.tablename = 'jobtable'
        self.uuid = ''

        if jobindex is not None and jobindex != "0" :
            self.jobdbid = "%s[%s]" % (self.jobdbid, jobindex)

        if taskindex is not None and taskindex != "0" :
            self.jobdbid = "%s_%s" % (self.jobdbid, taskindex)

        if taskindex is not None and self.taskts is not None :
            self.uuidfile = "%s.task.%s.%s" % (self.uuidfile, taskindex, taskts)
            self.tablename = 'tasktable'

    def start(self):
        if self.__collect_rusage() < 0 :
            return -1

    def __collect_rusage(self):
        resources = {'pids' : '0', 'utime' : '0', 'stime' : '0', 'mem' : '0' , 'swp' : '0' , 'tmp' : '0' , 'nthreads' : '0'}

        try:
            initdb(self.tablename, resources)
        except Exception as e:
            logger.log("failed to init db: %s." % str(e))
            return -1

        cgpath_dict = CGHelper.get_cgroup_path()
        memcg = cgpath_dict.get('memory', None)
        cpucg = cgpath_dict.get('cpuacct', None)

        if None in [ memcg, cpucg ]:
            logger.log("cgroup for memory and cpu is not enabled. exiting.")
            return -1

        logger.log("cgtype: %s, uuidfile: %s" % (cgtype, uuidfile))

        while True: # endless monitor script for docker job
            check_parent()

            if cgtype in ["cgroupfs", ""]:
                time.sleep(interval);
                continue

            if not os.path.exists(uuidfile):
                time.sleep(interval);
                continue  # uuidfile file should not be created yet
            
            try:
                fp = open(uuidfile, "r")
                self.uuid = fp.readline().strip()
            except Exception as e:
                logger.log("open uuid file exception: %s" % str(e))
                continue
            finally:
                fp.close()

            if is_str_empty(self.uuid):
                time.sleep(interval);
                continue

            logger.log("uuid: %s, jobdbid: %s" % (self.uuid, self.jobdbid))

            self.__get_mem_usage(memcg, resources)
            self.__get_cpu_usage(cpucg, resources)

            updatedb(self.tablename, resources)

            check_parent()
            time.sleep(interval);

            cuuid = self.uuid[0:12]
            checkcmd = '/usr/bin/docker ps -a -q -f status=dead | grep %s' % (cuuid)

            out, err, rc = runcmd(checkcmd)
            if rc == 0:
                output = out.split('\n')
                if  output is not None and cuuid == output[0] :
                    dockerrmcmd = '/usr/bin/docker rm %s' % (cuuid)
                    rumcmd(dockerrmcmd)

                    jobid = self.jobid
                    if self.jobindex is not None and self.jobindex != "0" :
                        jobid = "%s[%s]" % self.jobindex

                    jobkillcmd = 'bkill %s' % jobid
                    runcmd(jobkillcmd)

            check_parent()


    def __get_mem_usage(self, memcg, resources):
        mempath = get_container_cgpath(self.uuid, memcg)

        try:
            # get container pids
            fp = open("%s/cgroup.procs" % mempath, "r")

            content = fp.read()
            resources['pids'] = content.replace("\n", " ")
            fp.close()

            fp = open("%s/memory.usage_in_bytes" % mempath, "r")
            resources['mem'] = fp.readline().strip()
            logger.log("Container mem usage: %s" % resources['mem']);
            fp.close()

        except Exception as me:
            logger.log("Cannot get container mem usage: %s" % (str(me)))
            return 
        
        # check whether pid is valid if not, return
        if is_str_empty(resources['pids'].strip(), True):
            logger.log("Cannot get container pids")
            return 

        # ticket:231648, get container swap usage.
        try:
            fp = open("%s/memory.memsw.usage_in_bytes" % mempath, "r")
            content = fp.readline()
            logger.log("Container mem+swap usage: %s" % content);
            try:
                resources['swp'] =  int(content.strip()) - int(resources['mem'])
                if resources['swp'] < 0:
                    resources['swp'] = 0
            except ValueError as ve:
                logger.log("Cannot get container mem+swap usage: %s" % str(ve));
                resources['swp'] = 0

            fp.close()

        except Exception as se:
            logger.log("Cannot get container swap usage: %s" % str(se));

        mem = int(resources['mem']) / 1024
        resources['mem'] = str(mem)

        swap = int(resources['swp']) / 1024
        resources['swp'] = str(swap)

        logger.log("After get mem, resources: %s" % (str(resources)))

    def __get_cpu_usage(self, cpucg, resources):
        
        cpupath = get_container_cgpath(self.uuid, cpucg)

        utime = stime = 0
        try:
            fp = open("%s/cpuacct.stat" % (cpupath))

            while True:
                line = fp.readline()
                if line.strip() == '':
                    break;
                elements = line.split(' ')
                if elements[0] == 'user':
                    utime = int(elements[1])
                if elements[0] == 'system':
                    stime = int(elements[1])
            
            fp.close()

        except Exception as e:
            logger.log("Cannot get container CPU usage: %s" % (str(e)))

        # the job may exit. continue checking
        tck = os.sysconf('SC_CLK_TCK')
        logger.log("CPU utime: %s, CPU stime: %s, tck: %s" % (utime, stime, tck));
        utime = utime/tck
        resources['utime'] = str(utime)
        stime = stime/tck
        resources['stime'] = str(stime)

        logger.log("After get cpu, resources: %s" % (str(resources)))

def handler(signum, frame):
    logger.log("Exiting due to signal: %d " % signum)
    lsbexit(-signum)

def main(argv):
    updatets("in monitor")
    check_parent()

    global logger  
    global cgtype

    cgtype = CGHelper.get_docker_cgroup_type('/usr/bin', 'docker')
    check_parent()

    for s in [ signal.SIGINT, signal.SIGTERM, signal.SIGABRT, signal.SIGPIPE, signal.SIGSEGV ]:
        signal.signal(s, handler)

    logger = DriverLogger("monitor", driver_user, "debug")
    logger.init()

    validate_initvars()

    updatets("start monitor")
    monitor = DriverMonitor();
    if monitor.start() < 0 :
        lsbexit(-1)

    logger.log("Exiting")
    lsbexit(0)

if __name__ == "__main__":
    if (len(sys.argv) == 1) and None == driver_user:
        ret = show_file_info(sys.argv[0])
        lsbexit(ret)

    main(sys.argv[1:])
