#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

if [ $# -ne 3 ]
then
    echo "Usage: $0 <user_id> <docker_path> <job_id>" >&2
    exit 1
fi

user=$1
docker_path=$2
job_id=$3

# Run the nc info <jobid> command and capture its output
job_info="$(scontrol show jobs  -d $job_id)"
ret=$?
if [ $ret -ne 0 ]; then
  echo "Unable to retrieve bjobs output for job $job_id" 2>&1 | tee /dev/stderr
  # Check if the job_id consists of exactly 9 digits and nothing else
  if ! [[ $job_id =~ ^[0-9]{9}$ ]]; then
    echo "Invalid job_id format. It should be a 9-digit number." 2>&1 | tee /dev/stderr
    exit 1
  fi
  exit 1
fi


#job_status=$(echo "$job_info" | grep -oP 'Status\s+[\w\s]+')
job_status=$(echo "$job_info" | grep -P 'JobState=*' | awk '{print $1}' | awk -F '=' '{print $2}')
if [ "$job_status" != "RUNNING" ]; then
  echo "Job $job_id is not in running state" 2>&1 | tee /dev/stderr
  exit 1
else
   owner=$(echo "$job_info" | grep -P 'UserId=*' | awk '{print $1}' | awk -F '=' '{print $2}' |  awk -F '(' '{print $1}')
   if [ "$owner" != "$user" ]; then
      echo "$user is not authorized to access this job :: $job_id" 2>&1 | tee /dev/stderr
      exit 1
   fi
fi

host=$(echo "$job_info" | grep -P ' Nodes=*' | awk '{print $1}' | awk -F '=' '{print $2}')
if [ -z "$host" ]; then
  echo "Unable to retrieve execution host list for job $job_id" 2>&1 | tee /dev/stderr
  exit 1
fi

# Get the public IP address
ip_address=$(nslookup $host | awk '/^Address: / { print $2 }')
if [ -z "$ip_address" ]; then
  echo "Unable to retrieve IP address for host $host" 2>&1 | tee /dev/stderr
  exit 1
fi

if [[ $ip_address =~ ^(10\.|172\.(1[6-9]|2[0-9]|3[0-1])\.|192\.168\.) ]]; then
  :
else
  echo "Not a private IP address: $ip_address" 2>&1 | tee /dev/stderr
  exit 1
fi
#get the container id of the running  job
#cont_id=$($docker_path -H tcp://$ip_address:2375 ps -q -f name="vnc3_ip-10-0-0-74.us-west-1.compute.internal_sbx_nc_dock_test_$job_id")
cont_id=$($docker_path -H tcp://$ip_address:2375 ps -q -f name="$job_id.slurmjob")
if [ -z "$cont_id" ]
then
  echo "Unable to retrieve container ID for job $job_id" 2>&1 | tee /dev/stderr
  exit 1
fi
# Check if cont_id contains more than one value (multiple containers found)
if [ $(echo "$cont_id" | wc -l) -gt 1 ]; then
    echo "Error: Multiple containers found for job $job_id . Provide the complete job id number" 2>&1 | tee /dev/stderr
    exit 1
fi
echo "ATTACHING to the container :: $cont_id of the JOB:: $job_id"
#connect to the docker container in exec mode
$docker_path -H tcp://$ip_address:2375  exec -u ${user} -i $cont_id /bin/sh -c '/bin/bash -i'

# Print the extracted information


