#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

if [ "$SPILLBOX_DEBUG" != "" ] ; then
    set -x
fi

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
INTR=""

usage()
{
    echo "Usage : $0 <pid> [cgroup dir] <job_manager>" >&2
    exit 3
}

if test "$#" -lt 1; then
    usage;
fi

PID=$1
cg_dir=$2
transport_server=$3
job_manager=$4

sudocmd=""
id=$(id -u)
if [ "$id" == "0" ]; then
    sudocmd="sudo"
else
    CAN_I_RUN_SUDO=$(sudo -n uptime 2>&1|grep "load"|wc -l)
    if [ $(($CAN_I_RUN_SUDO)) -gt 0 ]; then
        sudocmd="sudo"
    else
        sudocmd="/bin/true"
    fi
fi

setup_check()
{
    if [ "${SPILLBOX_CR_IMAGE}" == "" ]; then
        echo "SPILLBOX_CR_IMAGE not set " >&2
        exit 3
    else
        if [ ! -d "${SPILLBOX_CR_IMAGE}" ]; then
            echo "Non-existent directory ${SPILLBOX_CR_IMAGE} " >&2
            exit 3
        fi
    fi
    if [ "${SPILLBOX_CR_JOBID}" == "" ]; then
        echo "SPILLBOX_CR_JOBID not set " >&2
        exit 3
    fi
    re='^[0-9]+$'
    if ! [[ $PID =~ $re ]] ; then
       echo "error: PID is not a number" >&2; exit 3
    fi
    if [ ! -x "${CRIU_EXE}" ]; then
        echo "criu not found" >&2
        exit 3
    fi
    #cap=$(getcap ${CRIU_EXE}  | grep cap_sys_admin)
    #if [ "$cap" == "" ]; then
    #    echo "linux cap cap_sys_admin not set on ${CRIU_EXE}" >&2
    #    exit 3
    #fi
    $sudocmd mkdir -p ${SPILLBOX_CR_IMAGE}/${SPILLBOX_CR_JOBID}
    if [ ! -d "${SPILLBOX_CR_IMAGE}/${SPILLBOX_CR_JOBID}" ] ; then
        echo "Directory ${SPILLBOX_CR_IMAGE}/${SPILLBOX_CR_JOBID} could not be created" >&2
        exit 3
    fi
}


checkpoint()
{
    if [ "${INTR}" != "" ]; then
        echo "Under unterruption..." >&2
        exit 0
    fi
    if ps -p $PID > /dev/null
    then
        $sudocmd /bin/rm -f ${SPILLBOX_CR_IMAGE}/${SPILLBOX_CR_JOBID}/spillbox_save_success >& /dev/null
        $sudocmd ${CRIU_EXE} dump -t $PID -R --images-dir ${SPILLBOX_CR_IMAGE}/${SPILLBOX_CR_JOBID} --shell-job --tcp-established |& $sudocmd tee ${SPILLBOX_CR_IMAGE}/${SPILLBOX_CR_JOBID}/spillbox_save_log >& /dev/null
        RETVAL=$?
        if test ${RETVAL} -eq 0 ; then
            echo "$PID" |& $sudocmd tee ${SPILLBOX_CR_IMAGE}/${SPILLBOX_CR_JOBID}/spillbox_save_success >& /dev/null
        fi
    fi
}

func_handler()
{
    raw_sig=$?
    sig=$(($raw_sig - 128))
    INTR=1
    echo "Under unterruption..." >&2
    exit 0
}

disable_on_intr()
{
    for signal in INT ; do
        trap "func_handler" ${signal}
    done
}

res_threshold_reached()
{ 
    warn_threshold=${SPLAPP_CR_THRESHOLDS_WARN}
    critical_threshold=$SPLAPP_CR_THRESHOLDS_CRITICAL
    warn_run_cmd="${SPLAPP_CR_ACTION_WARN_RUN_COMMAND}"
    requeue_partition=${SPLAPP_CR_ACTION_CRITICAL_REQUEUE_PARTITION}

    #docker_info=$(grep memory /proc/self/cgroup | awk -F: '{print $3}' | head -n1)
    #usage=$(cat "/sys/fs/cgroup/memory/${docker_info}/memory.usage_in_bytes")

    limit="max"
    usage="0"
    if [ -f "$cg_dir/memory.usage_in_bytes" ]; then
        usage=$(cat "$cg_dir/memory.usage_in_bytes")
    fi
    if [ -f "$cg_dir/memory.limit_in_bytes" ]; then
        limit=$(cat "$cg_dir/memory.limit_in_bytes")
    fi
    # detect unlimited limit in cgroups (v1 huge value or v2 "max")
    if [[ "$limit" == "max" ]] || [[ "$limit" -ge 9223372036854771712 ]]; then
        # check ulimit (rarely set in containers)
        shell_limit=$(ulimit -v)
        if [[ "$shell_limit" == "unlimited" ]] || [[ "$shell_limit" -le 0 ]]; then
            # fallback to container's view of host memory
            limit=$(awk '/MemTotal/ {print $2 * 1024}' /proc/meminfo)
            #echo "Using host MemTotal as limit: $limit bytes"
        else
            limit=$((shell_limit * 1024))
            #echo "Using shell ulimit as limit: $limit bytes"
        fi
    #else
    #    echo "Using cgroup limit: $limit bytes"
    fi
    if [[ -z "$limit" || "$limit" -le 0 ]]; then
        echo "No valid memory limit detected, skipping check. Value : $limit" >&2
        return 0
    fi

    usage_pct=$((usage * 100 / limit))
    if (( usage_pct >= critical_threshold )); then
        echo "CRITICAL threshold reached for $cg_dir" >&2
        return 1
    elif (( usage_pct >= warn_threshold )); then
        eval "$warn_run_cmd"
        #running once
        export SPLAPP_CR_ACTION_WARN_RUN_COMMAND=""
    fi
    return 0
}

setup_check
disable_on_intr
if [ "$SPLAPP_CR_MEMOPT" == "" ]; then
    # simply checkpoint in n mins
    min=60
    if [ "${SPLAPP_CR_INTERVAL}" != "" ]; then
        re='^[0-9]+$'
        if ! [[ ${SPLAPP_CR_INTERVAL} =~ $re ]] ; then
            echo "error: SPLAPP_CR_INTERVAL Not a number" >&2;
            unset SPLAPP_CR_INTERVAL
        else
            min=${SPLAPP_CR_INTERVAL}
        fi
    fi
    interval=$(($min * 60)) ## Convert to seconds
    while true; do
        sleep $interval
        checkpoint
    done
else
    REQUEUE_REQUEST_SENT=0
    #checkpoint only when memory threshold reached
    while true;  do
        case "$job_manager" in
            "lsf")
                #echo "lsf - resource checking"
                res_threshold_reached
                ret=$?
                if [[ $ret -eq 1  && $REQUEUE_REQUEST_SENT -eq 0 ]]; then
                    checkpoint
                    curl -k "https://${transport_server}/requeue?jobmanager=lsf&jobid=${SPILLBOX_CR_JOBID}&partition=${SPLAPP_CR_ACTION_CRITICAL_REQUEUE_PARTITION}&clouduser=${SPILLBOX_CLOUD_USER}"
                    RET=$?
                    if test $RET -eq 0; then
                        REQUEUE_REQUEST_SENT=1
                    fi
                fi
                ;;
            "slurm")
                res_threshold_reached
                ret=$?
                if [[ $ret -eq 1  && $REQUEUE_REQUEST_SENT -eq 0 ]]; then
                    checkpoint
                    curl -k "https://${transport_server}/requeue?jobmanager=slurm&jobid=${SPILLBOX_CR_JOBID}&partition=${SPLAPP_CR_ACTION_CRITICAL_REQUEUE_PARTITION}&clouduser=${SPILLBOX_CLOUD_USER}"
                    RET=$?
                    if test $RET -eq 0; then
                        REQUEUE_REQUEST_SENT=1
                    fi
                fi
                ;;
        esac
        sleep 60
    done
fi
