#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

## This scripts needs the basic LSF setup scripts sourced.
## Typically /usr/share/lsf/conf/profile.lsf

function docker_version_check () {
	docker_version=`docker --version 2>&1`
	RETVAL=$?
	if test $RETVAL -gt 0 ; then
		echo "[ERROR] : docker setup not available, please make sure docker setup is available" 2>&1 | tee /dev/stderr
		echo "$docker_version : $RETVAL" 2>&1 | tee /dev/stderr
		exit 3
	fi
	tarrVer=(${docker_version// / })
	arrVer=(${tarrVer[2]//./ })
	if [[ ${arrVer[0]} -eq 20 && ${arrVer[1]} -ge 10 ]] || 
	   [[ ${arrVer[0]} -gt 20 ]] ; then
		echo "Recommended Docker version satisfied >=20.10"
	else
		echo "Recommended Docker version not satisfied"
		echo "Expected version : >=20.10"
		echo "Got : ${tarrVer[2]}"
		echo "Will continue with the image build, behavior might not be as desired"
	fi
}


#Generating the sudoers file for lsf bapp..
function generate_sudoers_file()
{
    if [ "$runtime_container" == "docker" ] ; then
    	if [[ "$lsf_dockerized_cluster" == "true" ]] ; then

        sudoers_string="
Defaults        env_reset
Defaults        mail_badpass
Defaults        secure_path=\"/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/snap/bin\"

# User privilege specification
root    ALL=(ALL:ALL) ALL

# Members of the admin group may gain root privileges
%admin ALL=(ALL) ALL

# Allow members of group sudo to execute any command
%sudo   ALL=(ALL:ALL) ALL

"
        file_dir=$(realpath $DIR/../../image)
        if [ $(stat -c "%a" "$file_dir") != "775" ]; then
                echo "Please check the directory permission for the $file_dir" 2>&1 | tee /dev/stderr
                echo "Permission should be '775' " 2>&1 | tee /dev/stderr
                exit 1
        fi
	user=$(whoami)
	ldap_existance=$(grep ^"passwd:.* ldap" /etc/nsswitch.conf)
        if [ $? != 0 ] ; then
            sudoers_string+="# Without LDAP "$'\n'"$user ALL=(ALL) NOPASSWD:ALL"
        else
            sudoers_string+="# With LDAP "$'\n'"$user ALL=(ALL) NOPASSWD: /bin/mkdir, /usr/bin/mkdir, /bin/mount, /usr/bin/mount"
        fi
        ( umask 033; sudo sh -c "echo \"$sudoers_string\" > $DIR/../../image/sudoers")
    	fi
    fi
}

function docker_image_build () {
#	check_image="spillbox"
	dockerfile="${SPL_ROOT}/image/Dockerfile.wrapper"
#	dockerimages=`docker image list --format "table {{.Repository}}" | grep -w ^$check_image 2>&1`
#	RETVAL=$?
#	if [[ "$dockerimages" != "$check_image" ]] || [[ $RETVAL -ne 0 ]] ; then
#		echo "[ERROR]: Docker image $check_image which is required for building spillbox app image is not found."
#		exit 3
#	fi
	build_image_tag=$docker_base_image
	echo "Building docker image $build_image_tag using the below command"
	echo "docker build -f $dockerfile -t $build_image_tag ${SPL_ROOT}"
	docker_result=`docker build -f $dockerfile -t $build_image_tag --build-arg SERVER_IMAGE=$docker_wrapper_image ${SPL_ROOT} 2>&1`
	RETVAL=$?
	if [ $RETVAL -ne 0 ] ; then
		echo "[ERROR]: Builing the spillbox app image($build_image_tag) has failed." 2>&1 | tee /dev/stderr
		echo "Error code : $RETVAL" 2>&1 | tee /dev/stderr
		echo "Output: $docker_result" 2>&1 | tee /dev/stderr
		exit 3
	else
		echo "spillbox app image($build_image_tag) Successfully built"
		docker tag $build_image_tag $docker_host:$docker_registry_port/$build_image_tag
		docker_push_result=`docker push $docker_host:$docker_registry_port/$build_image_tag`
#		docker_push_result=`docker push $build_image_tag`
		RETVAL=$?
		if [ $RETVAL -ne 0 ] ; then
			echo "[ERROR]: Pushing the spillbox app image($docker_host:$docker_registry_port/$build_image_tag) to registry"
			echo "         Please check, correct and push the image manually..."
			echo Error message: $docker_push_result
			echo
		else
			echo "Pushing image to registry ($docker_host:$docker_registry_port/$build_image_tag).. Done."
		fi
	fi
}

function python_check () {
    if [ "$python2_path" == "" ]; then
        tmp=$(which python2)        
        if [ "$tmp" == "" ]; then 
            tmp=$(which python2)
            if [ "$tmp" == "" ]; then 
                tmp=$(which python)
                if [ "$tmp" == "" ]; then 
		            echo "[ERROR]: python2 not available. Please set python2_path in $spl_config_file " 2>&1 | tee /dev/stderr
		            exit 3
                fi
            fi
        fi
        v=$($tmp --version |& awk '{print $2}' | awk -F. '{print $1}')
        if [[ "$v" -ge "2" ]]; then
            export python2_path=$tmp
        else
		    echo "[ERROR]: python2 not available. Please set python2_path in $spl_config_file " 2>&1 | tee /dev/stderr
		    exit 3
        fi
	fi
    esc="${python2_path//\//\\/}"
    sed -i "1s/.*/#\!$esc/" ${SPL_ROOT}/etc/docker-control.py
    sed -i "1s/.*/#\!$esc/" ${SPL_ROOT}/etc/docker-monitor.py
    sed -i "1s/.*/#\!$esc/" ${SPL_ROOT}/etc/docker-starter.py
}

function lsf_env_setup () {
	if [ " ${LSF_ENVDIR}" == " " ] ; then
		echo "[ERROR]: LSF setup is not sourced, please source the LSF setup and run this script" 2>&1 | tee /dev/stderr
		echo "Typically the setup script will be found in /usr/share/lsf/conf/profile.lsf." 2>&1 | tee /dev/stderr
		echo "This one is for sh/bash, there other setup files available for different shell type." 2>&1 | tee /dev/stderr
		exit 3
	fi
}

function lsf_bapp_setup () {

	spl_bapp_file=${SPL_ROOT}/spl_lsb.applications
	spl_lsfconf_file=${SPL_ROOT}/spl_lsf.conf
	spl_lsfshared_file=${SPL_ROOT}/spl_lsf.shared
	spl_lsfcluster_file=${SPL_ROOT}/spl_lsf.cluster
	spl_lsbparams=${SPL_ROOT}/spl_lsb.params
	LSF_DIR=$(dirname $LSF_ENVDIR)
	SPILLBOX_SITE=`hostname`
	chmod 700 ${SPL_ROOT}/etc/docker-*.py
if [ "$runtime_container" == "docker" ]; then
	# Create the lsb.applications content locally
	cat > $spl_bapp_file << ELN

Begin Application
NAME = ${bapp_name}
CONTAINER = docker[image($build_image_tag) options(--rm $SPILLBOX_DNS_SETUP --privileged --env=SPILLBOX_SITE=${SPILLBOX_SITE} -v ${TERRAFORM_ROOT}/terraform/bin:/spillbox -v ${workdir}:/spillbox_workdir ${LOCAL_MOUNTS} $LDAP_DIR ${NIS_DIR} $BIND_MOUNT_VOLUME $FSCACHE_DIR  --entrypoint=/spillbox/bapp_entrypoint.sh)]
EXEC_DRIVER=context[user($USER)] starter[${SPL_ROOT}/etc/docker-starter.py] controller[${SPL_ROOT}/etc/docker-control.py] monitor[${SPL_ROOT}/etc/docker-monitor.py]
DESCRIPTION = Docker User Service
End Application

ELN

else

    cat > $spl_bapp_file << ELN
Begin Application
NAME = ${bapp_name}
DESCRIPTION = Singuarity User Service
CONTAINER = singularity[image($singularity_base_image) options(--writable-tmpfs --dns=$SPILLBOX_DNS -B ${workdir}:/spillbox_workdir -B ${TERRAFORM_ROOT}/terraform/bin:/spillbox ${LOCAL_MOUNTS} -B /tmp:/tmp $LDAP_DIR ${NIS_DIR} $BIND_MOUNT_VOLUME $FSCACHE_DIR -B /etc/sudoers:/etc/sudoers -B /etc/passwd:/etc/passwd -B /etc/group:/etc/group )]
EXEC_DRIVER=context[user($USER)] starter[${SPL_ROOT}/etc/docker-starter.py]
End Application
ELN

fi

	RETVAL=$?
	if test $RETVAL -ne 0 ; then
		echo "[ERROR]: Creation of $spl_bapp_file failed with error code : $RETVAL" 2>&1 | tee /dev/stderr
		exit 3
	fi
	echo
	echo "$spl_bapp_file file created, copy/append the required contents to the $LSF_ENVDIR/lsbatch/$lsf_cluster_name/configdir/lsb.applications"

	# Create lsf.conf content 
	cat > $spl_lsfconf_file << ELN
LSF_PROCESS_TRACKING=Y
LSF_LINUX_CGROUP_ACCT=Y
LSB_RESOURCE_ENFORCE="cpu memory"
ELN
	RETVAL=$?
	if test $RETVAL -ne 0 ; then
		echo "[ERROR]: Creation of $spl_lsfconf_file failed with error code : $RETVAL" 2>&1 | tee /dev/stderr
		exit 3
	fi
	echo
	echo "$spl_lsfconf_file file created, copy/append the required contents to the $LSF_ENVDIR/lsf.conf"
	echo "Some variables might have been already defined, exclude those variables while copying"

	# Create lsf.shared content 
	cat > $spl_lsfshared_file << ELN
Begin Resource
RESOURCENAME  TYPE    INTERVAL INCREASING CONSUMABLE DESCRIPTION  # Keywords
   $runtime_container     Boolean ()       ()         ()         ($runtime_container container)
End Resource

ELN
	RETVAL=$?
	if test $RETVAL -ne 0 ; then
		echo "[ERROR]: Creation of $spl_lsfshared_file failed with error code : $RETVAL" 2>&1 | tee /dev/stderr
		exit 3
	fi
	echo
	echo "$spl_lsfshared_file file created, copy/append the required contents to the $LSF_ENVDIR/lsf.shared"

	# Create lsf.cluster.<> content 
	cat > $spl_lsfcluster_file << ELN
Begin Host
    HOSTNAME  model  type  server  r1m  mem  swp  RESOURCES
    host1     !      !     1       3.5  ()   ()   (mg $runtime_container)
End Host
ELN
	RETVAL=$?
	if test $RETVAL -ne 0 ; then
		echo "[ERROR]: Creation of $spl_lsfcluster_file failed with error code : $RETVAL" 2>&1 | tee /dev/stderr
		exit 3
	fi
	echo
	echo "$spl_lsfcluster_file file created, copy/append the required contents"
	echo "	to the $LSF_ENVDIR/lsf.cluster.$lsf_cluster_name"
	echo "Substitute 'host1' with your required hostname"
	cat > $spl_lsbparams << ELN
JOB_SPOOL_DIR = /tmp
DEFAULT_QUEUE = sbox_compute_queue
ELN
	RETVAL=$?
	if test $RETVAL -ne 0 ; then
		echo "[ERROR]: Creation of $spl_lsbparams failed with error code : $RETVAL" 2>&1 | tee /dev/stderr
		exit 3
	fi
	echo
	echo "$spl_lsbparams file created, copy/append the required contents to the $LSF_ENVDIR/lsbatch/$lsf_cluster_name/configdir/lsb.params"

	echo
	echo "Please use these files as guidance and change as relevant to your environment"
	echo
	echo "Please run 'lsadmin ckconfig' and if there are no errors then run 'lsadmin reconfig'"
	echo "Please run 'badmin ckconfig' . If there are no errors then"
	echo "please run the following 3 commands : 'lsadmin reconfig', 'badmin reconfig', 'badmin mbdrestart' "
	echo "for the changes to be in effect"

    echo
    echo "On client, please set following variable:"
    echo "export SPLAPP_BATCH_RUN_OPTION=\"-app ${bapp_name}\""


} ### End of function

DIR=$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )
SPL_ROOT=$(realpath $DIR/../..)
spl_config_file=$(realpath ${SPL_ROOT}/terraform/etc/spillbox_config)
if [ -f  $spl_config_file ]; then
    . $spl_config_file
    if [ "$runtime_container" == "docker" ] ; then
         if [[ "$docker_registry_hostname" == "true" ]]; then
		 docker_host=$(hostname -f)
	 else
	   docker_host=`hostname -I | awk '{print $1}'`
   fi
	    if [ "$docker_registry_port" == "" ]; then
            docker_registry_port=5000
        fi
    fi
    SERVER_IMAGE_OUTPUT=$(${SPL_ROOT}/terraform/server_image.sh output)
    server_base_image=$(echo "$SERVER_IMAGE_OUTPUT" | grep '"server_image"' -A 3 | grep '"value"' | sed -E 's/.*"value": "([^"]+)".*/\1/')
else
    echo missing config file : $spl_config_file >&2
    exit 3
fi
if [ "${workdir}" == "" ]; then
    echo workdir not set in config file: $spl_config_file >&2
    exit 3
fi
SPILLBOX_SITE=`hostname`
TERRAFORM_ROOT="${workdir}/.spillbox/sites/local/${SPILLBOX_SITE}"

if [ "${bapp_name}" == "" ]; then
    echo "bapp_name not set, setting default bapp_name : $build_image_tag"
    bapp_name=$build_image_tag
fi
if [ "$lsf_cluster_name" == "" ]; then
    echo lsf_cluster_name not set in $spl_config_file >&2
    exit 3
fi

if [ "$setup_ldap" == "true" ]; then
    if [ "$runtime_container" == "singularity" ] ; then
        LDAP_DIR="-B /var/lib/sss/pipes:/var/lib/sss/pipes:rw"
    else
        LDAP_DIR="-v /var/lib/sss/pipes:/var/lib/sss/pipes:rw"
    fi
else
    LDAP_DIR=""
fi
if [ "$setup_nis" == "true" ]; then
    if [ "$runtime_container" == "singularity" ] ; then
        NIS_DIR="-B /etc/nsswitch.conf:/etc/nsswitch.conf:ro -B /etc/yp.conf:/etc/yp.conf:ro"
    else
        NIS_DIR="-v /etc/nsswitch.conf:/etc/nsswitch.conf:ro -v /etc/yp.conf:/etc/yp.conf:ro"
    fi
else
    NIS_DIR=""
fi

BIND_MOUNT_VOLUME=""
FSCACHE_DIR=""
LOCAL_MOUNTS=""
if [ "$boostmode" == "true" ]; then
    if [ -z "$boost_client_cache_dir" ]; then
        boost_client_cache_dir="/var/cache/fscache"
    fi
    if [ "$runtime_container" == "docker" ]; then
        FSCACHE_DIR="--env=SPILLBOX_BOOSTMODE=1 --env=SPILLBOX_BOOST_CLIENT_CACHE_DIR=$boost_client_cache_dir -v $boost_client_cache_dir:$boost_client_cache_dir"
    else
        FSCACHE_DIR="--env=SPILLBOX_BOOSTMODE=1 --env=SPILLBOX_BOOST_CLIENT_CACHE_DIR=$boost_client_cache_dir -B $boost_client_cache_dir:$boost_client_cache_dir"
    fi
fi

if [ "$local_mount_paths" != "" ]; then 
    for i in $local_mount_paths
    do
        if [ "$runtime_container" == "docker" ]; then
            LOCAL_MOUNTS="$LOCAL_MOUNTS --volume=$i:$i:rslave"
        else
            LOCAL_MOUNTS="$LOCAL_MOUNTS -B $i:$i"
        fi
    done
fi

SPILLBOX_DNS_SETUP=""
RESOLV="/etc/resolv.conf"
if [ -f "$RESOLV" ]; then
    while IFS="" read -r p || [ -n "$p" ]
    do
        d=$(echo $p | grep ^nameserver | awk '{print $2}')
        if [ "$d" != "" ]; then
            SPILLBOX_DNS_SETUP="$SPILLBOX_DNS_SETUP --dns=$d"
            SPILLBOX_DNS=$(echo $d)
        fi
        d=$(echo $p | grep ^search | awk '{$1=""; print $0}')
        if [ "$d" != "" ]; then
            for i in $d
            do
                SPILLBOX_DNS_SETUP="$SPILLBOX_DNS_SETUP --dns-search=$i"
            done
        fi
    done < $RESOLV
fi
export SPILLBOX_DNS_SETUP=$SPILLBOX_DNS_SETUP
if [ "$runtime_container" == "docker" ] ; then
    docker_version_check
fi
#generate_sudoers_file
#docker_image_build 
build_image_tag=$docker_host:$docker_registry_port/$server_base_image
python_check
lsf_env_setup
lsf_bapp_setup
