#!/bin/bash
# requeue_job.sh
# Usage: ./requeue_job.sh <job_manager> <job_id> <partition_name>
#
# job_manager: lsf or slurm
# job_id: job id to modify
# partition_name: target partition (slurm) or queue (lsf)

set -euo pipefail

usage() {
    echo "Usage: $0 <job_manager: lsf|slurm> <clouduser> <job_id> <partition_name>"
    exit 1
}

if [ $# -ne 4 ]; then
    usage
fi

JOB_MANAGER=$1
CLOUD_USER=$2
JOB_ID=$3
PARTITION=$4

case "$JOB_MANAGER" in
    slurm)
        echo "[INFO] Using Slurm..."
        # Requeue job
        echo "[INFO] Requeuing job $JOB_ID"
        sleep 10
        SCONTROL_PATH=$(which scontrol)
        sudo -E -u $CLOUD_USER $SCONTROL_PATH requeue "$JOB_ID"

        # Modify partition
        echo "[INFO] Changing partition to $PARTITION"
        sudo -E -u $CLOUD_USER $SCONTROL_PATH update JobId="$JOB_ID" Partition="$PARTITION"
        ;;

    lsf)
        echo "[INFO] Using LSF..."
        # Requeue job
        echo "[INFO] Requeuing job $JOB_ID"
        brequeue "$JOB_ID"

        # Modify queue
        echo "[INFO] Changing queue to $PARTITION"
        bmod -q "$PARTITION" "$JOB_ID"
        ;;

    *)
        echo "[ERROR] Unsupported job manager: $JOB_MANAGER"
        usage
        ;;
esac

echo "[SUCCESS] Job $JOB_ID updated on $JOB_MANAGER to partition/queue $PARTITION"
