#!/bin/bash -f

# Copyright © 2018-2025, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

if [ $# -lt 4 ]; then
  echo "Usage: $0 <user> <project> <docker_path> :<port1> :<port2> ..." >&2
  exit 1
fi

user="$1"
project="$2"
docker_path="$3"
shift 3  # Shift the first 3 arguments to process only port args

found=0

for arg in "$@"; do
  if [[ "$arg" =~ ^:[0-9]+$ ]]; then
    port="${arg#:}"  # Remove leading colon
    container_id=$($docker_path ps --format "{{.ID}} {{.Ports}}" | grep ":$port->" | awk '{print $1}')

    if [ -n "$container_id" ]; then
      # Inspect container and extract cloud user and project
      inspect=$($docker_path inspect "$container_id" 2>/dev/null)

      dock_user=$(echo "$inspect" | grep -i "SPILLBOX_CLOUD_USER" | awk -F '=' '{print $2}' | awk -F '"' '{print $1}')
      dock_project=$(echo "$inspect" | grep -i "SPILLBOX_PROJECT" | awk -F '=' '{print $2}' | awk -F '"' '{print $1}')

      if [ "$dock_user" == "$user" ] && [ "$dock_project" == "$project" ]; then
        echo "Killing container on port $port: $container_id (user: $dock_user, project: $dock_project)"
        $docker_path rm -f "$container_id"
        found=1
      else
        echo "Skipping container on port $port: Not owned by user/project"
      fi
    else
      echo "No container found on port $port"
    fi
  else
    echo "Invalid argument: $arg. Must be in format :<port>" >&2
  fi
done

if [ "$found" -eq 0 ]; then
  echo "No containers were killed for user '$user' and project '$project'"
fi
