terraform {
  required_version = ">= 1.10.4, < 1.10.5" 
  required_providers {
    aws = {
      version = "5.89.0"
    }
    null = {
      version = "3.2.3"
    }
    tls = {
      version = "4.0.6"
    }
    template = {
      version = "2.2.0"
    }
    local = {
      version = "2.5.2"
    }
  }
  backend "s3" {
    key     = "states/image-compute"
    encrypt = true
  }
}

provider "aws" {
  region = var.region
}


locals {
  key_name = "${var.imagetag}-auto-key"
  private_key_filename = format("%s/%s.pem", var.secrets, local.key_name)
}

data "template_file" "config" {
  template = file("${path.module}/init.tpl")
  vars = {
    cloud_ssh_port = var.cloud_ssh_port
  }
}

data "template_cloudinit_config" "config" {
  gzip          = true
  base64_encode = true

  part {
    content_type = "text/x-shellscript"
    content      = data.template_file.config.rendered
  }
}

resource "aws_instance" "compute" {
  instance_type          = var.image_instance_type
  ami                    = var.ami_name
  key_name               = var.keyname
  vpc_security_group_ids = [var.security_group_id]
  subnet_id              = var.subnet_id
 tags = {
    Name = "Spillbox-Image-${var.imagetag}" 
  }
 user_data = data.template_cloudinit_config.config.rendered

  root_block_device {
    volume_size           = var.image_root_volume_size
    kms_key_id            = var.kms_key_id == "" ? null : var.kms_key_id
    encrypted             = var.kms_key_id == "" ? false : true
    delete_on_termination = true
  }

  provisioner "local-exec" {
    command = "sleep ${var.wait_time_vm} && echo done waiting master VM ready"
  }
}

resource "null_resource" "image_install" {
  connection {
    user        = var.vm_user
    private_key = file(local.private_key_filename)
    host        = aws_instance.compute.private_ip
    port        = 22
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = ["sudo chmod 755 /tmp/install/install.sh; /tmp/install/install.sh '${var.vm_user}' '${var.webserverip}' '${var.docker_registry}' '${var.docker_images}' '${var.service_userid}' '${var.cluster_create}'"]
  }
}


resource "aws_instance" "docker_image" {
  ami                    = var.docker_base_ami_name
  instance_type          = var.image_instance_type
  subnet_id              = var.subnet_id
  vpc_security_group_ids = [var.security_group_id]
  key_name               = var.keyname

  root_block_device {
    volume_size           = var.image_root_volume_size
    kms_key_id            = var.kms_key_id == "" ? null : var.kms_key_id
    encrypted             = var.kms_key_id == "" ? false : true
    delete_on_termination = true
  }

  tags = {
    Name = "spillbox-dockerimage-${var.imagetag}"
  }
  provisioner "local-exec" {
    command = "sleep 30 && echo done waiting master VM ready"
  }
}

resource "aws_ebs_volume" "new_volume" {
  depends_on = [aws_instance.docker_image]
  snapshot_id        = var.snapshot_id
  size               = var.image_root_volume_size
  availability_zone  = aws_instance.docker_image.availability_zone
  kms_key_id         = var.kms_key_id == "" ? null : var.kms_key_id
  encrypted          = var.kms_key_id == "" ? false : true

  tags = {
    Name = "spillbox"
  }
}

resource "null_resource" "manage_instance_marketplace" {
  depends_on = [aws_instance.docker_image, aws_ebs_volume.new_volume]

  provisioner "local-exec" {
    command = <<EOT
      if [ ${var.market_place} -eq 1 ]; then
        aws ec2 stop-instances --instance-ids ${aws_instance.docker_image.id}
        aws ec2 wait instance-stopped --instance-ids ${aws_instance.docker_image.id}
        aws ec2 attach-volume --volume-id ${aws_ebs_volume.new_volume.id} --instance-id ${aws_instance.docker_image.id} --device /dev/sdg
        aws ec2 modify-instance-attribute --instance-id ${aws_instance.docker_image.id}  --block-device-mappings "[{\"DeviceName\": \"/dev/sdg\", \"Ebs\":{\"DeleteOnTermination\":true}}]"
        aws ec2 start-instances --instance-ids ${aws_instance.docker_image.id} 
      else
        aws ec2 attach-volume --volume-id ${aws_ebs_volume.new_volume.id} --instance-id ${aws_instance.docker_image.id} --device /dev/sdg
        aws ec2 modify-instance-attribute --instance-id ${aws_instance.docker_image.id}  --block-device-mappings "[{\"DeviceName\": \"/dev/sdg\", \"Ebs\":{\"DeleteOnTermination\":true}}]"
        aws ec2 start-instances --instance-ids ${aws_instance.docker_image.id} 
      fi
    EOT
  }
}

resource "null_resource" "detach_volume"{
  triggers = {
    instance_id = aws_instance.docker_image.id
    volume_id   = aws_ebs_volume.new_volume.id
  }
  provisioner "local-exec" {
  when    = destroy
    command = <<EOT
      aws ec2 stop-instances --instance-ids ${self.triggers.instance_id}
      aws ec2 wait instance-stopped --instance-ids ${self.triggers.instance_id}
      aws ec2 detach-volume --volume-id ${self.triggers.volume_id}
      aws ec2 wait volume-available --volume-id ${self.triggers.volume_id}
      aws ec2 delete-volume --volume-id ${self.triggers.volume_id}
    EOT
  }
}

resource "null_resource" "build_docker_image" {
  depends_on = [null_resource.manage_instance_marketplace]
  connection {
    user        = var.docker_vm_user
    private_key = file(local.private_key_filename)
    host        = element(flatten(aws_instance.docker_image.*.private_ip), 0)
    port        = var.cloud_ssh_port
    timeout     = "20m"
  }

  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_build.sh ; /tmp/install/docker_build.sh '${aws_instance.docker_image.private_ip}' '${var.service_userid}' '${var.webserverip}' '${var.cluster_create}'"]
  }
}

resource "null_resource" "setup_docker_imagenode" {
  depends_on = [null_resource.build_docker_image]
  connection {
    user        = var.vm_user
    private_key = file(local.private_key_filename)
    host        = aws_instance.compute.private_ip
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_setup.sh ; /tmp/install/docker_setup.sh '${var.webserverip}' '${var.service_userid}'"]
  }
}


resource "null_resource" "setup_docker_image_webserver" {
  count = var.webserver_create ? 1 :0
  depends_on = [null_resource.build_docker_image]
  connection {
    user        = var.vm_user
    private_key = file(local.private_key_filename)
    host        = var.webserverip
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_setup.sh ; /tmp/install/docker_setup.sh '${var.webserverip}' '${var.service_userid}'"]
  }
}
resource "null_resource" "setup_docker_image_webserver_local" {
  count = var.webserver_create ? 0 : 1
  depends_on = [null_resource.build_docker_image]
  provisioner "local-exec" {    
     command = "chmod 755 ${path.module}/install/docker_setup.sh ; ${path.module}/install/docker_setup.sh '${var.webserverip}' '${var.service_userid}'"
  }
}


