terraform {
  required_version = ">= 1.10.4, < 1.10.5"
  required_providers {
    aws = {
      version = "5.89.0"
    }
    null = {
      version = "3.2.3"
    }
    tls = {
      version = "4.0.6"
    }
    local = {
      version = "2.5.2"
    }
  }
}

provider "aws" {
  region = var.region
}

data "aws_caller_identity" "current" {
}


locals {
  key_name = "${var.imagetag}-auto-key"
  public_key_filename = format("%s/%s.pub", var.secrets, local.key_name)
  private_key_filename = format("%s/%s.pem", var.secrets, local.key_name)
  kms_key_id = ""
}

resource "aws_s3_bucket" "tf_backend_bucket" {
  bucket        = "ts-${var.customer}-${var.imagetag}"
  force_destroy = true
  tags = {
    Description        = "Terraform S3 Backend bucket which stores the terraform state for account ${data.aws_caller_identity.current.account_id}."
    ManagedByTerraform = "true"
    TerraformModule    = "terraform-aws-backend"
  }
  
  lifecycle {
    prevent_destroy = false
  }
}

data "aws_iam_policy_document" "tf_backend_bucket_policy" {

  statement {
    sid    = "RequireEncryptedTransport"
    effect = "Deny"

    actions = [
      "s3:*",
    ]

    resources = [
      "${aws_s3_bucket.tf_backend_bucket.arn}/*"
    ]

    condition {
      test     = "Bool"
      variable = "aws:SecureTransport"

      values = [
        false,
      ]
    }

    principals {
      type        = "*"
      identifiers = ["*"]
    }
  }

  statement {
    sid    = "RequireEncryptedStorage"
    effect = "Deny"

    actions = [
      "s3:PutObject",
    ]

    resources = [
      "${aws_s3_bucket.tf_backend_bucket.arn}/*"
    ]

    condition {
      test     = "StringNotEquals"
      variable = "s3:x-amz-server-side-encryption"

      values = [
        local.kms_key_id == "" ? "AES256" : "aws:kms",
      ]
    }

    principals {
      type        = "*"
      identifiers = ["*"]
    }
  }
}

resource "aws_iam_policy" "spillbox_kms_policy" {
  count      = var.kms_key_id != "" ? 1 : 0
  name       = "spillbox-${var.customer}-${var.imagetag}-kms"
  policy     = jsonencode({
    "Version": "2012-10-17",
    "Statement": [
      {
        "Effect": "Allow",
        "Action": [
          "kms:DescribeKey",
          "kms:ReEncrypt*",
          "kms:CreateGrant",
          "kms:Decrypt"
        ],
        "Resource": [
          var.kms_key_id
        ]
      }
    ]
  })
}

resource "aws_s3_bucket_policy" "tf_backend_bucket_policy" {
  bucket = aws_s3_bucket.tf_backend_bucket.id
  policy = data.aws_iam_policy_document.tf_backend_bucket_policy.json
}

resource "tls_private_key" "default" {
  algorithm = "RSA"
  rsa_bits  = 4096
}

resource "aws_iam_policy" "spillbox_s3_policy" {
  name        = "spillbox-${var.customer}-${var.imagetag}-s3-jfs"
  description = "Custom S3 permissions"
  policy = jsonencode({
    "Version" : "2012-10-17",
    "Statement" : [
      {
        "Effect" : "Allow",
        "Action" : [
          "s3:*"
        ],
        "Resource" : [
          "arn:aws:s3:::ts-*"
        ]
      }
    ]
  })
}


#creating a ssh-key pair
resource "local_file" "public_key_openssh" {
  depends_on           = [tls_private_key.default]
  sensitive_content    = tls_private_key.default.public_key_openssh
  filename             = local.public_key_filename
  file_permission      = "0600"
  directory_permission = "0700"
}

resource "local_file" "private_key_pem" {
  depends_on           = [tls_private_key.default]
  sensitive_content    = tls_private_key.default.private_key_pem
  filename             = local.private_key_filename
  file_permission      = "0600"
  directory_permission = "0700"
}

resource "null_resource" "chmod" {
  depends_on = [local_file.private_key_pem]

  triggers = {
    local_file_private_key_pem = local_file.private_key_pem.filename
  }

  provisioner "local-exec" {
    command = format(var.chmod_command, local.private_key_filename)
  }
}

resource "aws_key_pair" "import" {
 depends_on           = [local_file.public_key_openssh]
  key_name_prefix = local.key_name
  public_key      = join("", tls_private_key.default.*.public_key_openssh) 
}
