#!/bin/sh
#*************************************************************************
#
# Name: enable_aws_rc.sh
#
# (c) Copyright IBM Corp. 1992, 2017. All rights reserved.
# US Government Users Restricted Rights - Use, duplication or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.
#
# This script will enable resource connector to AWS for LSF.
#
#**************************************************************************

#--------------------------------------------------------------------------
#  Name: usage_awsrc
#
#  Description: This function shows help information of the script.
#--------------------------------------------------------------------------
usage_awsrc ()
{
echo "Usage: ${AWSRC_ENABLE_SCRIPT} -f aws_enable.config"
} #usage_awsrc


#--------------------------------------------------------------------------
#  Name: msg_info_log
#
#  Description: This function prints INFO messages to log file.
#--------------------------------------------------------------------------
msg_info_log ()
{
echo -e "$1" >> ${AWSRC_LOG}
} #msg_info_log

#--------------------------------------------------------------------------
#  Name: msg_info_output_log
#
#  Description: This function prints INFO messages to log file and stdout.
#--------------------------------------------------------------------------
msg_info_output_log ()
{
echo -e "$1" | tee -a ${AWSRC_LOG}
}

#--------------------------------------------------------------------------
#  Name: msg_error_output_log
#
#  Description: This function prints ERROR messages to log file and stderr.
#--------------------------------------------------------------------------
msg_error_output_log ()
{
echo -e "ERROR: $1" | tee -a ${AWSRC_LOG}
echo "Check $AWSRC_LOG for details."
#exit 1
}

#--------------------------------------------------------------------------
#  Name: error_case_awsrc
#  Synopsis: error_case_awsrc $parameter
#  Description: This function shows error cases of the script.
#--------------------------------------------------------------------------
error_case_awsrc ()
{
case "$1" in
    1)
        msg_error_output_log "The parameter ${2} is not defined in the $AWSRC_ENABLE_CONFIG."
        exit 1
     ;;
    2)
        msg_error_output_log "The value ${3} for the ${2} is not valid."
        exit 1
     ;;
    3)
        msg_error_output_log "Failed to update the file ${2}."
                msg_info_log "Backing up the ${2} to ${2}.aws_debug ..."
                cp -f "${2}" "${2}.aws_debug"
                rollback_awsrc
        exit 1
     ;;
    4)
        msg_error_output_log "The file ${2} does not exist or is not readable."
        exit 1
     ;;
    5)
        msg_error_output_log "Failed to enable resource connector for AWS."
        exit 1
     ;;
    6)
        msg_error_output_log "Cannot get the value of environment variable ${2}. \nSource the relative IBM Spectrum LSF shell script. \n * For csh or tcsh: 'source \$LSF_TOP/conf/cshrc.lsf' \n * For sh, ksh, or bash: 'source \$LSF_TOP/conf/profile.lsf'"
        exit 1
     ;;
    7)
        msg_error_output_log "Failed to backup the file ${2}."
                rollback_awsrc
        exit 1
     ;;
    8)
        msg_error_output_log "Failed to roll back the file ${2}."
        exit 1
     ;;
    9) 
        msg_error_output_log "The ${2} is not accessible. \nMake sure confPath and scriptPath are valid for AWS Provider in ${3}."
        exit 1
    ;;
    *)
            exit $1
         ;;
    esac
} #error_case_awsrc

#--------------------------------------------------------------------------
#  Name: getOwner
#
#  Synopsis: getOwner $file
#
#  Description:
#       This function gets ownership and group of the specified file, and
#       then remembers it.  The collected information will be used in
#       the call of setOwner (see below).
#--------------------------------------------------------------------------
getOwner()
{
    FILEOWNER=`ls -ln $1|awk ' { print $3 } ' `
    FILEGROUP=`ls -ln $1|awk ' { print $4 } ' `
    export FILEOWNER FILEGROUP
}

#--------------------------------------------------------------------------
#  Name: setOwner
#
#  Synopsis: setOwner $file
#
#  Description:
#       This function sets ownership and group of the specified file
#       according to the ownership information collected earlier by
#       getOwner (see above).
#--------------------------------------------------------------------------
setOwner()
{
    if [ -z $FILEOWNER ]; then
        return 0
    fi
    chown  $FILEOWNER:$FILEGROUP $1
}

#--------------------------------------------------------------------------
#  Name: getMode
#
#  Synopsis: getMode $file
#
#  Description:
#       This function gets mode of the specified file, and
#       then remembers it.  The collected information will be used in
#       the call of setMode (see below).
#--------------------------------------------------------------------------
getMode()
{
#    file=`basename $1`
#    dir=`dirname $1`
#    FILEMODE=`find $dir -name $file -maxdepth 1 -printf "%m\n"`
    FILEMODE=`ls -ln $1| awk ' { print $1 } '|cut -c2-10 `
    export FILEMODE
}

#--------------------------------------------------------------------------
#  Name: parseMode
#
#  Synopsis: parseMode $filemode
#
#  Description:
#       This function parses file permission string to number
#--------------------------------------------------------------------------
parseMode()
{
    _file_mode=$1
    _mode=0
    case "$_file_mode" in
        "---")
            _mode=0
            ;;
        "--x")
            _mode=1
            ;;
        "-w-")
            _mode=2
            ;;
        "-wx")
            _mode=3
            ;;
        "r--")
            _mode=4
            ;;
        "r-x")
            _mode=5
            ;;
        "rw-")
            _mode=6
            ;;
        "rwx")
            _mode=7
            ;;
    esac

    echo $_mode
} # parseMode

#--------------------------------------------------------------------------
#  Name: setMode
#
#  Synopsis: setMode $file
#
#  Description:
#       This function sets mode of the specified file
#       according to the mode collected earlier by
#       getOwner (see above).
#--------------------------------------------------------------------------
setMode()
{
    if [ -z $FILEMODE ]; then
        return
    else
        _u_mode=` echo $FILEMODE | cut -c1-3 `
        _g_mode=` echo $FILEMODE | cut -c4-6 `
        _o_mode=` echo $FILEMODE | cut -c7-9 `
        chmod  `parseMode $_u_mode``parseMode $_g_mode``parseMode $_o_mode` $1
    fi
}

#--------------------------------------------------------------------------
#  Name: getAttrib
#
#  Synopsis: getAttrib $file
#
#  Description: Get the ownership and mode of the file.
#--------------------------------------------------------------------------
getAttrib ()
{
_file="$1"
getOwner $_file
getMode $_file
}

#--------------------------------------------------------------------------
#  Name: setAttrib
#
#  Synopsis: setAttrib $file
#
#  Description: Set the ownership and mode of the file.
#--------------------------------------------------------------------------
setAttrib ()
{
_file="$1"
setOwner $_file
setMode $_file
}


chk_shell ()
{

_LSF_SHELL_CMDS="awk cat cp mv egrep grep mkdir rm ed chmod chown find"

SHELL_CMDS=""
_fail_sh_cmd=""

for _cmd in $_LSF_SHELL_CMDS
do
    which "$_cmd" >> $AWSRCNULL 2>&1

    if [ "$?" = "1" ] ; then
        _fail_sh_cmd="$_fail_sh_cmd $_cmd"
    else
        SHELL_CMDS="$SHELL_CMDS $_cmd"
    fi
done

if [ "$_fail_sh_cmd" != "" ]; then
    msg_error_output_log "The command cannot be found: $_fail_sh_cmd"
    exit 1
fi

#export SHELL_CMDS
return 0
}

#--------------------------------------------------------------------------
#  Name: backup_awsrc
#
#  Synopsis: backup_awsrc $file
#
#  Description: Backup file to $file.${_backup_file} before updating.
#--------------------------------------------------------------------------
backup_awsrc ()
{

_backup_file="$1"

msg_info_log "Backing up $_backup_file to ${_backup_file}.${backup_suffix} ..."

getAttrib "${_backup_file}"

if [ -r "${_backup_file}" ] ; then
    cp -f "${_backup_file}" "${_backup_file}.${backup_suffix}"
    if [ "$?" != "0" ]; then
        error_case_awsrc 7 "${_backup_file}"
    fi
else
    error_case_awsrc 4  "${_backup_file}"
fi

setAttrib "${_backup_file}.${backup_suffix}"

}

#--------------------------------------------------------------------------
#  Name: create_log_awsrc
#
#  Description: This function creates log file for the script.
#--------------------------------------------------------------------------
create_log_awsrc ()
{
if [ "x$PREFIX_LOG" = "x" ];then
    PREFIX_LOG=`pwd`
    for _tmpDir in  `pwd` /tmp ; do
        if `touch $_tmpDir/aws_enable.$$.AWSRCTMP > $AWSRCNULL 2>&1` ; then
            PREFIX_LOG=$_tmpDir
            date | tee -a $PREFIX_LOG/aws_enable.log > $AWSRCNULL 2>&1
            rm -rf $_tmpDir/aws_enable.$$.AWSRCTMP > $AWSRCNULL 2>&1
            break
        fi
    done
    AWSRC_LOG=$PREFIX_LOG/aws_enable.log
fi
export AWSRC_LOG
} #create_log_awsrc


#--------------------------------------------------------
#
# Name: get_value_awsrc
# Synopsis: get_value_awsrc "$FILE" "$OPTITON_NAME"
# Description:
#             It parses out the value from install.config file.
#----------------------------------------------------------
get_value_awsrc()
{
_para_file="$1"
_para_name="$2"
_para_val=""

if [ -r "$_para_file" ] ; then
    _para_val=`grep "^[ 	]*$_para_name=" ${_para_file} | sed -e "s/^.*=//g" | sed -e "s/\"//g"`
    # trim the trailing / if it is not "/"
    #if [ "$_para_val" != "/" ] ; then
    #    _para_val=`echo "$_para_val" | sed -e "s/\/$//g"`
    #fi
    #msg_info_log "The value of $_para_name is $_para_val."
else
    error_case_awsrc 4 "$_para_file"
fi
echo $_para_val
} #get_value_awsrc

#--------------------------------------------------------------------------
#  Name: get_clusters_name
#
#  Synopsis: get_clusters_name $LSF_SHARED
#
#  Environment Variable: LSF_CONFDIR
#
#  Description:
#       This function queries lsf.shared file and identifies all clusters.
#       It is different from the next function
#  Return Values:
#     0: get cluster and all custer names; otherwise, 1
#--------------------------------------------------------------------------
get_clusters_name ()
{
_lsf_shared="$1"

msg_info_log "Getting cluster name ..."

if [ ! -r "${_lsf_shared}" ]; then
    error_case_awsrc 4 "${_lsf_shared}"
fi

# get all the cluster names in CLUSTERNAME section of lsf.shared
# by 1) deleting line 1 to the line that contains Begin CLuster"
#    2) getting rid of cluster name
#    3) getting rid of all comment lines
#    4) getting rid of the line containing end cluster to the end of the file.
first_line=`head -n 1 $_lsf_shared`
echo $first_line | grep '^[ 	]*[Bb][Ee][Gg][Ii][Nn][ 	]*[Cc][Ll][Uu][Ss][Tt][Ee][Rr]' > /dev/null 2>&1
if [ "$?" = "0" ] ; then
    _lsf_clusters=`sed -e "1,1d" -e "/^[ 	]*[Ee][Nn][Dd][ 	]*[Cc][Ll][Uu][Ss][Tt][Ee][Rr]/,$ d" -e "/^[ 	]*[Cc][Ll][Uu][Ss][Tt][Ee][Rr][Nn][Aa][Mm][Ee][ 	]*.*/d" -e "/^[ 	]*#.*/d" -e "s/^[ 	]*//" -e "s/[ 	]*$//" $_lsf_shared | awk '{ print $1 }'`
else
    _lsf_clusters=`sed -e "1,/^[ 	]*[Bb][Ee][Gg][Ii][Nn][ 	]*[Cc][Ll][Uu][Ss][Tt][Ee][Rr]/d" -e "/^[ 	]*[Ee][Nn][Dd][ 	]*[Cc][Ll][Uu][Ss][Tt][Ee][Rr]/,$ d" -e "/^[ 	]*[Cc][Ll][Uu][Ss][Tt][Ee][Rr][Nn][Aa][Mm][Ee][ 	]*.*/d" -e "/^[ 	]*#.*/d" -e "s/^[ 	]*//" -e "s/[ 	]*$//" $_lsf_shared | awk '{ print $1 }'`
fi
_lsf_clusters=`echo $_lsf_clusters`
echo "$_lsf_clusters"
return 0
} # get_clusters_name

#--------------------------------------------------------------------------
#  Name: get_aws_path
#
#  Synopsis: get_aws_path $hostProv $opt_name
#
#  Environment Variable: LSF_TOP
#
#  Description:
#       This function get aws rc configured path in the hostProv files.
#  Return Values:
#     0: get related path; otherwise, 1
#--------------------------------------------------------------------------
get_aws_path(){

_hostProv="$1"
_opt_name="$2"
_opt_value=""

_found="n"
_get="n"

awsProvType="awsProv"

if [ ! -f "$_hostProv" ]; then
    error_case_awsrc 3 $_hostProv
    return 1
fi

exec 3<&0 < $_hostProv
while read LINE
do
case $LINE in
    *{*)
        if [ "$_found" = "y" ]; then
            break
        else
            _get="n"
            _opt_value=""
        fi
    ;;
    *\"type\"*:*\"$awsProvType\"*)
        _found="y"
    ;;
    *\"$_opt_name\"*:*)
        _opt_value=`echo $LINE |awk -F '"' '{ print $4 }'`; _get="y"
    ;;
    *}*)
        if [ "$_found" = "y" ]; then
            break
        fi
    ;;
    *) ;;
esac
done
exec 0<&3 3<&-

if [ "$_found" = "n" ]; then
    msg_error_output_log "No AWS provider found in $_hostProv."
    return 1
fi

if [ "$_get" = "n" ]; then
    msg_error_output_log "Cannot get $_opt_name of AWS provider from $_hostProv."
    return 1
fi

echo $_opt_value
} #get_aws_path

#--------------------------------------------------------------------------
#  Name: source_env_awsrc
#
#  Description: This function source environment variables for the script.
#--------------------------------------------------------------------------
source_env_awsrc ()
{

chk_shell

msg_info_log "Get the variables of running environment ..."

if [ "x$LSF_ENVDIR" = "x" ];then
    error_case_awsrc 6 LSF_ENVDIR
elif [ ! -r "$LSF_ENVDIR/profile.lsf" ];then
    error_case_awsrc 2 LSF_TOP $LSF_TOP
fi

LSF_TOP=`get_value_awsrc "$LSF_ENVDIR/lsf.conf" "LSF_TOP"`
if [ "x$LSF_TOP" = "x" ];then
    error_case_awsrc 6 LSF_TOP
else
    export LSF_TOP
fi

LSF_VERSION=`get_value_awsrc "$LSF_ENVDIR/lsf.conf" "LSF_VERSION"`
if [ "x$LSF_VERSION" = "x" ];then
   error_case_awsrc 6 LSF_VERSION
else
    export LSF_VERSION
fi

check_first_execution

LSF_CLUSTER_NAME=`get_clusters_name "$LSF_ENVDIR/lsf.shared"`
if [ "x$LSF_CLUSTER_NAME" = "x" ];then
    error_case_awsrc 6 LSF_CLUSTER_NAME
else
    if [ -r "$LSF_TOP/conf/lsf.cluster.${LSF_CLUSTER_NAME}" ];then
        export LSF_CLUSTER_NAME
    else
        error_case_awsrc 2 LSF_CLUSTER_NAME $LSF_CLUSTER_NAME
    fi
fi

} #source_env_awsrc

#--------------------------------------------------------------------------
#  Name: get_aws_provider
# Synopsis: get_aws_provider $example_hostProviders_file $hostProviders_file
#  Description: get aws host provide and check if valid.
#--------------------------------------------------------------------------

get_aws_provider ()
{

_example_prov=$1
_aws_prov=$2

if [ ! -f $_aws_prov ];then
    if [ -f $_example_prov ]; then
	    getAttrib $_example_prov	
        copy_aws_provider $_example_prov $_aws_prov
	        if [ "$?" != "0" ]; then
    		    msg_error_output_log "Failed copy $_example_prov to $_aws_prov."
        	    exit 1
    	    fi
	    setAttrib $_aws_prov
    else
        generate_hostProvider $_aws_prov
    fi
fi

awsProNum=`egrep -c '[ 	]*"type"[ 	]*:[ 	]*"awsProv"'  $_aws_prov`
if [ $awsProNum -gt 1 ]; then
        msg_error_output_log "More than one AWS provider is defined in $_aws_prov. This script only enables a single AWS provider."
        exit 1
elif [ $awsProNum -lt 1 ]; then
        append_awsProvide $_aws_prov
fi

AWS_confPath=`get_aws_path "$_aws_prov" confPath`
if [ "$?" != "0" ]; then
    msg_error_output_log "Failed to get confPath from $_aws_prov."
    exit 1
fi
case $AWS_confPath in 
    /*)
        AWSRC_CONFDIR="${AWS_confPath}/conf"
        ;;
    *)
        AWSRC_CONFDIR="$LSF_TOP/conf/${AWS_confPath}/conf"
        ;;
esac

AWS_scriptPath=`get_aws_path "$_aws_prov" scriptPath`
if [ "$?" != "0" ]; then
    msg_error_output_log "Failed to get scriptPath from $_aws_prov."
    exit 1
fi
case $AWS_scriptPath in 
    /*)
        AWSRC_SCRIPTDIR="${AWS_scriptPath}/scripts"
        ;;
    *)
        AWSRC_SCRIPTDIR="${LSF_TOP}/${LSF_VERSION}/${AWS_scriptPath}/scripts"
        ;;
esac

#AWSRC_CONFDIR="$LSF_TOP/conf/resource_connector/aws/conf"
for aws_dir in $AWSRC_CONFDIR $AWSRC_SCRIPTDIR; do
    if [ ! -d "$aws_dir" ]; then
        error_case_awsrc 9 "$aws_dir" "$_aws_prov"      
    fi
done
export AWSRC_CONFDIR AWSRC_SCRIPTDIR
} #get_aws_provider

#--------------------------------------------------------
# Name: check_first_execution
# Synopsis: check_first_execution
# Description: Check if the first time run the tool.
#---------------------------------------------------------

check_first_execution ()
{

aws_eabled_line=`cat "$LSF_ENVDIR/lsf.shared" | egrep -v "[^ 	]*#" | grep "$AWS_RC_FLAG"`
if [ "x$aws_eabled_line" != "x" ]; then
    msg_info_output_log ""
    msg_info_output_log "WARNING: AWS RC is already enabled in this LSF cluster."
    msg_info_output_log "The aws_enable.sh script is intended to enable AWS RC for the first time. Running this when AWS RC is already enabled might overwrite previous configurations."
    msg_info_output_log ""
    msg_info_output_log "Do you want to continue (Y/N)? "
    input_var=""
    while true; do
        read input_var
        if is_value_yes "$input_var"; then
            #msg_info_output_log "Starting to enable resource connector for AWS ..."
            break
        elif is_value_no "$input_var"; then
            msg_info_output_log "Cancelled. Exiting ..."
            exit 0
        else
            msg_info_output_log "Invalid input. To continue, press Y (otherwise, press N to exit):"
        fi
    done
fi
} #check_first_execution

#--------------------------------------------------------
# Name: is_value_yes
# Synopsis: is_value_yes $input_var
# Description: Check if the input is [Yy]([Ee][Ss]
#---------------------------------------------------------
is_value_yes() {
    local _input_var="$1"
    if echo "$_input_var" | egrep -q "^([Yy]([Ee][Ss])?|1)$"; then
        return 0
    fi
    return 1
} #is_value_yes

#--------------------------------------------------------
# Name: is_value_no
# Synopsis: is_value_no $input_var
# Description: Check if the input is [Nn][Oo]
#---------------------------------------------------------
is_value_no() {
    local _input_var="$1"
    if echo "$_input_var" | egrep -q "^([Nn][Oo]?|0)$"; then
        return 0
    fi
    return 1
} #is_value_no

#--------------------------------------------------------
# Name: check_file
# Synopsis: check_file $file
# Description: Check file if exist and have read permission.
#---------------------------------------------------------
check_file ()
{
_check_file_name=$1

if [ ! -r "${_check_file_name}" ];then
    error_case_awsrc 4 "${_check_file_name}"
fi

}
#--------------------------------------------------------
# Name: update_files_list
# Synopsis: update_files_list
# Description: Define the file list and name for updating or rolling back.
# Return value:
#       0 Success
#       1 Error
#---------------------------------------------------------
update_files_list ()
{

example_hostProviders_file="$LSF_TOP/conf/resource_connector/example_hostProviders.json"
hostProviders_file="$LSF_TOP/conf/resource_connector/hostProviders.json"

get_aws_provider $example_hostProviders_file $hostProviders_file

#awsprov_config_file="$AWSRC_CONFDIR/awsprov_config.json"

credentials_file="$AWSRC_CONFDIR/credentials"

awsprov_templates="$AWSRC_CONFDIR/awsprov_templates.json"

user_data_script="${AWSRC_SCRIPTDIR}/user_data.sh"

#lsb_modules_file="$LSF_TOP/conf/lsbatch/$LSF_CLUSTER_NAME/configdir/lsb.modules"

#lsf_shared_file="$LSF_TOP/conf/lsf.shared"

#lsfconf_file="$LSF_TOP/conf/lsf.conf"

#LSF_LIVE_CONFDIR=`get_value_awsrc "$LSF_ENVDIR/lsf.conf" "LSF_LIVE_CONFDIR"`
#if [ "x$LSF_LIVE_CONFDIR" != "x" -a -f "$LSF_LIVE_CONFDIR/lsbatch/$LSF_CLUSTER_NAME/configdir/lsb.queues" ];then
#    lsb_queues_file="$LSF_LIVE_CONFDIR/lsbatch/$LSF_CLUSTER_NAME/configdir/lsb.queues"
#else
#    lsb_queues_file="$LSF_TOP/conf/lsbatch/$LSF_CLUSTER_NAME/configdir/lsb.queues"
#fi

if [ "x$LSF_LIVE_CONFDIR" != "x" -a -f "$LSF_LIVE_CONFDIR/lsf.cluster.$LSF_CLUSTER_NAME" ];then
    lsf_cluster_file="$LSF_LIVE_CONFDIR/lsf.cluster.$LSF_CLUSTER_NAME"
else
    lsf_cluster_file="$LSF_TOP/conf/lsf.cluster.$LSF_CLUSTER_NAME"
fi

#policy_config_file="$LSF_TOP/conf/resource_connector/policy_config.json"
#if [ ! -f $policy_config_file ]; then
#    generate_policy $policy_config_file
#fi

#update_file_list="$awsprov_config_file $credentials_file $user_data_script $lsb_modules_file $lsf_shared_file $lsfconf_file $lsb_queues_file $lsf_cluster_file"
update_file_list="$credentials_file $user_data_script $lsf_cluster_file"

#for _file in $update_file_list $awsprov_templates $hostProviders_file; do
#    check_file $_file
#done

#export awsprov_config_file credentials_file user_data_script lsb_modules_file lsf_shared_file lsfconf_file lsb_queues_file lsf_cluster_file update_file_list
export credentials_file user_data_script lsf_cluster_file update_file_list

}

#--------------------------------------------------------------------------
#  Name: generate_policy
#
#  Synopsis: generate_policy $file
#
#  Description: generate policy file
#--------------------------------------------------------------------------
generate_policy ()
{

_policy_config=$1

getAttrib $LSF_ENVDIR/lsf.conf
cat <<POLICY > $_policy_config
{
  "UserDefinedScriptPath": "",
  "Policies":
  [
    {
      "Name": "Policy1",
      "Consumer":
      {
        "rcAccount": ["rc_compute"],
        "templateName": ["Template-VM-COMPUTE"],
        "provider": ["all"]
      },
      "MaxNumber": "10",
      "StepValue": "2:1"
    },
    {
      "Name": "Policy2",
      "Consumer":
      {
        "rcAccount": ["rc_nfs"],
        "templateName": ["Template-VM-NFS"],
        "provider": ["all"]
      },
      "MaxNumber": "10",
      "StepValue": "2:1"
    }
  ]
}
POLICY

if [ "$?" != "0" ]; then
    msg_error_output_log "Failed to generate $_policy_config."
    exit 1
fi
setAttrib $_policy_config

} #generate_policy

#--------------------------------------------------------------------------
#  Name: generate_hostProvider
#
#  Synopsis: generate_hostProvider $file
#-
#  Description: generate aws hostProvider file
#-------------------------------------------------------------------------
generate_hostProvider ()
{

_hostProvider=$1
getAttrib $LSF_ENVDIR/lsf.conf
cat <<PROVIDER > $_hostProvider
{
    "providers":[
        {
            "name": "aws",
            "type": "awsProv",
            "confPath": "resource_connector/aws",
            "scriptPath": "resource_connector/aws"
        }
    ]
}
PROVIDER

if [ "$?" != "0" ]; then
    msg_error_output_log "Failed to generate $_hostProvider."
    exit 1
fi
setAttrib $_hostProvider
} #generate_hostProvider

#--------------------------------------------------------------------------
#  Name: copy_aws_provider _example_prov _aws_prov
#
#  Synopsis: copy_aws_provider $file
#-
#  Description: copy aws hostProvider file from example hostProvider
#-------------------------------------------------------------------------
copy_aws_provider(){

_example_prov="$1"
_aws_prov="$2"
_found="n"

awsProvType="awsProv"

if [ ! -f "$_example_prov" ]; then
    error_case_awsrc 3 $_example_prov
    return 1
fi

exec 3<&0 < $_example_prov
while read LINE
do
case $LINE in
    *{*)
        if [ "$_found" = "y" ]; then
        cat <<PROVIDER >> $_aws_prov
        }
    ]
}
PROVIDER
            break
        else 
            _opt_value=""
            cat <<PROVIDER > $_aws_prov
{
    "providers":[
        {
PROVIDER
        fi
    ;;
    *\"type\"*:*\"$awsProvType\"*)
        echo "            $LINE" >> $_aws_prov
        _found="y"
    ;;
    *}*)
        if [ "$_found" = "y" ]; then
        cat <<PROVIDER >> $_aws_prov
        }
    ]
}
PROVIDER
            break
        fi
    ;;
    *)  echo "            $LINE" >> $_aws_prov
    ;;
esac
done
exec 0<&3 3<&-

if [ "$_found" = "n" ]; then
    msg_error_output_log "No AWS provider found in $_example_prov."
    return 1
fi

} #copy_aws_provider


#--------------------------------------------------------------------------
#  Name: append_awsProvide
#
#  Synopsis: append_awsProvide $file
#-
#  Description: generate aws hostProvider file
#--------------------------------------------------------------------------
append_awsProvide ()
{

_hostProvider=$1

    ed $_hostProvider <<ED_PROVIDER > $AWSRCNULL 2>&1
/[ 	]*"providers"[ 	]*:[ 	]*/a
        {
            "name": "aws",
            "type": "awsProv",
            "confPath": "resource_connector/aws",
            "scriptPath": "resource_connector/aws"
        },
.
w
q
ED_PROVIDER

if [ "$?" != "0" ]; then
    msg_error_output_log "Failed to add AWS provide for $_hostProvider."
    exit 1
fi

}
#--------------------------------------------------------------------------
#  Name: rollback_awsrc
#
#  Synopsis: rollback_awsrc $file
#
#  Description: Rollback file if some files are updated failed.
#--------------------------------------------------------------------------
rollback_awsrc ()
{

msg_info_output_log "Rolling back the updated files ..."

for _update_file in $update_file_list; do
    if [ -f "${_update_file}.${backup_suffix}" ] ; then
        getAttrib "${_update_file}.${backup_suffix}"
        cp -f "${_update_file}.${backup_suffix}" "${_update_file}"
        if [ "$?" != "0" ]; then
            error_case_awsrc 8 "$_file"
        fi
        setAttrib "${_update_file}"
        rm -f "${_update_file}.${backup_suffix}"
        msg_info_log "The ${_update_file} is rolled back."
    fi
done

}

#--------------------------------------------------------
# Name: pre_check_awsrc
#
# Description: The function perchecks variables for the script.
#----------------------------------------------------------
pre_check_awsrc ()
{

msg_info_log "Pre-checking definitions in the $AWSRC_ENABLE_CONFIG ..."

#check parameter AWS_LSF_TOP
AWS_LSF_TOP=`get_value_awsrc "$AWSRC_ENABLE_CONFIG" "AWS_LSF_TOP"`
if [ "x$AWS_LSF_TOP" = "x" ]; then
    error_case_awsrc 1 AWS_LSF_TOP
fi

#check parameter AWS_IAM_CREDENTIAL_ID, AWS_IAM_CREDENTIAL_KEY and AWS_FED_CREDENTIAL_SCRIPT
AWS_IAM_CREDENTIAL_ID=`get_value_awsrc "$AWSRC_ENABLE_CONFIG" "AWS_IAM_CREDENTIAL_ID"`
AWS_IAM_CREDENTIAL_KEY=`get_value_awsrc "$AWSRC_ENABLE_CONFIG" "AWS_IAM_CREDENTIAL_KEY"`
AWS_FED_CREDENTIAL_SCRIPT=`get_value_awsrc "$AWSRC_ENABLE_CONFIG" "AWS_FED_CREDENTIAL_SCRIPT"`
if [ "x$AWS_FED_CREDENTIAL_SCRIPT" = "x" ];then
    if [ "x$AWS_IAM_CREDENTIAL_ID" = "x" -o "x$AWS_IAM_CREDENTIAL_KEY" = "x" ];then
        msg_error_output_log "Either AWS_FED_CREDENTIAL_SCRIPT or both AWS_IAM_CREDENTIAL_ID & AWS_IAM_CREDENTIAL_KEY are required."
                exit 1
    fi
else
    if [ ! -f  $AWS_FED_CREDENTIAL_SCRIPT ];then
        error_case_awsrc 2 "AWS_FED_CREDENTIAL_SCRIPT" "$AWS_FED_CREDENTIAL_SCRIPT"
    fi
fi

#check parameter AWS_REGION
AWS_REGION=`get_value_awsrc "$AWSRC_ENABLE_CONFIG" "AWS_REGION"`
if [ "x$AWS_REGION" = "x" ]; then
    error_case_awsrc 1 AWS_REGION
fi

#check parameter AWS_SSH_KEY_PATH
AWS_SSH_KEY_PATH=`get_value_awsrc "$AWSRC_ENABLE_CONFIG" "AWS_SSH_KEY_PATH"`
if [ "x$AWS_SSH_KEY_PATH" = "x" ];then
    if [ ! -d $LSF_TOP/conf/resource_connector/aws/data ];then
        mkdir -p $LSF_TOP/conf/resource_connector/aws/data
    fi
    #AWS_SSH_KEY_PATH="$LSF_TOP/conf/resource_connector/aws/data"
else
    if [ ! -d  $AWS_SSH_KEY_PATH ];then
        error_case_awsrc 2 "AWS_SSH_KEY_PATH" "$AWS_SSH_KEY_PATH"
    fi
fi

#check parameter LSF_HOST_ADDR_RANGE
LSF_HOST_ADDR_RANGE=`get_value_awsrc "$AWSRC_ENABLE_CONFIG" "LSF_HOST_ADDR_RANGE"`
if [ "x$LSF_HOST_ADDR_RANGE" = "x" ];then
    error_case_awsrc 1 LSF_HOST_ADDR_RANGE
fi


} #pre_check_awsrc

#--------------------------------------------------------
#
# Name: update_awsprov_config
# Synopsis: update_awsprov_config "OPTION_NAME" "$OPTITON_NEW_VALUE"
# Description:
#             It updates awsprov_config.json file with a valid
#             new value. If the entry does not exist,
#             This function will add a new entry.
# Return value:
#       0 Success
#       1 Error
#----------------------------------------------------------
update_awsprov_config ()
{
_awsprov_config="$1"
_opt_name="$2"
_opt_val="$3"

_old_awsprov_config="/tmp/awsprov_config.$$.AWSRCTMP"

getAttrib $_awsprov_config

mv $_awsprov_config $_old_awsprov_config

if [ ! -f "$_old_awsprov_config" ]; then
    error_case_awsrc 3 $_awsprov_config
    return 1
fi

_done=n

exec 3<&0 <$_old_awsprov_config
while read LINE || [ -n "$LINE" ]
do
    case $LINE in
        *\"$_opt_name\"*:*,)
            echo "  \"$_opt_name\": \"$_opt_val\"," >> $_awsprov_config; _done=y
            ;;
        *\"$_opt_name\"*:*)
            echo "  \"$_opt_name\": \"$_opt_val\"" >> $_awsprov_config; _done=y
            ;;
        "{")
            echo "$LINE" >> $_awsprov_config
            ;;
        "}")
            echo "$LINE" >> $_awsprov_config
            ;;
        *)  echo "  $LINE" >> $_awsprov_config
            ;;
    esac
done
exec 0<&3 3<&-

rm -f $_old_awsprov_config

if [ "$_done" = "n" ] ; then
    ed $_awsprov_config <<ED_PARAMS_FILE > $AWSRCNULL 2>&1
/^[ 	]*{[ 	]*/a
  "$_opt_name": "$_opt_val",
.
w
q
ED_PARAMS_FILE
    if [ "$?" != "0" ]; then
        error_case_awsrc 3 $_awsprov_config
        return 1
    fi
fi

setAttrib $_awsprov_config

return 0
} #update_awsprov_config


#--------------------------------------------------------
#
# Name: update_credentials
# Synopsis: update_credentials "OPTION_NAME" "$OPTITON_NEW_VALUE"
# Description:
#             It updates credentials.json file with a valid
#             new value. If the entry does not exist,
#             This function will add a new entry.
# Return value:
#       0 Success
#       1 Error
#----------------------------------------------------------
update_credentials ()
{
_credentials="$1"
_opt_name="$2"
_opt_val="$3"

_old_credentials="/tmp/credentials.$$.AWSRCTMP"

getAttrib $_credentials
mv $_credentials $_old_credentials

if [ ! -f "$_old_credentials" ]; then
    error_case_awsrc 3 $_credentials
    return 1
fi

_done=n

exec 3<&0 <$_old_credentials
while read LINE
do
    case $LINE in
        *"$_opt_name="*)
            echo $_opt_name=$_opt_val >> $_credentials; _done=y
            ;;
        *)  echo "$LINE" >> $_credentials
            ;;
    esac
done
exec 0<&3 3<&-

rm -f $_old_credentials

if [ "$_done" = "n" ] ; then
    error_case_awsrc 3 $_credentials
        return 1
fi

setAttrib $_credentials

return 0
} #update_credentials

#--------------------------------------------------------
#
# Name: update_user_data
# Synopsis: update_user_data user_data.sh "OPTION_NAME" "$OPTITON_NEW_VALUE"
# Description:
#             It updates user_data.sh file with a valid
#             new value. If the entry does not exist,
#             This function will add a new entry.
# Return value:
#       0 Success
#       1 Error
#----------------------------------------------------------
update_user_data (){

_user_data="$1"
_opt_name="$2"
_opt_val="$3"

_old_user_data="/tmp/user_data.$$.LSFTMP"
getAttrib $_user_data

mv $_user_data $_old_user_data

if [ ! -f "$_old_user_data" ]; then
    error_case_awsrc 3 "$_user_data"
    return 1
fi

_done=n

exec 3<&0 <$_old_user_data
while read LINE
do
    _is_comment=`echo $LINE | grep "^[ 	]*#" 2>$AWSRCNULL`

    if [ "$_is_comment" != "" ]; then
        echo $LINE >> $_user_data
        continue
    fi

    case $LINE in
        *"$_opt_name="*)
            echo $_opt_name=$_opt_val >> $_user_data; _done=y
            ;;
        *)  echo $LINE >> $_user_data
            ;;
    esac
done
exec 0<&3 3<&-

rm -f $_old_user_data

if [ "$_done" = "n" ] ; then
    error_case_awsrc 3 $_user_data
        return 1
fi

LSF_MASTER_NODE=`get_value_awsrc "$LSF_ENVDIR/lsf.conf" "LSF_MASTER_LIST"`
for _lsf_master_node in $LSF_MASTER_NODE; do
    egrep "^[ 	]*echo \"${_lsf_master_node}\"[ ]*>" ${_user_data} > $AWSRCNULL 2>&1
    if [ "$?" = "0" ]; then
        : #echo "the user_data.sh script has been updated."
    else
        egrep "^#[ 	]*Start[ 	]*LSF[ 	]*Daemons" ${_user_data} > $AWSRCNULL 2>&1
        if [ "$?" = "0" ]; then
            ed $_user_data <<ED_PARAMS_FILE > $AWSRCNULL 2>&1
/^#[ 	]*Start[ 	]*LSF[ 	]*Daemons/i
echo "${_lsf_master_node}" >> \$LSF_ENVDIR/hostregsetup
.
w
q
ED_PARAMS_FILE

            if [ "$?" != "0" ]; then
                error_case_awsrc 3 $_user_data
                return 1
            fi
        fi
    fi
done
egrep "^[ 	]*lsreghost -s.*hostregsetup" ${_user_data} > $AWSRCNULL 2>&1
if [ "$?" = "0" ]; then
    : #echo "the user_data.sh script has been updated."
else
    egrep "^#[ 	]*Start[ 	]*LSF[ 	]*Daemons" ${_user_data} > $AWSRCNULL 2>&1
    if [ "$?" = "0" ]; then
        ed $_user_data <<ED_PARAMS_FILE > $AWSRCNULL 2>&1
/^#[ 	]*Start[ 	]*LSF[ 	]*Daemons/i
lsreghost -s \$LSF_ENVDIR/hostregsetup

.
w
q
ED_PARAMS_FILE

        if [ "$?" != "0" ]; then
            error_case_awsrc 3 $_user_data
            return 1
        fi
    fi
fi

setAttrib $_user_data

return 0

} #update_user_data

#-------------------------------------------------------------------
# Name: update_labmodules
# Synopsis: update_labmodules
# Description:
#    Add schmod_demand to lsb.modules;
# Return:
#    0 on success, 1 on error.
#------------------------------------------------------------------
update_lsbmodules ()
{
_lsb_modules_file="$1"

getOwner $_lsb_modules_file
#
# Add demand calculation scheduler module to the file lsb.modules
#
msg_info_log "Enabling schmod_demand in ${_lsb_modules_file} ..."
egrep "^[ 	]*schmod_demand" ${_lsb_modules_file} > $AWSRCNULL 2>&1
if [ "$?" = "0" ]; then
    msg_info_log "schmod_demand has been enabled."
else
    ed $_lsb_modules_file <<ED_PARAMS_FILE > $AWSRCNULL 2>&1
/^[ 	]*End[ 	]*PluginModule/i
schmod_demand                     ()                              ()
.
w
q
ED_PARAMS_FILE

    if [ "$?" != "0" ]; then
        error_case_awsrc 3 $_lsb_modules_file
        return 1
    fi
fi

return 0

} #update_lsbmodules


#----------------------------------------------------------------
# Name: update_lsfshared
# Synopsis: update_lsfshared
# Description:
#      Add awshost host type in lsf.shared
# Return value: 0 Succss 1 Error
#-----------------------------------------------------------------

update_lsfshared()
{
_lsf_shared_file="$1"

# add awshost host type into lsf.shared
msg_info_log "Adding $AWS_RC_FLAG in ${_lsf_shared_file} ..."
egrep "^[ 	]*$AWS_RC_FLAG" "${_lsf_shared_file}" >$AWSRCNULL 2>&1
if [ "$?" = "0" ]; then
    msg_info_log "The $AWS_RC_FLAG has been added in the lsf.shared."
else
    sharedKey=(`cat $_lsf_shared_file | grep '^[^#]' |grep -i "RESOURCENAME" |grep -i "INTERVAL" |grep -i "INCREASING" |sed s/#.*//i`)

    declare -A sharedValue
    sharedValue=([RESOURCENAME]="$AWS_RC_FLAG" [TYPE]="Boolean" [INTERVAL]="()" [INCREASING]="()" [DESCRIPTION]="(instances from AWS)")

    for i in `eval echo {0..$((${#sharedKey[*]}-1))}`
    do
        sharedLine="$sharedLine ${sharedValue["${sharedKey[i]}"]}"
    done
    if [ "x$sharedLine" = "x" ];then
        sharedLine="$AWS_RC_FLAG    Boolean    ()       ()       (instances from AWS)"
    fi

    ed "$_lsf_shared_file" <<ED_PARAMS_FILE >$AWSRCNULL 2>&1
/^[ 	]*End[ 	]*Resource/i
$sharedLine
.
w
q
ED_PARAMS_FILE
    if [ "$?" != "0" ]; then
        error_case_awsrc 3 $_lsf_shared_file
        return 1
    fi
fi
} #update_lsfshared

#--------------------------------------------------------
#
# Name: update_lsfconf
# Synopsis: update_lsfconf lsf.conf "OPTION_NAME" "$OPTITON_NEW_VALUE"
# Description:
#             It updates lsf.conf file with a valid
#             new value. If the entry does not exist,
#             This function will add a new entry.
# Return value:
#       0 Success
#       1 Error
#----------------------------------------------------------
update_lsfconf ()
{
_lsfconf="$1"
_opt_name="$2"
_opt_val="$3"

pws_end_lsfconf="###PWS Generated Section - DO NOT EDIT ABOVE THIS LINE - #END#LsfConf"
rmt_end_lsfconf="###RTM Generated Section - DO NOT EDIT ABOVE THIS LINE - #END#"

_old_lsfconf="/tmp/lsf.conf.$$.LSFTMP"
getAttrib $_lsfconf

msg_info_log "Updating lsf.conf to modify $_opt_name ..."
mv $_lsfconf $_old_lsfconf

if [ ! -f "$_old_lsfconf" ]; then
    error_case_awsrc 3 $_lsfconf
    return 1
fi

_done=n

exec 3<&0 <$_old_lsfconf
while read LINE
do
    _is_comment=`echo $LINE | grep "^[ 	]*#" 2>$AWSRCNULL`

    if [ "$_is_comment" != "" ]; then
        echo $LINE >> $_lsfconf
        continue
    fi

    case $LINE in
        *"$_opt_name="*)
            echo $_opt_name=$_opt_val >> $_lsfconf; _done=y
            ;;
        *)  echo $LINE >> $_lsfconf
            ;;
    esac
done
exec 0<&3 3<&-

rm -f $_old_lsfconf

if [ "$_done" = "n" ] ; then
    _has_pws_tag=`egrep "$pws_end_lsfconf" $_lsfconf`
    _has_rtm_tag=`egrep "$rmt_end_lsfconf" $_lsfconf`
    if [ "x$_has_pws_tag" != "x" ]; then
        ed "$_lsfconf" <<ED_PARAMS_FILE >$AWSRCNULL 2>&1
/$pws_end_lsfconf/i
$_opt_name=$_opt_val
.
w
q
ED_PARAMS_FILE
    elif [ "x$_has_rtm_tag" != "x" ];then 
        ed "$_lsfconf" <<ED_PARAMS_FILE >$AWSRCNULL 2>&1
/$rmt_end_lsfconf/i
$_opt_name=$_opt_val
.
w
q
ED_PARAMS_FILE
    else
        echo $_opt_name=$_opt_val >> $_lsfconf
    fi
    if [ "$?" != "0" ]; then
        error_case_awsrc 3 $_lsfconf
        return 1
    fi
fi

setAttrib $_lsfconf

return 0
} #update_lsfconf

#--------------------------------------------------------
#
# Name: update_lsbqueues
# Synopsis: update_lsbqueues lsb.queues
# Description:
#       Add a new sample queue awsexample for cloud bursting to AWS
# Return value:
#       0 Success
#       1 Error
#----------------------------------------------------------
update_lsbqueues ()
{

_lsb_queues="$1"
pws_end_lsbqueues="###PWS Generated Section - DO NOT EDIT ABOVE THIS LINE - #END#Queue"

msg_info_log "Adding a new sample queue \"awsexample\" for cloud bursting to AWS in $_lsb_queues ..."
egrep "^[ 	]*QUEUE_NAME[ 	]*=[ 	]*awsexample" ${_lsb_queues}  >$AWSRCNULL 2>&1
if [ "$?" = "0" ]; then
    msg_info_log "The awsexample queue has been added in the $_lsb_queues."
else
    egrep "$pws_end_lsbqueues" $_lsb_queues  >$AWSRCNULL 2>&1
    if [ "$?" = "0" ]; then
        ed "$_lsb_queues" <<ED_PARAMS_FILE >$AWSRCNULL 2>&1
/$pws_end_lsbqueues/i
Begin Queue
QUEUE_NAME   = awsexample
DESCRIPTION  = sample queue for cloud bursting to AWS
HOSTS        = all
RC_ACCOUNT   = rc_compute
RC_HOSTS     = awshost
RC_DEMAND_POLICY  = THRESHOLD[[2,0]]
End Queue
.
w
q
ED_PARAMS_FILE
    else
        echo "
Begin Queue
QUEUE_NAME   = awsexample
DESCRIPTION  = sample queue for cloud bursting to AWS
HOSTS        = all
RC_ACCOUNT   = rc_compute
RC_HOSTS     = awshost
RC_DEMAND_POLICY  = THRESHOLD[[2,0]]
End Queue
" >> ${_lsb_queues}
    fi
    if [ "$?" != "0" ]; then
        error_case_awsrc 3 "$_lsb_queues"
        return 1
    fi
fi

return 0
} #update_lsbqueues

#--------------------------------------------------------
#
# Name: update_lsfcluserconf
# Synopsis: update_lsfcluserconf lsf.cluster.$CLSUTER_NAME
# Description:
#       Update LSF_HOST_ADDR_RANGE in the lsf.cluster.$CLSUTER_NAME
# Return value:
#       0 Success
#       1 Error
#----------------------------------------------------------
update_lsfcluserconf()
{
_lsf_cluster_file="$1"
_opt_name="$2"
_opt_val="$3"
_lsf_cluster_tmp="/tmp/lsf.cluster.$$.LSFTMP"

getAttrib $_lsf_cluster_file

msg_info_log "Updating _lsf_cluster_file to modify $_opt_name ..."
mv $_lsf_cluster_file $_lsf_cluster_tmp

if [ ! -f "$_lsf_cluster_tmp" ]; then
    return 1
fi

_done=n

exec 3<&0 <$_lsf_cluster_tmp
while read LINE
do
    _is_comment=`echo $LINE | grep "^[ 	]*#" 2>$AWSRCNULL`

    if [ "$_is_comment" != "" ]; then
        echo $LINE >> $_lsf_cluster_file
        continue
    fi

    case $LINE in
        *"$_opt_name="*)
            echo $_opt_name=$_opt_val >> $_lsf_cluster_file; _done=y
            ;;
        *)  echo $LINE >> $_lsf_cluster_file
            ;;
    esac
done
exec 0<&3 3<&-

rm -f $_lsf_cluster_tmp

if [ "$_done" = "n" ] ; then
    ed $_lsf_cluster_file <<ED_SCRIPT2 >$AWSRCNULL 2>&1
/^[ 	]*End[ 	]*Parameters/i
LSF_HOST_ADDR_RANGE=$LSF_HOST_ADDR_RANGE
.
w
q
ED_SCRIPT2
    if [ "$?" != "0" ]; then
        error_case_awsrc 3 $_lsf_cluster_file
        return 1
    fi
fi

setAttrib $_lsf_cluster_file

} #update_lsfcluserconf


#--------------------------------------------------------
# Name: update_rc_awsrc
# Synopsis: update_rc_awsrc
# Description: It Invokes related functions to update resource connector configurations.
# Return value:
#       0 Success
#       1 Error
#----------------------------------------------------------
update_rc_awsrc ()
{

# update awsprov_config.json
#msg_info_log "Updating ${awsprov_config_file} ..."
#backup_awsrc "${awsprov_config_file}"
#update_awsprov_config "${awsprov_config_file}" AWS_CREDENTIAL_FILE "$AWSRC_CONFDIR/credentials"
#update_awsprov_config "${awsprov_config_file}" AWS_REGION $AWS_REGION
#if [ "x$AWS_SSH_KEY_PATH" != "x" ];then
#    update_awsprov_config "${awsprov_config_file}" AWS_KEY_FILE $AWS_SSH_KEY_PATH
#fi
#if [ "x$AWS_FED_CREDENTIAL_SCRIPT" != "x" ];then
#    update_awsprov_config "${awsprov_config_file}" AWS_CREDENTIAL_SCRIPT $AWS_FED_CREDENTIAL_SCRIPT
#fi
# update credentials
msg_info_log "Updating $credentials_file ..."
backup_awsrc "$credentials_file"
update_credentials "$credentials_file" aws_access_key_id "$AWS_IAM_CREDENTIAL_ID"
update_credentials "$credentials_file" aws_secret_access_key "$AWS_IAM_CREDENTIAL_KEY"

#update user_data.sh script
msg_info_log "Updating $user_data_script ..."
backup_awsrc "$user_data_script"
update_user_data "$user_data_script" LSF_TOP "$AWS_LSF_TOP"

} #update_rc_awsrc

#--------------------------------------------------------
# Name: update_lsf_awsrc
# Synopsis: update_lsf_awsrc
# Description: It Invokes related functions to update LSF configurations.
# Return value:
#       0 Success
#       1 Error
#----------------------------------------------------------
update_lsf_awsrc ()
{

#update lsb.modules file
#msg_info_log "Updating $lsb_modules_file ..."
#backup_awsrc "$lsb_modules_file"
#update_lsbmodules "$lsb_modules_file"

#update lsf.shared file
#msg_info_log "Updating $lsf_shared_file ..."
#backup_awsrc "$lsf_shared_file"
#update_lsfshared "$lsf_shared_file"

#update lsf.conf file
#msg_info_log "Updating $lsfconf_file ..."
#backup_awsrc "$lsfconf_file"
#update_lsfconf "$lsfconf_file" LSB_RC_EXTERNAL_HOST_FLAG "$AWS_RC_FLAG"
#update_lsfconf "$lsfconf_file" ENABLE_DYNAMIC_HOSTS "Y"
#update_lsfconf "$lsfconf_file" LSF_REG_FLOAT_HOSTS "Y"
#update_lsfconf "$lsfconf_file" LSB_RC_EXTERNAL_HOST_IDLE_TIME "1"
#update_lsfconf "$lsfconf_file" LSF_DYNAMIC_HOST_TIMEOUT "10m"
#update_lsfconf "$lsfconf_file" LSB_RC_EXTERNAL_HOST_MAX_TTL "0"

#LSF_DYNAMIC_HOST_WAIT_TIME=`get_value_awsrc "$LSF_ENVDIR/lsf.conf" "LSF_DYNAMIC_HOST_WAIT_TIME"`
#if [ "x$LSF_DYNAMIC_HOST_WAIT_TIME" = "x" ];then
#update_lsfconf "$lsfconf_file" LSF_DYNAMIC_HOST_WAIT_TIME "2"
#fi

#update lsb.queues file
#msg_info_log "Updating $lsb_queues_file ..."
#backup_awsrc "$lsb_queues_file"
#update_lsbqueues "$lsb_queues_file"

#update lsf.cluster.$LSF_CLUSTER_NAME file
msg_info_log "Updating $lsf_cluster_file ..."
backup_awsrc "$lsf_cluster_file"
update_lsfcluserconf "$lsf_cluster_file" LSF_HOST_ADDR_RANGE "$LSF_HOST_ADDR_RANGE"

} #update_lsf_awsrc


#--------------------------------------------------------
# Name: update_lsf_awsrc
#
# Description:
# This is MAIN function that invokes other function
# to enable resource connector for AWS.
#----------------------------------------------------------
enable_awsrc ()
{
msg_info_output_log ""
msg_info_output_log "The aws_enable.sh script enables the AWS RC for the first time and enables a single AWS provider."
source_env_awsrc
msg_info_output_log ""
msg_info_output_log "Starting to enable resource connector for AWS ..."
update_files_list
msg_info_output_log ""
msg_info_output_log "Validating parameters for resource connector for AWS ..."
pre_check_awsrc
msg_info_output_log ""
msg_info_output_log "Updating resource connector configuration ..."
update_rc_awsrc
msg_info_output_log ""
msg_info_output_log "Updating LSF configuration to support AWS resource connector ..."
update_lsf_awsrc
msg_info_output_log ""
msg_info_output_log "Resource connector for AWS is enabled."
msg_info_output_log ""
msg_info_output_log "This script updated configuration files in the cluster."
msg_info_output_log "To make these changes take effect, you must restart the LSF daemons on the master host."
msg_info_output_log ""
}

################## Main Entry #######################
AWSRC_ENABLE_SCRIPT=$0
AWSRC_ENABLE_CONFIG=$2
AWS_RC_FLAG="awshost"
AWSRCNULL=/dev/null
backup_suffix="aws_backup"
export AWSRC_ENABLE_SCRIPT AWSRC_ENABLE_CONFIG AWS_RC_FLAG AWSRCNULL backup_suffix

create_log_awsrc

if [ $# -eq 2 ]; then
    if [ "$1" = "-f" ]; then
        if [ -r "$AWSRC_ENABLE_CONFIG" ];then
            enable_awsrc
        else
            error_case_awsrc 4 "$AWSRC_ENABLE_CONFIG"
        fi
    else
        usage_awsrc
    fi
else
    usage_awsrc
fi
