terraform {
  required_version = ">= 1.10.4, < 1.10.5"
  required_providers {
    aws = {
      version = "5.62.0"
    }
    null = {
      version = "3.2.3"
    }
    template = {
      version = "2.2.0"
    }
  }
  backend "s3" {
    key     = "states/image-compute"
    encrypt = true
  }
}

provider "aws" {
  region = var.region
}

locals {
  master_private_ip = var.webserver_create == "true" ? element(flatten(data.terraform_remote_state.webserver.outputs.master_private_ip), 0) : var.webserver[0]
}

data "terraform_remote_state" "webserver" {
  backend = "s3"

  config = {
    bucket = "ts-${var.customer}-${var.servername}"
    region = var.region
    key    = "states/webserver"
  }
}

data "template_file" "config" {
  template = file("${path.module}/init.tpl")
  vars = {
    cloud_ssh_port = var.cloud_ssh_port
  }
}

data "template_cloudinit_config" "config" {
  gzip          = true
  base64_encode = true

  part {
    content_type = "text/x-shellscript"
    content      = data.template_file.config.rendered
  }
}

resource "aws_instance" "updatescaleup" {
  instance_type          = var.compute_instance_type == "" ? "m5.large" : var.compute_instance_type
  ami                    = var.base_ami_name == "" ? "ami-0dee0f906cf114191" : var.base_ami_name
  key_name               = data.terraform_remote_state.webserver.outputs.key_name
  vpc_security_group_ids = [data.terraform_remote_state.webserver.outputs.public_security_group_id]
  subnet_id              = data.terraform_remote_state.webserver.outputs.public_subnet_id

  tags = {
    Name = "${var.servername}-image"
  }

  user_data = data.template_cloudinit_config.config.rendered

  root_block_device {
    volume_size           = var.compute_volume_size
    kms_key_id            = var.kms_key_id == "" ? null : var.kms_key_id
    encrypted             = var.kms_key_id == "" ? false : true
    delete_on_termination = true
  }
  provisioner "local-exec" {
    command = "sleep 120 && echo done waiting master VM ready"
  }
}

resource "null_resource" "update-image" {
  connection {
    user        = var.vm_user
    private_key = file(var.project_private_key)
    host        = element(flatten(aws_instance.updatescaleup.*.private_ip), 0)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }
  provisioner "file" {
    source      = "${path.module}/../../../../config/lsf_std_entitlement.dat"
    destination = "/tmp/lsf_std_entitlement.dat"
  }
  provisioner "remote-exec" {
    inline = ["chmod 775 /tmp/install/update-scaleupimage.sh ; /tmp/install/update-scaleupimage.sh '${var.spillbox_container}' '${var.vm_user}' '${local.master_private_ip}' '${var.docker_registry}' '${var.docker_images}' '${var.service_userid}' '${var.cluster_create}'"]
  }
}

resource "null_resource" "update-image_singularity_sif" {
  count = var.spillbox_container == "singularity" ? 1 : 0
  provisioner "file" {
    source      = "${var.singularity_sifpath}"
    destination = "/spillbox_sif/${basename(var.singularity_sifpath)}"
  }
}

resource "aws_instance" "docker_image" {
  count                  = var.spillbox_container == "docker" ? 1 : 0
  ami                    = var.docker_baseami_id
  instance_type          = var.compute_instance_type == "" ? "m5.large" : var.compute_instance_type
  subnet_id              = var.subnet_id
  vpc_security_group_ids = [var.security_group_id]
  key_name               = data.terraform_remote_state.webserver.outputs.key_name

  root_block_device {
    volume_size           = var.compute_volume_size
    kms_key_id            = var.kms_key_id == "" ? null : var.kms_key_id
    encrypted             = var.kms_key_id == "" ? false : true
    delete_on_termination = true
  }

  tags = {
    Name = "spillbox-dockerimage-${var.servername}"
  }
  provisioner "local-exec" {
    command = "sleep 30 && echo done waiting master VM ready"
  }
}

resource "aws_ebs_volume" "new_volume" {
  count              = var.spillbox_container == "docker" ? 1 : 0
  depends_on         = [aws_instance.docker_image]
  snapshot_id        = var.snapshot_id
  size               = var.compute_volume_size
  availability_zone  = aws_instance.docker_image[0].availability_zone
  kms_key_id         = var.kms_key_id == "" ? null : var.kms_key_id
  encrypted          = var.kms_key_id == "" ? false : true

  tags = {
    Name = "spillbox"
  }
}

resource "null_resource" "manage_instance_marketplace" {
  count      = var.spillbox_container == "docker" ? 1 : 0
  depends_on = [aws_instance.docker_image, aws_ebs_volume.new_volume]

  provisioner "local-exec" {
    command = <<EOT
      aws ec2 stop-instances --instance-ids ${aws_instance.docker_image[0].id}
      aws ec2 wait instance-stopped --instance-ids ${aws_instance.docker_image[0].id}
      aws ec2 attach-volume --volume-id ${aws_ebs_volume.new_volume[0].id} --instance-id ${aws_instance.docker_image[0].id} --device /dev/sdg
      aws ec2 modify-instance-attribute --instance-id ${aws_instance.docker_image[0].id}  --block-device-mappings "[{\"DeviceName\": \"/dev/sdg\", \"Ebs\":{\"DeleteOnTermination\":true}}]"
      aws ec2 start-instances --instance-ids ${aws_instance.docker_image[0].id}
    EOT
  }
}

resource "null_resource" "detach_volume"{
  count         = var.spillbox_container == "docker" ? 1 : 0
  triggers = {
    instance_id = aws_instance.docker_image[0].id
    volume_id   = aws_ebs_volume.new_volume[0].id
  }
  provisioner "local-exec" {
  when    = destroy
    command = <<EOT
      aws ec2 stop-instances --instance-ids ${self.triggers.instance_id}
      aws ec2 wait instance-stopped --instance-ids ${self.triggers.instance_id}
      aws ec2 detach-volume --volume-id ${self.triggers.volume_id}
      aws ec2 wait volume-available --volume-id ${self.triggers.volume_id}
      aws ec2 delete-volume --volume-id ${self.triggers.volume_id}
    EOT
  }
}

resource "null_resource" "build_docker_image" {
  count         = var.spillbox_container == "docker" ? 1 : 0
  depends_on    = [null_resource.manage_instance_marketplace]
  connection {
    user        = var.docker_vm_user
    private_key = file(var.project_private_key)
    host        = element(flatten(aws_instance.docker_image[0].*.private_ip), 0)
    port        = var.cloud_ssh_port
    timeout     = "20m"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_build.sh ; /tmp/install/docker_build.sh '${aws_instance.docker_image[0].private_ip}' '${var.service_userid}' '${var.webserverip}' '${var.cluster_create}' '${var.region}'"]
  }
}

resource "null_resource" "setup_docker_imagenode" {
  count         = var.spillbox_container == "docker" ? 1 : 0
  depends_on    = [null_resource.update-image, null_resource.build_docker_image]
  connection {
    user        = var.vm_user
    private_key = file(var.project_private_key)
    host        = element(flatten(aws_instance.updatescaleup.*.private_ip), 0)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_setup.sh ; /tmp/install/docker_setup.sh '${var.webserverip}' '${var.service_userid}'"]
  }
}


resource "null_resource" "setup_docker_image_webserver" {
  count      = var.spillbox_container == "docker" && var.webserver_create == "true" ? 1 : 0
  depends_on = [null_resource.build_docker_image]
  connection {
    user        = var.vm_user
    private_key = file(var.project_private_key)
    host        = var.webserverip
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_setup.sh ; /tmp/install/docker_setup.sh '${var.webserverip}' '${var.service_userid}'"]
  }
}
resource "null_resource" "setup_docker_image_webserver_local" {
  count = var.spillbox_container == "docker" && var.webserver_create == "false" ? 1 : 0
  depends_on = [null_resource.build_docker_image]
  provisioner "local-exec" {
     command = "chmod 755 ${path.module}/install/docker_setup.sh ; ${path.module}/install/docker_setup.sh '${var.webserverip}' '${var.service_userid}'"
  }
}
