terraform {
  required_version = ">= 1.10.4, < 1.10.5"
  required_providers {
    aws = {
      version = "5.62.0"
    }
    null = {
      version = "3.2.3"
    }
    tls = {
      version = "4.0.6"
    }
    local = {
      version = "2.5.2"
    }
    template = {
      version = "2.2.0"
    }
  }
  backend "s3" {
    key     = "states/webserver"
    encrypt = true
  }
}

provider "aws" {
  region = var.region
}
resource "aws_iam_role" "spillbox_s3_role" {
  name = "spillbox-${var.customer}-${var.servername}-s3-jfs-role"

  assume_role_policy = jsonencode({
    Version = "2012-10-17",
    Statement = [
      {
        Effect = "Allow",
        Principal = {
          Service = "ec2.amazonaws.com"
        },
        Action = "sts:AssumeRole"
      }
    ]
  })
}
resource "aws_iam_policy" "spillbox_s3_policy" {
  name        = "spillbox-${var.customer}-${var.servername}-s3-jfs-policy"
  description = "Combined S3 and EBS permissions"

  policy = jsonencode({
    "Version" : "2012-10-17",
    "Statement" : [
      {
        "Effect" : "Allow",
        "Action" : [
          "s3:*"
        ],
        "Resource" : [
          "arn:aws:s3:::ts-*"
        ]
      },
      {
        "Effect" : "Allow",
        "Action" : [
          "ec2:CreateVolume",
          "ec2:AttachVolume",
          "ec2:DescribeVolumes",
          "ec2:DetachVolume",
          "ec2:DeleteVolume"
        ],
        "Resource" : "*"
      }
    ]
  })
}

resource "aws_iam_role_policy_attachment" "s3_policy_attachment" {
  role       = aws_iam_role.spillbox_s3_role.name
  policy_arn = aws_iam_policy.spillbox_s3_policy.arn
}
resource "aws_iam_instance_profile" "spillbox_s3_instance_profile" {
  name = "${var.customer}-${var.servername}-profile"
  role = aws_iam_role.spillbox_s3_role.name
}

module "subnet" {
  count              = var.enable_sgo == "false" ? 0 : var.subnet_id == "" ? 1 : 0
  source             = "../network/"
  customer           = "${var.customer}-${var.servername}"
  region             = var.region
  vpc_id             = var.vpc_id
  public_subnet_cidr = var.public_subnet_cidr
}

module "security_group" {
  count              = var.enable_sgo == "false" ? 0 : var.security_group_id == "" ? 1 : 0
  source             = "../security"
  vpc-id             = var.vpc_id == "" ? module.subnet[0].vpc_id : var.vpc_id
  public_cidr_blocks = var.public_subnet_cidr != "" ? [var.public_subnet_cidr] : module.subnet[0].public_cidr_blocks
  customer           = "${var.customer}-${var.servername}"
  public_subnet_id   = var.subnet_id == "" ? module.subnet[0].public_id : var.subnet_id
  cloud_ssh_port     = var.cloud_ssh_port
}

module "ssh_key_pair" {
  source           = "../ssh_key_pair"
  region           = var.region
  secrets          = var.secrets
  cloud_provider   = var.cloud_provider
  customer         = var.customer
  servername       = var.servername
  generate_ssh_key = "false"
}

resource "aws_fsx_lustre_file_system" "fsxshared" {
  storage_capacity            = 6000
  storage_type                = "HDD"
  per_unit_storage_throughput = 12
  drive_cache_type            = "NONE"
  subnet_ids                  = [var.subnet_id] == [""] ? [module.subnet[0].public_id] : [var.subnet_id]
  security_group_ids          = [var.security_group_id] == [""] ? [module.security_group[0].public_id] : [var.security_group_id]
  data_compression_type       = "LZ4"
  deployment_type             = "PERSISTENT_1"
  tags = {
    Name = "${var.customer}-${var.servername}"
  }
}

module "webserver" {
  depends_on             = [aws_fsx_lustre_file_system.fsxshared]
  count                  = var.enable_sgo == "false" ? 0 : 1
  source                 = "../../aws/compute_fw/"
  webserver_create       = var.webserver_create
  number                 = "1"
  Name                   = "${var.servername}-hub"
  subnet_id              = var.subnet_id == "" ? module.subnet[0].public_id : var.subnet_id
  customer               = "${var.customer}-${var.servername}"
  key_name               = module.ssh_key_pair.key_name
  vpc_security_group_ids = var.security_group_id == "" ? [module.security_group[0].public_id] : [var.security_group_id]
  ami_name               = var.base_ami_name == "" ? "ami-0dee0f906cf114191" : var.base_ami_name
  instance_type          = var.webserver_instance_type == "" ? var.splpipe_instance_type[var.model] : var.webserver_instance_type
  iam_instance_profile   = aws_iam_instance_profile.spillbox_s3_instance_profile.name
  vm_user                = var.vm_user
  project                = "not needed for aws"
  public_key             = module.ssh_key_pair.public_key_filename
  private_key            = module.ssh_key_pair.private_key_filename
  kms_key_id             = var.kms_key_id
  zone                   = "Not needed for aws"
  dnsname                = aws_fsx_lustre_file_system.fsxshared.dns_name
  mountname              = aws_fsx_lustre_file_system.fsxshared.mount_name
  cloud_ssh_port         = var.cloud_ssh_port
  root_volume_size       = var.webserver_volume_size
  docker_registry        = var.docker_registry
  docker_images          = var.docker_images
  service_userid         = var.service_userid
  spillbox_container     = var.spillbox_container
  singularity_sifpath    = var.singularity_sifpath
  cluster_create         = var.cluster_create
}

resource "aws_iam_role" "spotfleet_role" {
  name = "spillbox-${var.customer}-${var.servername}-spotfleetrole"
  assume_role_policy = jsonencode({
    Version = "2012-10-17",
    Statement = [
      {
        Effect = "Allow",
        Principal = {
          Service = "spotfleet.amazonaws.com"
        },
        Action = "sts:AssumeRole"
      }
    ]
  })
}

resource "aws_iam_policy" "spotfleet_policy" {
  name        = "spillbox-${var.customer}-${var.servername}-spotfleet"
  description = "Access to bid, start and terminate spot fleet"
  policy = jsonencode({
    "Version" : "2012-10-17",
    "Statement" : [
      {
        "Effect" : "Allow",
        "Action" : [
          "ec2:DescribeImages",
          "ec2:DescribeSubnets",
          "ec2:RequestSpotInstances",
          "ec2:TerminateInstances",
          "ec2:DescribeInstanceStatus",
          "ec2:CreateTags",
          "ec2:RunInstances"
        ],
        "Resource" : "*"
      }
    ]
  })
}

resource "aws_iam_role_policy_attachment" "custom_policy_attachment" {
  role       = aws_iam_role.spotfleet_role.name
  policy_arn = aws_iam_policy.spotfleet_policy.arn
}

resource "aws_iam_role" "ebs_management_role" {
  name = "spillbox-${var.customer}-${var.servername}-ebs-volume-role"
  assume_role_policy = jsonencode({
    Version = "2012-10-17",
    Statement = [
      {
        Effect = "Allow",
        Principal = {
          Service = "ec2.amazonaws.com"
        },
        Action = "sts:AssumeRole"
      }
    ]
  })
}
resource "aws_iam_policy" "ebs_management_policy" {
  name        = "spillbox-${var.customer}-${var.servername}-ebs-volume-policy"
  description = "Permissions to manage EBS volumes"
  policy = jsonencode({
    Version = "2012-10-17",
    Statement = [
      {
        Effect = "Allow",
        Action = [
          "ec2:CreateVolume",
          "ec2:AttachVolume",
          "ec2:DescribeVolumes",
          "ec2:DetachVolume",
          "ec2:DeleteVolume"
        ],
        Resource = "*"
      }
    ]
  })
}
resource "aws_iam_role_policy_attachment" "ebs_policy_attachment" {
  role       = aws_iam_role.ebs_management_role.name
  policy_arn = aws_iam_policy.ebs_management_policy.arn
}
resource "aws_iam_instance_profile" "ebs_instance_profile" {
  name = "${var.customer}-${var.servername}-ebs-instance-profile"
  role = aws_iam_role.ebs_management_role.name
}
