locals {
  is_image_gallery = can(regex("Microsoft.Compute/galleries/([^/]+)/images/([^/]+)", var.ami_name)) ? true : false
  is_image_custom = can(regex("Microsoft.Compute/images/([^/]+)", var.ami_name)) ? true : false
  is_image_market = can(regex(":", var.ami_name)) ? true : false
  plan_given = var.ami_plan == "" ? "" : var.ami_plan
}

data "azurerm_shared_image" "shared_image" {
  count               = local.is_image_gallery ? 1 : 0
  name                = element(split("/", var.ami_name), 10)
  gallery_name        = element(split("/", var.ami_name), 8)
  resource_group_name = element(split("/", var.ami_name), 4)
}
/*
data "azurerm_image" "search" {
  count              = local.is_image_custom ? 1 : 0
  name                = element(split("/", var.ami_name), 8)
  resource_group_name = element(split("/", var.ami_name), 4)
}
*/
resource "azurerm_virtual_machine" "instance" {
  count                         = var.webserver_create == "true" ? 1 : 0
  name                          = format("${var.Name}-%02d", count.index)
  resource_group_name           = var.project
  location                      = var.zone
  vm_size                       = var.instance_type
  delete_os_disk_on_termination = true

  network_interface_ids = ["${element(azurerm_network_interface.public.*.id, count.index)}"]

  os_profile {
    computer_name  = format("${var.Name}%02d", count.index)
    admin_username = var.vm_user
    custom_data    = data.template_cloudinit_config.config.rendered
  }

  dynamic "storage_image_reference" {
    for_each = var.ami_name != null ? [1] : []

    content {
      id        = !local.is_image_market ? var.ami_name : null
      publisher = local.is_image_market ? element(split(":", var.ami_name), 0) : null
      offer     = local.is_image_market ? element(split(":", var.ami_name), 1) : null
      sku       = local.is_image_market ? element(split(":", var.ami_name), 2) : null
      version   = local.is_image_market ? element(split(":", var.ami_name), 3) : null
    }
  }
  plan {
    name      = local.plan_given == "" ? local.is_image_gallery ? data.azurerm_shared_image.shared_image[0].purchase_plan[0].name : element(split(":", var.ami_name), 2) : element(split(":", local.plan_given), 0)
    product   = local.plan_given == "" ? local.is_image_gallery ? data.azurerm_shared_image.shared_image[0].purchase_plan[0].product : element(split(":", var.ami_name), 1) : element(split(":", local.plan_given), 1)
    publisher = local.plan_given == "" ? local.is_image_gallery ? data.azurerm_shared_image.shared_image[0].purchase_plan[0].publisher : element(split(":", var.ami_name), 0) : element(split(":", local.plan_given), 2)
  }

  os_profile_linux_config {
    disable_password_authentication = true
    ssh_keys {
      path     = "/home/${var.vm_user}/.ssh/authorized_keys"
      key_data = file("${var.public_key}")
    }
  }

  storage_os_disk {
    name              = format("${var.Name}-%02d-osdisk", count.index)
    caching           = "ReadWrite"
    create_option     = "FromImage"
    managed_disk_type = "Standard_LRS"
    disk_size_gb      = var.root_volume_size
  }
}

resource "azurerm_public_ip" "public" {
  name                = "${var.Name}-public-ip"
  location            = var.zone
  resource_group_name = var.project
  allocation_method   = "Static"
}

resource "azurerm_network_interface" "public" {
  count               = var.number
  name                = format("${var.Name}-public-nic-%02d", count.index)
  location            = var.zone
  resource_group_name = var.project

  ip_configuration {
    name                          = "${var.Name}-public-ip-configuration"
    subnet_id                     = var.subnet_id
    private_ip_address_allocation = "Dynamic"
  }
}

resource "null_resource" "fsx_client" {
  count      = var.webserver_create == "true" ? 1 : 0
  depends_on = [azurerm_virtual_machine.instance]
  connection {
    user        = var.vm_user
    private_key = file(var.private_key)
    host        = element(flatten(azurerm_network_interface.public.*.private_ip_address), 0)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = ["chmod 755 /tmp/install/install.sh; /tmp/install/install.sh '${var.spillbox_container}' '${var.vm_user}' '${var.docker_registry}' '${var.docker_images}' '${var.tenantid}' '${var.clientid}' '${var.clientsecret}' '${var.service_userid}'"]
  }
}

resource "null_resource" "fsx_client_singularity_sif" {
  count = var.spillbox_container == "singularity" ? 1 : 0
  provisioner "file" {
    source      = "${var.singularity_sifpath}"
    destination = "/spillbox_sif/${basename(var.singularity_sifpath)}"
  }
}

resource "null_resource" "fsx_client_local" {
  count = var.webserver_create == "false" ? 1 : 0

  provisioner "local-exec" {
    command = "chmod 755 ${path.module}/install/install.sh && ${path.module}/install/install.sh '${var.spillbox_container}' '${var.vm_user}' '${var.docker_registry}' '${var.docker_images}' '${var.tenantid}' '${var.clientid}' '${var.clientsecret}' '${var.service_userid}'"
  }
}

resource "null_resource" "fsx_client_local_singularity_sif" {
  count = var.spillbox_container == "singularity" ? 1 : 0
   provisioner "local-exec" {
    command = "sudo cp -r ${var.singularity_sifpath} /spillbox_sif/${basename(var.singularity_sifpath)}"
  }
}
