# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

locals {
  nfs_tag_map = merge(
    { Name = var.Name },
    var.nfs_tag == "" ? {} : tomap({
      for item in [for pair in split(",", var.nfs_tag) : split("=", pair)] : item[0] => item[1]
    })
  )
}

resource "azurerm_virtual_machine" "instance" {
  count                         = var.number
  name                          = format("${var.Name}-%02d", count.index)
  resource_group_name           = var.project
  location                      = var.zone
  vm_size                       = var.instance_type
  delete_os_disk_on_termination = true

  network_interface_ids = [azurerm_network_interface.private[count.index].id]

  os_profile {
    computer_name  = format("${var.Name}%02d", count.index)
    admin_username = var.vm_user

    custom_data = data.template_cloudinit_config.config.rendered
  }

  storage_image_reference {
    id = var.ami_name
  }

  os_profile_linux_config {
    disable_password_authentication = true
    ssh_keys {
      path     = "/home/${var.vm_user}/.ssh/authorized_keys"
      key_data = file("${var.public_key}")
    }
  }
  plan {
    name      = element(split(":", var.ami_plan), 0)
    product      = element(split(":", var.ami_plan), 1)
    publisher     = element(split(":", var.ami_plan), 2)
  }
  tags = local.nfs_tag_map
  storage_os_disk {
    name              = format("${var.Name}-%02d-osdisk", count.index)
    caching           = "ReadWrite"
    create_option     = "FromImage"
    managed_disk_type = "Standard_LRS"
    disk_size_gb      = var.root_volume_size
  }
}
resource "azurerm_public_ip" "public" {
  count               = var.number
  name                = format("${var.Name}-public-ip-private-%02d", count.index)
  location            = var.zone
  resource_group_name = var.project
  allocation_method   = "Static"
}

resource "azurerm_network_interface" "private" {
  count               = var.number
  name                = format("${var.Name}-private-nic-%02d", count.index)
  location            = var.zone
  resource_group_name = var.project

  ip_configuration {
    name                          = format("${var.Name}-private-ip-configuration-%02d", count.index)
    subnet_id                     = var.subnet_id
    private_ip_address_allocation = "Dynamic"
  }
}
resource "null_resource" "fsx_client" {
  depends_on = [azurerm_virtual_machine.instance]
  count      = var.number
  connection {
    user        = var.vm_user
    private_key = file(var.private_key)
    host        = element(flatten(azurerm_network_interface.private.*.private_ip_address), count.index)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = ["sudo chmod 755 /tmp/install/install.sh; /tmp/install/install.sh"]
  }

  provisioner "remote-exec" {
    inline = ["sudo chmod 755 /tmp/install/mount_nvme_azure.sh; /tmp/install/mount_nvme_azure.sh"]
  }
}
