terraform {
  required_version = ">= 1.10.4, < 1.10.5"
  required_providers {
    azurerm = {
      version = "4.17.0"
    }
    null = {
      version = "3.2.3"
    }
    template = {
      version = "2.2.0"
    }
  }
  backend "azurerm" {
    key     = "states/image-compute"
    use_msi = true
  }
}

provider "azurerm" {
  features {
    resource_group {
      prevent_deletion_if_contains_resources = false
    }
  }
}

data "terraform_remote_state" "webserver" {
  backend = "azurerm"
  config = {
    resource_group_name  = var.resource_group == "" ? "ts-${var.customer}-${var.servername}-rg" : var.resource_group
    storage_account_name = "${var.customer}${var.servername}"
    container_name       = "${var.customer}"
    key                  = "states/webserver"
    use_msi              = true
  }
}


locals {
  master_private_ip = var.webserver_create == "true" ? element(flatten(data.terraform_remote_state.webserver.outputs.master_private_ip), 0) : var.webserver[0]
  resourcegroup     = data.terraform_remote_state.webserver.outputs.project
  ami_name          = var.base_ami_name == "" ? null : var.base_ami_name
  is_image_gallery  = can(regex("Microsoft.Compute/galleries/([^/]+)/images/([^/]+)", var.base_ami_name)) ? true : false
  is_image_custom   = can(regex("Microsoft.Compute/images/([^/]+)", var.base_ami_name)) ? true : false
  is_image_market   = can(regex(":", var.base_ami_name)) ? true : false
  plan_given        = var.base_ami_plan == "" ? "" : var.base_ami_plan

  docker_ami_name   = var.docker_baseami_id == "" ? null : var.docker_baseami_id
  docker_is_image_gallery  = can(regex("Microsoft.Compute/galleries/([^/]+)/images/([^/]+)", var.docker_baseami_id)) ? true : false
  docker_is_image_custom   = can(regex("Microsoft.Compute/images/([^/]+)", var.docker_baseami_id)) ? true : false
  docker_is_image_market   = can(regex(":", var.docker_baseami_id)) ? true : false
  docker_plan_given        = var.docker_base_ami_plan == "" ? "" : var.docker_base_ami_plan
}
data "azurerm_shared_image" "shared_image" {
  count               = local.is_image_gallery ? 1 : 0
  name                = element(split("/", var.base_ami_name), 10)
  gallery_name        = element(split("/", var.base_ami_name), 8)
  resource_group_name = element(split("/", var.base_ami_name), 4)
}
/*
data "azurerm_image" "search" {
  count              = local.is_image_custom ? 1 : 0
  name                = element(split("/", var.base_ami_name), 8)
  resource_group_name = element(split("/", var.base_ami_name), 4)
}
*/
resource "azurerm_virtual_machine" "scaleup" {
  name                          = "${var.servername}-imagejob"
  resource_group_name           = data.terraform_remote_state.webserver.outputs.project
  location                      = data.terraform_remote_state.webserver.outputs.region
  vm_size                       = var.compute_instance_type == "" ? "Standard_D2s_v3" : var.compute_instance_type
  delete_os_disk_on_termination = true

  network_interface_ids = [azurerm_network_interface.public.id]

  os_profile {
    computer_name  = "${var.servername}-imagejob"
    admin_username = var.vm_user

    custom_data = data.template_cloudinit_config.config.rendered
  }


  dynamic "storage_image_reference" {
    for_each = local.ami_name != null ? [1] : []

    content {
      id        = !local.is_image_market ? local.ami_name : null
      publisher = local.is_image_market ? element(split(":", local.ami_name), 0) : null
      offer     = local.is_image_market ? element(split(":", local.ami_name), 1) : null
      sku       = local.is_image_market ? element(split(":", local.ami_name), 2) : null
      version   = local.is_image_market ? element(split(":", local.ami_name), 3) : null
    }
  }
  plan {
    name      = local.plan_given == "" ? local.is_image_gallery ? data.azurerm_shared_image.shared_image[0].purchase_plan[0].name : element(split(":", var.base_ami_name), 2) : element(split(":", local.plan_given), 0)
    product   = local.plan_given == "" ? local.is_image_gallery ? data.azurerm_shared_image.shared_image[0].purchase_plan[0].product : element(split(":", var.base_ami_name), 1) : element(split(":", local.plan_given), 1)
    publisher = local.plan_given == "" ? local.is_image_gallery ? data.azurerm_shared_image.shared_image[0].purchase_plan[0].publisher : element(split(":", var.base_ami_name), 0) : element(split(":", local.plan_given), 2)
  }

  os_profile_linux_config {
    disable_password_authentication = true
    ssh_keys {
      path     = "/home/${var.vm_user}/.ssh/authorized_keys"
      key_data = file("${var.project_public_key}")
    }
  }

  storage_os_disk {
    name              = "${var.customer}-${var.servername}-osdisk-imagecompute"
    caching           = "ReadWrite"
    create_option     = "FromImage"
    managed_disk_type = "Standard_LRS"
    disk_size_gb      = var.compute_volume_size
  }
}

resource "azurerm_public_ip" "public" {
  name                = "${var.customer}-${var.servername}-imgcompute-public"
  location            = data.terraform_remote_state.webserver.outputs.region
  resource_group_name = data.terraform_remote_state.webserver.outputs.project
  allocation_method   = "Static"
}

resource "azurerm_network_interface" "public" {
  name                = "${var.customer}-${var.servername}-imgcompute-public"
  location            = data.terraform_remote_state.webserver.outputs.region
  resource_group_name = data.terraform_remote_state.webserver.outputs.project

  ip_configuration {
    name                          = "${var.customer}-${var.servername}-imgcompute-pubconf"
    subnet_id                     = data.terraform_remote_state.webserver.outputs.public_subnet_id
    private_ip_address_allocation = "Dynamic"
  }
}
resource "null_resource" "image_install" {
  connection {
    user        = var.vm_user
    private_key = file(var.project_private_key)
    host        = element(flatten(azurerm_network_interface.public.*.private_ip_address), 0)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }
  provisioner "file" {
    source      = "${path.module}/../../../../config/lsf_std_entitlement.dat"
    destination = "/tmp/lsf_std_entitlement.dat"
  }
  provisioner "remote-exec" {
    inline = ["chmod 775 /tmp/install/update-scaleupimage.sh ; /tmp/install/update-scaleupimage.sh '${var.spillbox_container}' '${var.vm_user}' '${local.master_private_ip}' '${var.docker_registry}' '${var.docker_images}' '${var.service_userid}'"]
  }
}

resource "null_resource" "image_install_singularity_sif" {
  count     = var.spillbox_container == "singularity" ? 1 : 0
  provisioner "file" {
    source      = "${var.singularity_sifpath}"
    destination = "/spillbox_sif/${basename(var.singularity_sifpath)}"
   }
}

#For docker image
data "azurerm_shared_image" "docker_shared_image" {
  count               = var.spillbox_container == "docker" && local.docker_is_image_gallery ? 1 : 0
  name                = element(split("/", var.docker_baseami_id), 10)
  gallery_name        = element(split("/", var.docker_baseami_id), 8)
  resource_group_name = element(split("/", var.docker_baseami_id), 4)
}

resource "azurerm_virtual_machine" "docker_image" {
  count                         = var.spillbox_container == "docker" ? 1 : 0
  name                          = "${var.servername}-dockerimage"
  resource_group_name           = data.terraform_remote_state.webserver.outputs.project
  location                      = data.terraform_remote_state.webserver.outputs.region
  vm_size                       = var.compute_instance_type == "" ? "Standard_D2s_v3" : var.compute_instance_type
  delete_os_disk_on_termination = true

  network_interface_ids = [azurerm_network_interface.docker_image_public[count.index].id]

  os_profile {
    computer_name  = "${var.servername}-dockerimage"
    admin_username = var.docker_vm_user

    custom_data = data.template_cloudinit_config.config.rendered
  }


  dynamic "storage_image_reference" {
    for_each = local.docker_ami_name != null ? [1] : []

    content {
      id        = !local.docker_is_image_market ? local.docker_ami_name : null
      publisher = local.docker_is_image_market ? element(split(":", local.docker_ami_name), 0) : null
      offer     = local.docker_is_image_market ? element(split(":", local.docker_ami_name), 1) : null
      sku       = local.docker_is_image_market ? element(split(":", local.docker_ami_name), 2) : null
      version   = local.docker_is_image_market ? element(split(":", local.docker_ami_name), 3) : null
    }
  }
  plan {
    name      = local.docker_plan_given == "" ? local.docker_is_image_gallery ? data.azurerm_shared_image.docker_shared_image[0].purchase_plan[0].name : element(split(":", var.docker_baseami_id), 2) : element(split(":", local.docker_plan_given), 0)
    product   = local.docker_plan_given == "" ? local.docker_is_image_gallery ? data.azurerm_shared_image.docker_shared_image[0].purchase_plan[0].product : element(split(":", var.docker_baseami_id), 1) : element(split(":", local.docker_plan_given), 1)
    publisher = local.docker_plan_given == "" ? local.docker_is_image_gallery ? data.azurerm_shared_image.docker_shared_image[0].purchase_plan[0].publisher : element(split(":", var.docker_baseami_id), 0) : element(split(":", local.docker_plan_given), 2)
  }

  os_profile_linux_config {
    disable_password_authentication = true
    ssh_keys {
      path     = "/home/${var.docker_vm_user}/.ssh/authorized_keys"
      key_data = file("${var.project_public_key}")
    }
  }

  storage_os_disk {
    name              = "${var.customer}-${var.servername}-osdisk-dockerimage"
    caching           = "ReadWrite"
    create_option     = "FromImage"
    managed_disk_type = "Standard_LRS"
    disk_size_gb      = var.compute_volume_size
  }
}

data "azurerm_managed_disk" "os_disk" {
  depends_on          = [azurerm_virtual_machine.docker_image]
  name                = "${var.customer}-${var.servername}-osdisk-dockerimage"
  resource_group_name = data.terraform_remote_state.webserver.outputs.project
}

resource "azurerm_managed_disk" "docker_data_disk" {
  depends_on           = [azurerm_virtual_machine.docker_image]
  name                 = "${var.customer}-${var.servername}-spillbox-datadisk"
  resource_group_name  = data.terraform_remote_state.webserver.outputs.project
  location             = data.terraform_remote_state.webserver.outputs.region
  storage_account_type = "Standard_LRS"
  create_option        = "Copy"
  source_resource_id   = data.azurerm_managed_disk.os_disk.id
  disk_size_gb         = var.compute_volume_size
}

resource "azurerm_virtual_machine_data_disk_attachment" "docker_data_attach" {
  depends_on         = [azurerm_managed_disk.docker_data_disk]
  managed_disk_id    = azurerm_managed_disk.docker_data_disk.id
  virtual_machine_id = azurerm_virtual_machine.docker_image[0].id
  lun               = 1  # Logical Unit Number (should be unique per VM)
  caching           = "ReadWrite"
}

resource "azurerm_public_ip" "docker_image_public" {
  count               = var.spillbox_container == "docker" ? 1 : 0
  name                = "${var.customer}-${var.servername}-dockerimage-public"
  location            = data.terraform_remote_state.webserver.outputs.region
  resource_group_name = data.terraform_remote_state.webserver.outputs.project
  allocation_method   = "Static"
}

resource "azurerm_network_interface" "docker_image_public" {
  count               = var.spillbox_container == "docker" ? 1 : 0
  name                = "${var.customer}-${var.servername}-dockerimage-public"
  location            = data.terraform_remote_state.webserver.outputs.region
  resource_group_name = data.terraform_remote_state.webserver.outputs.project

  ip_configuration {
    name                          = "${var.customer}-${var.servername}-dockerimage-pubconf"
    subnet_id                     = data.terraform_remote_state.webserver.outputs.public_subnet_id
    private_ip_address_allocation = "Dynamic"
  }
}

resource "null_resource" "build_docker_image" {
  count         = var.spillbox_container == "docker" ? 1 : 0
  depends_on    = [azurerm_virtual_machine_data_disk_attachment.docker_data_attach, azurerm_virtual_machine.docker_image]
  connection {
    user        = var.docker_vm_user
    private_key = file(var.project_private_key)
    host        = element(flatten(azurerm_network_interface.docker_image_public[0].*.private_ip_address), 0)
    port        = var.cloud_ssh_port
    timeout     = "20m"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }
  provisioner "remote-exec" {
    inline = ["chmod 775 /tmp/install/docker_build.sh ; /tmp/install/docker_build.sh '${element(azurerm_network_interface.docker_image_public[*].private_ip_address, 0)}' '${var.service_userid}' '${local.master_private_ip}'"]
  }
}
resource "null_resource" "setup_docker_imagenode" {
  count     = var.spillbox_container == "docker" ? 1 : 0
  depends_on    = [null_resource.image_install, null_resource.build_docker_image]
  connection {
     user        = var.docker_vm_user
     private_key = file(var.project_private_key)
     host        = element(flatten(azurerm_network_interface.public.*.private_ip_address), 0)
     port        = var.cloud_ssh_port
     timeout     = "600s"
   }
   provisioner "file" {
     source      = "${path.module}/install"
     destination = "/tmp"
   }
   provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_setup.sh ; /tmp/install/docker_setup.sh '${local.master_private_ip}' '${var.service_userid}'"]
   }
 }
resource "null_resource" "setup_docker_image_webserver" {
   count      = (var.spillbox_container == "docker" && var.webserver_create) ? 1 : 0
   depends_on = [null_resource.build_docker_image]
   connection {
    user        = var.vm_user
    private_key = file(var.project_private_key)
    host        = local.master_private_ip
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }
  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_setup.sh ; /tmp/install/docker_setup.sh '${local.master_private_ip}' '${var.service_userid}'"]
  }
}

resource "null_resource" "setup_docker_image_webserver_local" {
  count = (var.webserver_create == false && var.spillbox_container == "docker") ? 1 : 0
  depends_on = [null_resource.build_docker_image]
  provisioner "local-exec" {
     command = "chmod 755 ${path.module}/install/docker_setup.sh ; ${path.module}/install/docker_setup.sh '${local.master_private_ip}' '${var.service_userid}'"
  }
}
