terraform {
  required_version = ">= 1.10.4, < 1.10.5"
  required_providers {
    azurerm = {
      version = "4.17.0"
    }
    local = {
      version = "2.5.2"
    }
    null = {
      version = "3.2.3"
    }
    template = {
      version = "2.2.0"
    }
    tls= {
      version = "4.0.6"
    }
  }
  backend "azurerm" {
    key     = "states/webserver"
    use_msi = true
  }
}


provider "azurerm" {
  features {
    resource_group {
      prevent_deletion_if_contains_resources = false
    }
  }
}

module "subnet" {
  source              = "../network/"
  count               = var.enable_sgo == "false" ? 0 : var.subnet_id == "" ? 1 : 0
  customer            = "${var.customer}-${var.customer_project}"
  resource_group_name = var.resource_group == "" ? "ts-${var.customer}-${var.customer_project}-rg" : var.resource_group
  location            = var.region
  vpc_id              = var.vpc_id
  public_subnet_cidr  = var.public_subnet_cidr
}

module "security_group" {
  source              = "../security"
  count               = var.enable_sgo == "false" ? 0 : var.security_group_id == "" ? 1 : 0
  customer            = "${var.customer}-${var.customer_project}"
  location            = var.region
  public_cidr_blocks  = var.public_subnet_cidr == "" ? module.subnet[0].public_cidr_blocks : [[var.public_subnet_cidr]]
  resource_group_name = var.resource_group == "" ? "ts-${var.customer}-${var.customer_project}-rg" : var.resource_group
  cloud_ssh_port      = var.cloud_ssh_port
}

resource "azurerm_subnet_network_security_group_association" "public" {
  depends_on                = [module.subnet[0]]
  count                     = var.subnet_id == "" ? 1 : 0
  subnet_id                 = var.subnet_id == "" ? module.subnet[0].public_id : var.subnet_id
  network_security_group_id = var.security_group_id == "" ? module.security_group[0].public_id : var.security_group_id
}

resource "azurerm_private_dns_zone" "filedns" {
  name                = "${var.storage_account}.file.core.windows.net"
  resource_group_name = var.resource_group == "" ? "ts-${var.customer}-${var.customer_project}-rg" : var.resource_group
}

resource "azurerm_private_dns_zone_virtual_network_link" "example" {
  depends_on            = [module.subnet[0]]
  name                  = "${var.customer}-${var.servername}-link"
  resource_group_name   = var.resource_group == "" ? "ts-${var.customer}-${var.customer_project}-rg" : var.resource_group
  private_dns_zone_name = azurerm_private_dns_zone.filedns.name
  virtual_network_id    = var.vpc_id == "" ? module.subnet[0].vpc_id : var.vpc_id
}

resource "azurerm_private_endpoint" "file" {
  depends_on          = [module.subnet[0]]
  name                = "${var.customer}-${var.servername}-nfsfile"
  location            = var.region
  resource_group_name = var.resource_group == "" ? "ts-${var.customer}-${var.customer_project}-rg" : var.resource_group
  subnet_id           = var.subnet_id == "" ? module.subnet[0].public_id : var.subnet_id

  private_service_connection {
    name                           = "${var.customer}-${var.servername}-private"
    private_connection_resource_id = azurerm_storage_account.nfs.id
    subresource_names              = ["file"]
    is_manual_connection           = false
  }

  private_dns_zone_group {
    name                 = "${var.customer}-${var.servername}-dns-zone-group"
    private_dns_zone_ids = [azurerm_private_dns_zone.filedns.id]
  }
}

resource "azurerm_storage_account" "nfs" {
  name                      = "${var.storage_account}nfs"
  resource_group_name       = var.resource_group == "" ? "ts-${var.customer}-${var.customer_project}-rg" : var.resource_group
  location                  = var.region
  account_tier              = "Premium"
  account_replication_type  = "LRS"
  account_kind              = "FileStorage"
  https_traffic_only_enabled = false
}

resource "azurerm_storage_share" "example" {
  name                 = "sharename"
  storage_account_name = azurerm_storage_account.nfs.name
  quota                = 100
  enabled_protocol     = "NFS"
}

module "ssh_key_pair" {
  source           = "../ssh_key_pair"
  region           = var.region
  secrets          = var.secrets
  customer         = var.customer
  customer_project = var.customer_project
  generate_ssh_key = false
}

module "webserver" {
  source                 = "../../azure/compute_fw/"
  count                  = var.enable_sgo == "false" ? 0 : 1
  webserver_create       = var.webserver_create
  number                 = 1
  Name                   = "${var.servername}-hub"
  subnet_id              = var.subnet_id == "" ? module.subnet[0].public_id : var.subnet_id
  customer               = "${var.customer}-${var.servername}"
  key_name               = module.ssh_key_pair.key_name
  vpc_security_group_ids = var.security_group_id == "" ? [module.security_group[0].public_id] : [var.security_group_id]
  ami_name               = var.base_ami_name == "" ? "" : var.base_ami_name
  ami_plan               = var.base_ami_plan == "" ? "" : var.base_ami_plan
  instance_type          = var.webserver_instance_type == "" ? var.splpipe_instance_type[var.model] : var.webserver_instance_type
  vm_user                = var.vm_user
  public_key             = module.ssh_key_pair.public_key_filename
  private_key            = module.ssh_key_pair.private_key_filename
  zone                   = var.region
  project                = var.resource_group == "" ? "ts-${var.customer}-${var.customer_project}-rg" : var.resource_group
  dnsname                = azurerm_private_endpoint.file.private_service_connection[0].private_ip_address
  mountname              = "/${var.storage_account}nfs/sharename"
  cloud_ssh_port         = var.cloud_ssh_port
  root_volume_size       = var.webserver_volume_size
  tenantid               = var.tenantid
  clientid               = var.clientid
  clientsecret           = var.clientsecret
  docker_registry        = var.docker_registry
  docker_images          = var.docker_images
  service_userid         = var.service_userid
  spillbox_container     = var.spillbox_container
  singularity_sifpath    = var.singularity_sifpath
}

