# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

locals {
  compute_tag_map = merge(
    { name = var.Name },
    var.compute_tag == "" ? {} : tomap({
      for item in [for pair in split(",", var.compute_tag) : split("=", pair)] : item[0] => item[1]
    })
  )
}

resource "google_compute_instance" "compute" {
  project      = var.project
  zone         = var.zone
  count        = var.number
  machine_type = var.instance_type

  boot_disk {
    kms_key_self_link = var.kms_key_id == "" ? null : var.kms_key_id
    initialize_params {
      image = var.ami_name
      size  = var.root_volume_size
    }
  }
  metadata = {
    serial-port-enable      = "true"
    block-project-ssh-keys  = "true"
    enable-guest-attributes = "TRUE"
    ssh-keys                = <<EOF
${var.vm_user}:${replace(file(var.public_key), "\n", "")} ${var.vm_user}
EOF

  }
  labels = local.compute_tag_map
  network_interface {
    subnetwork         = var.subnet_id
    subnetwork_project = var.project
    dynamic "access_config" {
      for_each = var.disable_compute_public_ips == true ? [] : [1]
      content {}
    }
  }
  name                    = format("${var.Name}%02d", count.index)
  metadata_startup_script = data.template_cloudinit_config.config.rendered
}

resource "null_resource" "host_setup" {
  count = var.number
  connection {
    user        = var.vm_user
    private_key = file(var.private_key)
    host        = element(flatten(google_compute_instance.compute.*.network_interface.0.network_ip), count.index)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = ["sudo chmod 755 /tmp/install/install.sh; sudo /tmp/install/install.sh"]
  }
}
