# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

resource "google_compute_instance" "compute" {
  project      = var.project
  zone         = var.zone
  count        = var.webserver_create == "true" ? 1 : 0
  machine_type = var.instance_type

  boot_disk {
    kms_key_self_link = var.kms_key_id == "" ? null : var.kms_key_id
    initialize_params {
      image = var.ami_name
      size  = var.root_volume_size
    }
  }
  metadata = {
    serial-port-enable      = "true"
    block-project-ssh-keys  = "true"
    enable-guest-attributes = "TRUE"
    ssh-keys                = <<EOF
${var.vm_user}:${replace(file(var.public_key), "\n", "")} ${var.vm_user}
EOF

  }

  network_interface {
    subnetwork         = var.subnet_id
    subnetwork_project = var.project
    dynamic "access_config" {
      for_each = var.disable_compute_public_ips == true ? [] : [1]
      content {}
    }
  }
  name                    = var.Name
  metadata_startup_script = data.template_cloudinit_config.config.rendered
}
resource "null_resource" "host_setup" {
  count = var.webserver_create == "true" ? 1 : 0
  connection {
    user        = var.vm_user
    private_key = file(var.private_key)
    host        = element(flatten(google_compute_instance.compute.*.network_interface.0.network_ip), count.index)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = ["chmod 755 /tmp/install/install.sh; /tmp/install/install.sh '${var.spillbox_container}' '${var.vm_user}' '${var.docker_registry}' '${var.docker_images}' '${var.service_userid}'"]
  }
}

resource "null_resource" "host_setup_singularity_sif" {
  count = var.spillbox_container == "singularity" ? 1 : 0
  provisioner "file" {
    source      = "${var.singularity_sifpath}"
    destination = "/spillbox_sif/${basename(var.singularity_sifpath)}"
  }
}

resource "null_resource" "host_setup_local" {
  count = var.webserver_create == "false" ? 1 : 0

  provisioner "local-exec" {
    command = "chmod 755 ${path.module}/install/install.sh && ${path.module}/install/install.sh '${var.spillbox_container}' '${var.vm_user}' '${var.docker_registry}' '${var.docker_images}' '${var.service_userid}'"
  }
}

resource "null_resource" "host_setup_local_singularity_sif" {
  count = var.spillbox_container == "singularity" ? 1 : 0
   provisioner "local-exec" {
    command = "sudo cp -r ${var.singularity_sifpath} /spillbox_sif/${basename(var.singularity_sifpath)}"
  }
}
