# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

locals {
  nfs_tag_map = merge(
    { name = var.Name },
    can(var.nfs_tag) && var.nfs_tag != "" ? tomap({
      for item in [for pair in split(",", var.nfs_tag) : split("=", pair)] : item[0] => item[1]
    }) : {}
  )
}

#  CREATE NFS INSTANCE DIRECTLY
resource "google_compute_instance" "nfs_instance" {
  count        = var.number
  name         = format("${var.Name}-nfs%02d", count.index)
  project      = var.project
  zone         = var.zone
  machine_type = var.instance_type

  # Networking
  network_interface {
    subnetwork         = var.subnet_id
    subnetwork_project = var.project
    dynamic "access_config" {
      for_each = var.disable_compute_public_ips == true ? [] : [1]
      content {}
    }
  }

  # Boot Disk
  boot_disk {
    initialize_params {
      image = var.spillbox_imageid
      size  = var.compute_volume_size
    }
    auto_delete = true
  }

  # NVMe Local SSD
  scratch_disk {
    interface = "NVME"
  }

  metadata = {
    serial-port-enable      = "true"
    block-project-ssh-keys  = "true"
    enable-guest-attributes = "TRUE"
    ssh-keys                = "${var.vm_user}:${replace(file(var.project_public_key), "\n", "")} ${var.vm_user}"
  }

  metadata_startup_script = data.template_cloudinit_config.config.rendered
}

# SSH-BASED PROVISIONING (INSTALL + NVMe MOUNT)
resource "null_resource" "host_setup" {
  count = var.number
  connection {
    user        = var.vm_user
    private_key = file(var.private_key)
    host        = google_compute_instance.nfs_instance[count.index].network_interface[0].network_ip
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }

  # Copy installation and NVMe mount scripts
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  # Run install.sh and nvme_mount.sh
  provisioner "remote-exec" {
    inline = [
      "for i in {1..5}; do sudo chmod 755 /tmp/install/install.sh && sudo /tmp/install/install.sh && break || sleep 10; done",
      "for i in {1..5}; do sudo chmod 755 /tmp/install/mount_nvme_google.sh && sudo /tmp/install/mount_nvme_google.sh && break || sleep 10; done"
    ]
  }
}
