#!/usr/bin/env bash
# https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/device_naming.html#available-ec2-device-names

set -o nounset
set -o errexit
set -o pipefail

sleep 60
VOLUME_NAME=$(find /dev | grep -i  -Po '/dev/(sd[b-z]|xvd[b-z]|md)')
for volume_device in $${VOLUME_NAME}; do
        if ! grep -qs "$volume_device" /proc/mounts; then
                VOLUME_NAME=$volume_device
                break
        fi
done

NAME=$(echo $${VOLUME_NAME}) 
echo "---> Waiting for $NAME (. = 5s)" >> /var/log/cloud-init.log
timeout 180 /bin/bash -c "until stat $NAME 2>/dev/null; do echo waiting for $NAME ...; sleep 5; done"

# == Vars
#
MOUNT_POINT="${mount_point}"
MOUNT_POINT_BACKUP="${mount_point}-backup"
INMEMORY="${inmemory}"
FS="${file_system}"
LIMIT="${limit}"
MEMTOTAL=$(expr $(vmstat -s | grep "total memory" | awk '{print $1}'))
MEMTOTAL=$(($MEMTOTAL / 1048576 ))
TMPFS_SIZE="${tmpfs_size}"
DISK_SIZE=$(/sbin/fdisk -l $NAME | grep ^Disk | awk '{print $3}')
DISK_UNIT=$(/sbin/fdisk -l $NAME | grep ^Disk | awk '{print $4}' | sed 's/,//g') 
if [ "$DISK_UNIT" == "TiB" ] ; then
    DISK_SIZE=$(echo $DISK_SIZE | sed 's/,//g' | awk -F. '{print $1}')
    let DISK_SIZE=$DISK_SIZE*1024
fi

if [ "$MEMTOTAL" -lt "$LIMIT" ] ; then
    if [ "$INMEMORY" = true ] ; then
        INMEMORY=false
    fi
fi

# == Main
#
echo "===> UserData Script: Start" >> /var/log/cloud-init.log

if $INMEMORY ; then
    echo '---> Creating Mount point'
    mkdir -p $${MOUNT_POINT}

    echo '---> Creating swap volume'
    mkswap $${VOLUME_NAME}

    echo '---> Swapon EBS'
    swapon $${VOLUME_NAME}

    TMPFS_SIZE=$(expr $MEMTOTAL - $LIMIT)
    echo '---> Mount tmpfs'
    let TMPFS_SIZE=($TMPFS_SIZE+$DISK_SIZE)
    mount -t tmpfs -o size=$${TMPFS_SIZE}G tmpfs $${MOUNT_POINT}

else

    echo '---> Creating File System'
    mkfs.$${FS} -F  $${VOLUME_NAME}


    #echo '---> Move Data'
    #mkdir -p $${MOUNT_POINT}
    #mv $${MOUNT_POINT} $${MOUNT_POINT_BACKUP}

    echo '---> Creating Mount point'
    mkdir -p $${MOUNT_POINT}

    echo '---> Retrieve UUID EBS'
    VOLUME_UUID=$(blkid -s UUID -o value $${VOLUME_NAME})

    echo '---> Mount EBS'
    mount UUID="$${VOLUME_UUID}" $${MOUNT_POINT}

    echo '---> RollMove Data'
    shopt -s dotglob
    #mv $${MOUNT_POINT_BACKUP}/* $${MOUNT_POINT}/
    #chown --reference=$${MOUNT_POINT_BACKUP} $${MOUNT_POINT}
    #chmod --reference=$${MOUNT_POINT_BACKUP} $${MOUNT_POINT}
    #rm -r $${MOUNT_POINT_BACKUP}

    echo '---> Saving Mount point'
    echo "UUID="$${VOLUME_UUID}" $${MOUNT_POINT} $${FS} discard,defaults,nofail 0 2" | sudo tee -a /etc/fstab
fi

echo "===> UserData Script: Done" >> /var/log/cloud-init.log
