# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

terraform {
  required_version = ">= 1.10.4, < 1.10.5"
  required_providers {
    google = {
        version = "6.19.0"
    }
    null = {
        version = "3.2.3"
    }
    template = {
        version = "2.2.0"
    }
  }

  backend "gcs" {
    prefix = "states/image-compute"
  }
}

provider "google" {
  region      = var.region
  project     = var.project
  credentials = var.google_credentials
}
data "terraform_remote_state" "webserver" {
  backend = "gcs"

  config = {
    bucket = "ts-${var.customer}-${var.servername}"
    prefix = "states/webserver"
  }
}

data "template_file" "config" {
  template = file("${path.module}/init.tpl")
  vars = {
    cloud_ssh_port = var.cloud_ssh_port
  }
}

data "template_cloudinit_config" "config" {
  gzip          = true
  base64_encode = true

  part {
    content_type = "text/x-shellscript"
    content      = data.template_file.config.rendered
  }
}
locals {
  master_private_ip = var.webserver_create == "true" ? element(flatten(data.terraform_remote_state.webserver.outputs.master_private_ip), 0) : var.webserver[0]
}

resource "google_compute_instance" "compute" {
  project      = var.project
  zone         = var.zone
  machine_type = var.compute_instance_type == "" ? var.job_instance_type[var.model] : var.compute_instance_type

  boot_disk {
    kms_key_self_link = var.kms_key_id == "" ? null : var.kms_key_id
    initialize_params {
      image = var.base_ami_name == "" ? "centos-7-v20240213" : var.base_ami_name
      size  = var.compute_volume_size
    }
  }
  metadata = {
    serial-port-enable      = "true"
    block-project-ssh-keys  = "true"
    enable-guest-attributes = "TRUE"
    ssh-keys                = <<EOF
${var.vm_user}:${replace(file(var.project_public_key), "\n", "")} ${var.vm_user}
EOF

  }

  network_interface {
    subnetwork         = data.terraform_remote_state.webserver.outputs.public_subnet_id
    subnetwork_project = var.project
   dynamic "access_config" {
      for_each = var.disable_compute_public_ips == true ? [] : [1]
      content {}
    }
  }
  name                    = "${var.servername}-imagejob"
  metadata_startup_script = data.template_cloudinit_config.config.rendered
}
resource "null_resource" "hostsetup_lsfinstall" {
  connection {
    user        = var.vm_user
    private_key = file(var.project_private_key)
    host        = element(flatten(google_compute_instance.compute.*.network_interface.0.network_ip), 0)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }
  provisioner "file" {
    source      = "${path.module}/../../../../config/lsf_std_entitlement.dat"
    destination = "/tmp/lsf_std_entitlement.dat"
  }
  provisioner "remote-exec" {
    inline = ["chmod 755 /tmp/install/update-scaleupimage.sh; /tmp/install/update-scaleupimage.sh '${var.spillbox_container}' '${var.vm_user}' '${local.master_private_ip}' '${var.docker_registry}' '${var.docker_images}' '${var.service_userid}'"]
  }
}

resource "null_resource" "hostsetup_lsfinstall_singularity_sif" {
  count = var.spillbox_container == "singularity" ? 1 : 0
  provisioner "file" {
    source      = "${var.singularity_sifpath}"
    destination = "/spillbox_sif/${basename(var.singularity_sifpath)}"
   }
}

# Creating a docker image machine
resource "google_compute_instance" "docker_image" {
  count        = var.spillbox_container == "docker" ? 1 : 0
  project      = var.project
  zone         = var.zone
  machine_type = var.compute_instance_type == "" ? var.job_instance_type[var.model] : var.compute_instance_type

  boot_disk {
    kms_key_self_link = var.kms_key_id == "" ? null : var.kms_key_id
    initialize_params {
      image = var.docker_baseami_id == "" ? "centos-7-v20240213" : var.docker_baseami_id
      size  = var.compute_volume_size
    }
  }
  metadata = {
    serial-port-enable      = "true"
    block-project-ssh-keys  = "true"
    enable-guest-attributes = "TRUE"
    ssh-keys                = <<EOF
${var.vm_user}:${replace(file(var.project_public_key), "\n", "")} ${var.vm_user}
EOF

  }

  network_interface {
    subnetwork         = data.terraform_remote_state.webserver.outputs.public_subnet_id
    subnetwork_project = var.project
    dynamic "access_config" {
      for_each = var.disable_compute_public_ips == true ? [] : [1]
      content {}
    }
  }
  name                    = "${var.servername}-docker"
  metadata_startup_script = data.template_cloudinit_config.config.rendered
}

resource "google_compute_disk" "spillbox_disk" {
  count = var.spillbox_container == "docker" ? 1 : 0

  name  = "spillbox-disk"
  type  = "pd-standard"  # Use "pd-standard", "pd-balanced", or "pd-extreme" as needed
  zone  = var.zone
  size  = var.compute_volume_size
  image = var.docker_baseami_id
  disk_encryption_key {
    kms_key_self_link = var.kms_key_id == "" ? null : var.kms_key_id
  }
}
resource "google_compute_attached_disk" "spillbox_disk_attachment" {
  count    = var.spillbox_container == "docker" ? 1 : 0
  disk     = google_compute_disk.spillbox_disk[0].id
  instance = google_compute_instance.docker_image[0].id
  zone     = var.zone

  lifecycle {
    prevent_destroy = false  # Allows Terraform to detach the disk when needed
  }
}

resource "null_resource" "build_docker_image" {
  count         = var.spillbox_container == "docker" ? 1 : 0
  depends_on    = [google_compute_attached_disk.spillbox_disk_attachment]
  connection {
    user        = var.vm_user
    private_key = file(var.project_private_key)
    host        = element(flatten(google_compute_instance.docker_image[0].*.network_interface.0.network_ip), 0)
    port        = var.cloud_ssh_port
    timeout     = "20m"
  }

  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_build.sh ; /tmp/install/docker_build.sh '${google_compute_instance.docker_image[0].network_interface[0].network_ip}' '${var.service_userid}' '${local.master_private_ip}' "]
  }
}

resource "null_resource" "setup_docker_imagenode" {
  count         = var.spillbox_container == "docker" ? 1 : 0
  depends_on    = [null_resource.hostsetup_lsfinstall, null_resource.build_docker_image]
  connection {
    user        = var.vm_user
    private_key = file(var.project_private_key)
    host        = element(flatten(google_compute_instance.compute.*.network_interface.0.network_ip), 0)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_setup.sh ; /tmp/install/docker_setup.sh '${local.master_private_ip}' '${var.service_userid}'"]
  }
}

resource "null_resource" "setup_docker_image_webserver" {
  count      = var.spillbox_container == "docker" && var.webserver_create == "true" ? 1 : 0
  depends_on = [null_resource.build_docker_image]
  connection {
    user        = var.vm_user
    private_key = file(var.project_private_key)
    host        = local.master_private_ip
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_setup.sh ; /tmp/install/docker_setup.sh '${local.master_private_ip}' '${var.service_userid}'"]
  }
}

resource "null_resource" "setup_docker_image_webserver_local" {
  count = var.spillbox_container == "docker" && var.webserver_create == "false" ? 1 : 0
  depends_on = [null_resource.build_docker_image]
  provisioner "local-exec" {
     command = "chmod 755 ${path.module}/install/docker_setup.sh ; ${path.module}/install/docker_setup.sh '${local.master_private_ip}' '${var.service_userid}'"
  }
}
