data "template_file" "config" {
  template = file("${path.module}/init.tpl")
  vars = {
    cloud_ssh_port = var.cloud_ssh_port
  }
}

data "template_cloudinit_config" "config" {
  gzip          = true
  base64_encode = true

  part {
    content_type = "text/x-shellscript"
    content      = data.template_file.config.rendered
  }
}

resource "oci_core_instance" "compute" {
  count                  = var.webserver_create == "true" ? 1 : 0
  availability_domain    = var.zone
  compartment_id         = var.compartment_ocid
  shape                  = "VM.Standard.E2.2"

  source_details {
    source_type = "image"
    source_id   = var.ami_name 
    boot_volume_size_in_gbs = var.root_volume_size
    kms_key_id = var.kms_key_id == "" ? null : var.kms_key_id
  }

  create_vnic_details { 
    subnet_id = var.subnet_id
  }
  metadata = {
    ssh_authorized_keys = file(var.public_key)
    user_data = data.template_cloudinit_config.config.rendered
  } 
  display_name = var.Name
  freeform_tags = {
    Name = var.Name
  }
  preserve_boot_volume = false

  provisioner "local-exec" {
    command = "sleep ${var.wait_time_vm} && echo done waiting master VM ready"
  }
}

resource "null_resource" "fsx_client" {
  count = var.webserver_create == "true" ? 1 : 0

  connection {
    user        = var.vm_user
    private_key = file(var.private_key)
    host        = oci_core_instance.compute[count.index].public_ip
    port        = var.cloud_ssh_port
    timeout     = "300s"
  }

  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = ["chmod 755 /tmp/install/install.sh; /tmp/install/install.sh '${var.spillbox_container}' '${var.vm_user}' '${var.docker_registry}' '${var.docker_images}' '${var.service_userid}' 'true'"]
  }
}

resource "null_resource" "fsx_client_singularity_sif" {
  count = var.spillbox_container == "singularity" ? 1 : 0
  provisioner "file" {
    source      = "${var.singularity_sifpath}"
    destination = "/spillbox_sif/${basename(var.singularity_sifpath)}"
   }
}

resource "null_resource" "fsx_client_local" {
  count = var.webserver_create == "false" ? 1 : 0

  provisioner "local-exec" {
    command = "chmod 755 ${path.module}/install/install.sh && ${path.module}/install/install.sh '${var.spillbox_container}' '${var.vm_user}' '${var.docker_registry}' '${var.docker_images}' '${var.service_userid}' 'false'"
  }
}

resource "null_resource" "fsx_client_local_singularity_sif" {
  count = var.spillbox_container == "singularity" ? 1 : 0
  provisioner "local-exec" {
    command = "sudo cp -r ${var.singularity_sifpath} /spillbox_sif/${basename(var.singularity_sifpath)}"
   }
}
