terraform {
  required_version = ">= 1.10.4, < 1.10.5"

  backend "s3" {
    skip_region_validation      = true
    skip_credentials_validation = true
    skip_metadata_api_check     = true
    force_path_style            = true
    key     = "states/image-compute"
  }
}

provider "oci" {
  region          = var.region
  tenancy_ocid    = var.tenancy_ocid
  user_ocid       = var.user_ocid
  fingerprint     = var.fingerprint
  private_key_path = var.private_key_path
}

locals {
  master_private_ip = var.webserver_create == "true" ? element(flatten(data.terraform_remote_state.webserver.outputs.master_private_ip), 0) : var.webserver[0]
}

data "terraform_remote_state" "webserver" {
  backend = "s3"

  config = {
    bucket = "ts-${var.customer}-${var.servername}"
    region = var.region
    skip_region_validation      = true
    skip_credentials_validation = true
    skip_metadata_api_check     = true
    force_path_style            = true
    key = "states/webserver"
  }
}

data "template_file" "config" {
  template = file("${path.module}/init.tpl")
  vars = {
    cloud_ssh_port = var.cloud_ssh_port
  }
}

data "template_cloudinit_config" "config" {
  gzip          = true
  base64_encode = true

  part {
    content_type = "text/x-shellscript"
    content      = data.template_file.config.rendered
  }
}

resource "oci_core_instance" "updatescaleup" {
  shape                  = var.compute_instance_type == "" ? "VM.Standard.E2.2" : var.compute_instance_type
  availability_domain = var.zone
  compartment_id      = var.compartment_ocid
  display_name        = "${var.servername}-image"
  create_vnic_details {
    subnet_id =  data.terraform_remote_state.webserver.outputs.public_subnet_id
  }
  metadata = {
    ssh_authorized_keys = file(data.terraform_remote_state.webserver.outputs.public_key)
    user_data = data.template_cloudinit_config.config.rendered
  }
  freeform_tags = {
    Name = "${var.servername}-image"
  }
  source_details {
    source_type = "image"
    source_id   = var.base_ami_name
    boot_volume_size_in_gbs = var.compute_volume_size
    kms_key_id = var.kms_key_id == "" ? null : var.kms_key_id
  }

  provisioner "local-exec" {
    command = "sleep 120 && echo done waiting master VM ready"
  }
}

resource "null_resource" "update-image" {
  connection {
    user        = var.vm_user
    private_key = file(data.terraform_remote_state.webserver.outputs.private_key)
    host        = oci_core_instance.updatescaleup.public_ip
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }
  
  provisioner "file" {
    source      = "${path.module}/../../../../config/lsf_std_entitlement.dat"
    destination = "/tmp/lsf_std_entitlement.dat"
  }

  provisioner "remote-exec" {
    inline = ["chmod 775 /tmp/install/update-scaleupimage.sh ; /tmp/install/update-scaleupimage.sh '${var.spillbox_container}' '${var.vm_user}' '${local.master_private_ip}' '${var.docker_registry}' '${var.docker_images}' '${var.service_userid}'"]
  }
}

resource "null_resource" "update-image_singularity_sif" {
  count = var.spillbox_container == "singularity" ? 1 : 0
  provisioner "file" {
    source      = "${var.singularity_sifpath}"
    destination = "/spillbox_sif/${basename(var.singularity_sifpath)}"
  }
}

