terraform {
  required_version = ">= 1.10.4, < 1.10.5"

  backend "s3" {
    bucket         = var.bucket
    skip_region_validation      = true
    skip_credentials_validation = true
    skip_metadata_api_check     = true
    force_path_style            = true
    key         = "states/image-nfs"
  }
}

provider "oci" {
  region          = var.region
  tenancy_ocid    = var.tenancy_ocid
  user_ocid       = var.user_ocid
  fingerprint     = var.fingerprint
  private_key_path = var.private_key_path
}

locals {
  master_private_ip = var.webserver_create == "true" ? element(data.terraform_remote_state.webserver.outputs.master_private_ip, 0) : var.webserver[0]
}

data "terraform_remote_state" "webserver" {
  backend = "s3"

  config = {
    bucket = "ts-${var.customer}-${var.servername}"
    region = var.region
    skip_region_validation      = true
    skip_credentials_validation = true
    skip_metadata_api_check     = true
    force_path_style            = true
    key = "states/webserver"
    endpoint="https://axsnzbjo30ri.compat.objectstorage.us-sanjose-1.oraclecloud.com"
  }
}

resource "oci_core_instance" "image_nfs" {
  compartment_id          = var.compartment_ocid
  shape                   = var.nfs_instance_type == "" ? "VM.Standard.E2.2" : var.nfs_instance_type
  availability_domain     = var.zone
  display_name        = "${var.servername}-imagenfs"
  create_vnic_details {
    subnet_id = data.terraform_remote_state.webserver.outputs.public_subnet_id
  }
  metadata = {
    ssh_authorized_keys = file(data.terraform_remote_state.webserver.outputs.public_key)
    user_data = data.template_cloudinit_config.config.rendered
  }
  source_details {
    source_type = "image"
    source_id   = var.spillbox_imageid
    boot_volume_size_in_gbs = var.nfs_volume_size
    kms_key_id = var.kms_key_id == "" ? null : var.kms_key_id
  }
  preserve_boot_volume = false
  freeform_tags = {
    Name = "${var.servername}-imagenfs"
  }
}

resource "oci_core_volume" "nvme_volume" {
  compartment_id = var.compartment_ocid
  size_in_gbs    = var.disk_size[var.model]
  availability_domain = var.zone
  freeform_tags = {
    Name = "${var.servername}-imagenfs"
  }
}

resource "oci_core_volume_attachment" "nvme_mount" {
  attachment_type = "paravirtualized"
  compartment_id   = var.compartment_ocid
  instance_id      = oci_core_instance.image_nfs.id
  volume_id        = oci_core_volume.nvme_volume.id
#  device          = "/dev/xvdf"
}

resource "null_resource" "update-image" {
  connection {
    user        = var.vm_user
    private_key = file(data.terraform_remote_state.webserver.outputs.private_key)
    host        = oci_core_instance.image_nfs.public_ip
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install/update-scaleupimage.sh"
    destination = "/tmp/install/update-scaleupimage.sh"
  }

  provisioner "remote-exec" {
    inline = ["chmod 775 /tmp/install/update-scaleupimage.sh ; /tmp/install/update-scaleupimage.sh '${var.vm_user}' '${local.master_private_ip[0]}' '${var.docker_registry}' '${var.docker_images}'"]
  }
}

