#!/bin/bash -f

# Copyright � 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

# Wrapper Script for Executing Terraform with Flexible Input and SSH Directory Cleanup

saved_args=$@

usage() {
    echo ""
    echo "Usage: $0 <provider> <action> <customer_id|username|all> [project_name]"
    echo ""
    echo "provider     : aws | google | azure | local"
    echo "action       : destroycluster | destroyinfra | destroyall"
    echo "customer_id  : Required for cloud projects (aws/google/azure)"
    echo "username|all : For local projects, use a username to destroy all that user's projects, or 'all' to destroy every project"
    echo "project_name : (Optional) specific project name for cloud projects"
    echo ""
    exit 3
}


# Validate number of arguments
if [ "$#" -lt 3 ] || [ "$#" -gt 4 ]; then
    usage
fi

# Check AWS credentials for 'aws' provider
if [ "$1" = "aws" ]; then
    if [ -z "$AWS_ACCESS_KEY_ID" ] || [ -z "$AWS_SECRET_ACCESS_KEY" ]; then
        echo "Error: AWS credentials (AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY) are not set."
        echo "Please set these environment variables or attach an IAM role to the EC2 instance."
        echo "See https://registry.terraform.io/providers/hashicorp/aws/latest/docs#authentication for details."
        exit 3
    fi
elif [ "$1" = "azure" ]; then
    if [[ -z "$ARM_CLIENT_ID" || -z "$ARM_CLIENT_SECRET" || -z "$ARM_SUBSCRIPTION_ID" || -z "$ARM_TENANT_ID"  ]]; then
        echo "Cloud Crendentials missing"
        echo "Check if these varaiables are set : ARM_CLIENT_ID  ARM_CLIENT_SECRET ARM_SUBSCRIPTION_ID ARM_TENANT_ID"
        exit 3
    fi
elif [ "$1" = "google" ]; then
    if [ -z "$GOOGLE_APPLICATION_CREDENTIALS" ] || [ -z "$GOOGLE_CREDENTIALS" ]; then
        echo "Error: Google Cloud credentials are not set."
        echo "Please set either GOOGLE_APPLICATION_CREDENTIALS (path to JSON file) or GOOGLE_CREDENTIALS (JSON content)."
        exit 3
    fi
fi

# Set cloud provider environment variables
DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
export TERRAFORM_ROOT="/mnt/efs/terraform"
case "$1" in
    google)
        export TF_VAR_cloud_key=$GOOGLE_APPLICATION_CREDENTIALS
        export TF_VAR_cloud_id=$GOOGLE_PROJECT
        export TF_VAR_project=$GOOGLE_PROJECT
        export TF_VAR_google_credentials=$GOOGLE_APPLICATION_CREDENTIALS
        ;;
    aws)
        export TF_VAR_cloud_id=$AWS_ACCESS_KEY_ID
        export TF_VAR_cloud_key=$AWS_SECRET_ACCESS_KEY
        ;;
    azure)
        export TF_VAR_cloud_id=$ARM_CLIENT_ID
        export TF_VAR_cloud_key=$ARM_CLIENT_SECRET
        export TF_VAR_subscription_id=$ARM_SUBSCRIPTION_ID
        export TF_VAR_tenant_id=$ARM_TENANT_ID
        ;;
    local)
        export TF_VAR_vpn_container=true
        ;;
    *)
        usage
        ;;
esac

# Validate action
case "$2" in
    destroyinfra|destroycluster|destroyall)
        if [ -f "$DIR/etc/spillbox_config" ]; then
            tmp=$(grep workdir "$DIR/etc/spillbox_config" | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
            if [ -n "$tmp" ]; then
                LOGDIR="$tmp"
            else
                LOGDIR="/mnt/efs/customers"
            fi
        else
            LOGDIR="/mnt/efs/customers"
        fi
        ;;
    *)
        usage
        ;;
esac

if [ "$1" == "local" ]; then
    case "$3" in
        all)
            proc=spld.hub
            ;;
        *)
            proc=spld.hub.$3
            ;;
    esac

    for spld in $(pgrep -f "$proc")
    do
        spldproj=$(ps aux | grep $spld | grep -v grep | awk '{print $13}')
        USER=$(echo $spldproj | awk -F "sites/local/hub/" '{print $2}' | awk -F '/' '{print $1}')
        PROJECTNAME=$(echo "$spldproj" | awk -F "sites/local/hub/$USER" '{print $2}' | awk -F - '{print $1}' | sed 's|^/||')
        LOGDIR=$(echo $spldproj | awk -F '/.spillbox' '{print $1}')
        if [ -f $spldproj ]; then
            site_tag=$(cat $spldproj | awk -F '"' '/^siteid/ {print $2}')
            export SPILLBOX_SITE="$site_tag"
            export TF_VAR_terraformroot="$LOGDIR/.spillbox/sites/$1/$site_tag//terraform"
            ret=$?
            if [ $ret -ne 0 ]; then
                mkdir -p $LOGDIR/.spillbox/sites/$1/tmpjunk/junk
                site_tag="tmpjunk"
            fi
        else
            kill -9 $spld
            mkdir -p $LOGDIR/.spillbox/sites/$1/tmpjunk/junk
            site_tag="tmpjunk"
        fi
        site_port=$(cat $LOGDIR/.spillbox/sites/$1/$site_tag/portlog 2> /dev/null)
        ret=$?
        if [ $ret -ne 0 ]; then
            site_port="Not known"
        fi
        WDIR="$LOGDIR/.spillbox/customers"
        PROJ_WDIR="$WDIR/$USER/$PROJECTNAME-$site_tag"
        if ! [[  -d $PROJ_WDIR && -d $LOGDIR/.spillbox/sites/local/hub/$USER/$PROJECTNAME-$site_tag ]]; then
            echo "Project $USER-$PROJECTNAME , SERVER=`hostname -f | awk '{print $1}'`:$site_port is getting destroyed "
            if  [ -d $PROJ_WDIR ]; then
                . $PROJ_WDIR/etc/spillbox_config
                if [ "$job_manager" != "" ] ; then
                    . $user_setup_file
                fi
            fi
        fi
        if [ "$job_manager" != "" ] ; then
            . $user_setup_file
        fi
        echo "Project $USER-$PROJECTNAME , SERVER=`hostname -f | awk '{print $1}'`:$site_port is getting destroyed "
        if [ "$2" = "destroyall" ]; then
                 (cd $LOGDIR/.spillbox/sites/$1/$site_tag/terraform; ./terraform_infra.sh $1 destroycluster $USER $PROJECTNAME)
                 (cd $LOGDIR/.spillbox/sites/$1/$site_tag/terraform; ./terraform_infra.sh $1 destroyinfra $USER $PROJECTNAME -rmcache -rmworkdir)
                (export TERRAFORM_ROOT=$PROJ_WDIR; cd $LOGDIR/.spillbox/sites/$1/$site_tag/terraform; ./hub.sh $USER-$PROJECTNAME stop &> /dev/null)
        elif [ "$2" = "destroyinfra" ]; then
                (cd $LOGDIR/.spillbox/sites/$1/$site_tag/terraform; ./terraform_infra.sh $1 destroyinfra $USER $PROJECTNAME -rmcache -rmworkdir)
                (export TERRAFORM_ROOT=$PROJ_WDIR; cd $LOGDIR/.spillbox/sites/$1/$site_tag/terraform; ./hub.sh $USER-$PROJECTNAME stop &> /dev/null)
        fi
done
else
    # Base directory for customers and SSH
    BASE_DIR="/mnt/efs/customers"
    SSH_DIR="$HOME/.ssh_spillbox"

    # Process input arguments
    if [ "$#" -eq 3 ]; then
        # Case 2: Customer ID only
        CUSTOMER_ID="$3"
        PROJ_WDIR="$BASE_DIR/$CUSTOMER_ID"
        PROJECTS=()
        if [ -d "$PROJ_WDIR" ]; then
            # Check directory readability
            if [ ! -r "$PROJ_WDIR" ]; then
                echo "Warning: Directory $PROJ_WDIR is not readable, attempting cleanup anyway"
            else
                # Use find to ensure directories are detected
                while IFS= read -r PROJ_DIR; do
                if [ -d "$PROJ_DIR" ]; then
                    PROJECT_NAME=$(basename "$PROJ_DIR")
                    PROJECTS+=("$CUSTOMER_ID-$PROJECT_NAME")
                fi
                done < <(find "$PROJ_WDIR" -maxdepth 1 -type d -not -path "$PROJ_WDIR")
                if [ ${#PROJECTS[@]} -eq 0 ]; then
                    echo "No project directories found under $PROJ_WDIR, proceeding with cleanup"
                fi
            fi
        else
            echo "Customer directory $PROJ_WDIR does not exist, proceeding with instance cleanup"
        fi
    elif [ "$#" -eq 4 ]; then
        # Case 3: Customer ID and project name
        CUSTOMER_ID="$3"
        PROJECT_NAME="$4"
        PROJ_WDIR="$BASE_DIR/$CUSTOMER_ID/$PROJECT_NAME"
        PROJECTS=("$CUSTOMER_ID-$PROJECT_NAME")
    else
        usage
    fi

    if [ "$1" == "aws" ]; then
        AWS_DEFAULT_REGION=$(awk -F'"' '/region/ {print $4}' $PROJ_WDIR/vars/aws/terraform.u.auto.tfvars.json)
        destroycluster_aws() {
            local CUSTOMER_ID=$1
            local PROJECTNAME=$2
            local JOB_MANAGER=$3
            local AWS_DEFAULT_REGION=$4

            local MASTER_INSTANCE_ID=""
            local WORKER_INSTANCE_ID=""

            # Helper function for fetching EC2 instance IDs by tag
            get_instance_ids() {
                local tag=$1
                aws ec2 describe-instances \
                    --region "$AWS_DEFAULT_REGION" \
                    --filters "Name=tag:Name,Values=$tag" \
                    --query "Reservations[].Instances[].InstanceId" \
                    --output text
            }
            case "$JOB_MANAGER" in
                ssh)
                    MASTER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-master-worker-*")
                    WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-slave-worker-*")

                    # Fallback if no slave-worker instances
                    if [ -z "$WORKER_INSTANCE_ID" ]; then
                        WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-worker-*")
                    fi
                    ;;
                slurm)
                    MASTER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-master-*")
                    WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-worker-*")
                    ;;
                null)
                    MASTER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-master-worker-*")
                    WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-worker-*")

                    # Fallback if no worker instances
                    if [ -z "$WORKER_INSTANCE_ID" ]; then
                        WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-slave-worker-*")
                    fi
                    ;;
                *)
                    echo "Unknown JOB_MANAGER: $JOB_MANAGER" >&2
                    return 1
                    ;;
            esac
            # Convert the output to arrays for processing
            read -ra MASTER_IDS_ARRAY <<< "$MASTER_INSTANCE_ID"
            read -ra WORKER_IDS_ARRAY <<< "$WORKER_INSTANCE_ID"

            # Combine all instance IDs into a single array
            ALL_INSTANCE_IDS=("${MASTER_IDS_ARRAY[@]}" "${WORKER_IDS_ARRAY[@]}")
            if [ ${#ALL_INSTANCE_IDS[@]} -eq 0 ]; then
                echo "No instances found for $CUSTOMER_ID-$PROJECTNAME"
            else
                echo "Terminating instances: ${ALL_INSTANCE_IDS[@]}"
                aws ec2 terminate-instances \
                --instance-ids "${ALL_INSTANCE_IDS[@]}" \
                --no-cli-pager --output text >/dev/null 2>&1 || {
                    echo "Warning: Failed to terminate some or all instances: ${ALL_INSTANCE_IDS[@]}"
                }
            fi
        }

        # Process each project
        for PROJECT in "${PROJECTS[@]}"; do
            USER=$(echo "$PROJECT" | awk -F- '{print $1}')
            PROJECTNAME=$(echo "$PROJECT" | cut -d- -f2-)
            if [ "$#" -ne 3 ] || [[ "$3" != *"/"* ]]; then
                PROJ_WDIR="$BASE_DIR/$USER/$PROJECTNAME"
            fi
            # Check if project directory exists
            if [ -d "$PROJ_WDIR" ]; then
                echo "Processing project $USER-$PROJECTNAME"
                (
                    cd "$PROJ_WDIR" || { echo "Failed to cd to $PROJ_WDIR"; exit 1; }
                    export TERRAFORM_ROOT="$PROJ_WDIR"
                    if [ -f "./terraform_command.sh" ]; then
                        JOB_MANAGER=$(awk -F'"' '/job_manager/ {print $4}' $PROJ_WDIR/vars/aws/terraform.u.auto.tfvars.json)
                        if [ -z "$JOB_MANAGER" ]; then
                            JOB_MANAGER="null"
                        fi
                        if [ "$2" = "destroyall" ]; then
                            echo "Executing destroycluster for $USER-$PROJECTNAME"
                            destroycluster_aws "$CUSTOMER_ID" "$PROJECTNAME" "$JOB_MANAGER" "$AWS_DEFAULT_REGION"
                            sleep 30
                            echo "Executing destroyinfra for $USER-$PROJECTNAME"
                            $PROJ_WDIR/terraform_command.sh "$1" "destroyinfra" "$USER" "$PROJECTNAME" >/dev/null 2>&1  || {
                                echo "Warning: terraform_command.sh destroyinfra failed for $USER-$PROJECTNAME";
                            }
                        elif [ "$2" = "destroycluster" ]; then
                            # For now destroying with the aws cli
                            destroycluster_aws "$CUSTOMER_ID" "$PROJECTNAME" "$JOB_MANAGER" "$AWS_DEFAULT_REGION"

                        else
                            echo "Executing $2 for $USER-$PROJECTNAME"
                            $PROJ_WDIR/terraform_command.sh "$1" "$2" "$USER" "$PROJECTNAME" || {
                                echo "Warning: terraform_command.sh failed for $USER-$PROJECTNAME";
                            }
                        fi
                    else
                        echo "Error: terraform_command.sh not found in $PROJ_WDIR";
                        exit 1;
                    fi
                )
                # Additional check for s3 bucket and tran & splfs machine if terraform failed to destroy machines.
                ret=$?
                if test $ret -ne 0 ; then
                    if [[ "$JOB_MANAGER" == "ssh" || "$JOB_MANAGER" == "slurm"  ]]; then
                        TRAN_INSTANCE_ID=$(aws ec2 describe-instances \
                        --region "$AWS_DEFAULT_REGION" \
                        --filters "Name=tag:Name,Values=$CUSTOMER_ID-$PROJECTNAME-tran-00" \
                        --query "Reservations[].Instances[].InstanceId" \
                        --output text)

                        SPLFS_INSTANCE_ID=$(aws ec2 describe-instances \
                        --region "$AWS_DEFAULT_REGION" \
                        --filters "Name=tag:Name,Values=$CUSTOMER_ID-$PROJECTNAME-splfs-00" \
                        --query "Reservations[].Instances[].InstanceId" \
                        --output text)
                    elif [ "$JOB_MANAGER" == "null" ]; then
                        SPLFS_INSTANCE_ID=$(aws ec2 describe-instances \
                        --region "$AWS_DEFAULT_REGION" \
                        --filters "Name=tag:Name,Values=$CUSTOMER_ID-$PROJECTNAME-splfs-00" \
                        --query "Reservations[].Instances[].InstanceId" \
                        --output text)
                    fi
                    # --- Collect instances into array ---
                    ALL_INSTANCE_IDS=()
                    [[ -n "$TRAN_INSTANCE_ID" ]] && ALL_INSTANCE_IDS+=($TRAN_INSTANCE_ID)
                    [[ -n "$SPLFS_INSTANCE_ID" ]] && ALL_INSTANCE_IDS+=($SPLFS_INSTANCE_ID)
                    # --- Terminate only if we have values ---
                    if [ ${#ALL_INSTANCE_IDS[@]} -eq 0 ]; then
                        echo "No tran/splfs instances found for $CUSTOMER_ID-$PROJECTNAME"
                    else
                        INSTANCE_NAMES=$(aws ec2 describe-instances \
                        --region "$AWS_DEFAULT_REGION" \
                        --instance-ids "${ALL_INSTANCE_IDS[@]}" \
                        --query "Reservations[].Instances[].Tags[?Key=='Name'].Value" \
                        --output text)
                        echo "Terminating instances: $INSTANCE_NAMES"
                        aws ec2 terminate-instances \
                        --instance-ids "${ALL_INSTANCE_IDS[@]}" \
                        --no-cli-pager >/dev/null 2>&1 || {
                        echo "Warning: Failed to terminate some or all instances: $INSTANCE_NAMES"
                        }
                    fi
                    S3_BUCKET="ts-$CUSTOMER_ID-$PROJECTNAME"
                    S3_BUCKET_LOGS="ts-$CUSTOMER_ID-$PROJECTNAME-logs"
                    delete_bucket_if_exists() {
                        local BUCKET_NAME=$1
                        if aws s3api head-bucket --bucket "$BUCKET_NAME" --region "$AWS_DEFAULT_REGION" 2>/dev/null; then
                            echo "Bucket $BUCKET_NAME exists. Deleting..."
                            aws s3 rb "s3://$BUCKET_NAME" --force --region "$AWS_DEFAULT_REGION"
                        else
                            echo "Bucket $BUCKET_NAME does not exist."
                            fi
                    }
                    delete_bucket_if_exists "$S3_BUCKET"
                    delete_bucket_if_exists "$S3_BUCKET_LOGS"
                fi
                #Removing Project directory
                ret=$?
                if test $ret -eq 0 ; then
                    if [ -d "$PROJ_WDIR" ]; then
                        echo "Removing project directory $PROJ_WDIR"
                        rm -rf "$PROJ_WDIR" 2>/dev/null || echo "Warning: Failed to remove project directory $PROJ_WDIR"
                    fi
                else
                    echo "Failed to finish the terraform_command.sh"
                fi
                # Remove SSH directory
                ret=$?
                if test $ret -eq 0 ; then
                    SSH_PROJECT_DIR="$SSH_DIR/$USER-$PROJECTNAME"
                    if [ -d "$SSH_PROJECT_DIR" ]; then
                        echo "Removing SSH directory $SSH_PROJECT_DIR"
                        rm -rf "$SSH_PROJECT_DIR" 2>/dev/null || {
                            echo "Warning: Failed to remove SSH directory $SSH_PROJECT_DIR";
                        }
                    else
                        echo "SSH directory $SSH_PROJECT_DIR does not exist"
                    fi
                else
                    echo "Failed to remove $PROJ_WDIR directory"
                fi
            else
                echo "Project directory $PROJ_WDIR does not exist"
                exit 1
            fi
        done
        # If customer ID only, attempt to remove empty customer directory
        if [ "$#" -eq 3 ] && [[ "$3" != *"/"* ]]; then
            CUSTOMER_DIR="$BASE_DIR/$CUSTOMER_ID"
            if [ -d "$CUSTOMER_DIR" ] && [ -z "$(ls -A "$CUSTOMER_DIR")" ]; then
                echo "Customer directory $CUSTOMER_DIR is empty, removing it"
                rmdir "$CUSTOMER_DIR" 2>/dev/null || echo "Warning: Failed to remove empty customer directory $CUSTOMER_DIR"
            fi
        fi

    elif [[ "$1" == "azure" ]]; then
        # Function to delete Azure VMs by name pattern
        destroycluster_azure() {
            local CUSTOMER_ID=$1
            local PROJECTNAME=$2
            local JOB_MANAGER=$3
            local RESOURCE_GROUP=$4
            local MASTER_INSTANCE_ID=""
            local WORKER_INSTANCE_ID=""

            # Helper function to fetch VM IDs by name
            get_instance_ids() {
                local name_pattern=$1
                az vm list -g "$RESOURCE_GROUP" \
                    --query "[?name=='$name_pattern'].id" \
                    --output tsv 2>/dev/null
                }

            case "$JOB_MANAGER" in
                ssh)
                    MASTER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-master-worker-00")
                    WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-slave-worker-00")

                    # Fallback if no slave-worker instances
                    if [ -z "$WORKER_INSTANCE_ID" ]; then
                        WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-worker-00")
                    fi
                    ;;
                slurm)
                    MASTER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-master-00")
                    WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-worker-00")
                    ;;
                null)
                    MASTER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-master-worker-00")
                    WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-worker-00")

                    # Fallback if no worker instances
                    if [ -z "$WORKER_INSTANCE_ID" ]; then
                        WORKER_INSTANCE_ID=$(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-slave-worker-00")
                   fi
                   ;;
                *)
                    echo "Unknown JOB_MANAGER: $JOB_MANAGER" >&2
                    return 1
                    ;;
            esac

            # Combine all instance IDs into a single array
            read -ra MASTER_IDS_ARRAY <<< "$MASTER_INSTANCE_ID"
            read -ra WORKER_IDS_ARRAY <<< "$WORKER_INSTANCE_ID"
            ALL_INSTANCE_IDS=("${MASTER_IDS_ARRAY[@]}" "${WORKER_IDS_ARRAY[@]}")

            if [ ${#ALL_INSTANCE_IDS[@]} -eq 0 ]; then
                echo "No instances found for $CUSTOMER_ID-$PROJECTNAME in resource group $RESOURCE_GROUP"
            else
                # Get VM names for logging
                ALL_INSTANCE_NAMES=()
                for vm_id in "${ALL_INSTANCE_IDS[@]}"; do
                    vm_name=$(az vm show --ids "$vm_id" --query "name" --output tsv 2>/dev/null)
                    [[ -n "$vm_name" ]] && ALL_INSTANCE_NAMES+=("$vm_name")
                done
                echo "Deleting Azure VMs: ${ALL_INSTANCE_NAMES[*]}"
                for vm_id in "${ALL_INSTANCE_IDS[@]}"; do
                    vm_name=$(az vm show --ids "$vm_id" --query "name" --output tsv 2>/dev/null)
                    if az resource delete --ids "$vm_id" >/dev/null 2>&1; then
                        echo "Successfully deleted VM: $vm_name"
                    else
                        echo "Warning: Failed to delete VM with ID: $vm_id"
                    fi
                done
            fi
        }
        # For each project
        for PROJECT in "${PROJECTS[@]}"; do
            USER=$(echo "$PROJECT" | awk -F- '{print $1}')
            PROJECTNAME=$(echo "$PROJECT" | cut -d- -f2-)
            RESOURCE_GROUP="ts-$CUSTOMER_ID-$PROJECTNAME-rg"
            if [ "$#" -ne 3 ] || [[ "$3" != *"/"* ]]; then
                PROJ_WDIR="$BASE_DIR/$USER/$PROJECTNAME"
            fi
            # Check if project directory exists
            if [ -d "$PROJ_WDIR" ]; then
                echo "Processing project $USER-$PROJECTNAME"
                (
                    cd "$PROJ_WDIR" || { echo "Failed to cd to $PROJ_WDIR"; exit 1; }
                    export TERRAFORM_ROOT="$PROJ_WDIR"
                    if [ -f "./terraform_command.sh" ]; then
                        JOB_MANAGER=$(awk -F'"' '/job_manager/ {print $4}' $PROJ_WDIR/vars/azure/terraform.u.auto.tfvars.json)
                        if [ -z "$JOB_MANAGER" ]; then
                            JOB_MANAGER="null"
                        fi
                        if [ "$2" = "destroyall" ]; then
                            echo "Executing destroycluster for $USER-$PROJECTNAME"
                            destroycluster_azure "$CUSTOMER_ID" "$PROJECTNAME" "$JOB_MANAGER" "$RESOURCE_GROUP"
                            echo "Executing destroyinfra for $USER-$PROJECTNAME"
                            $PROJ_WDIR/terraform_command.sh "$1" "destroyinfra" "$USER" "$PROJECTNAME" >/dev/null 2>&1 || {
                                echo "Warning: terraform_command.sh destroyinfra failed for $USER-$PROJECTNAME";
                            }
                        elif [ "$2" = "destroycluster" ]; then
                            destroycluster_azure "$CUSTOMER_ID" "$PROJECTNAME" "$JOB_MANAGER" "$RESOURCE_GROUP"
                        else
                            echo "Executing $2 for $USER-$PROJECTNAME"
                            $PROJ_WDIR/terraform_command.sh "$1" "$2" "$USER" "$PROJECTNAME" >/dev/null 2>&1 || {
                                echo "Warning: terraform_command.sh failed for $USER-$PROJECTNAME";
                            }
                        fi
                    else
                        echo "Error: terraform_command.sh not found in $PROJ_WDIR"
                        exit 1
                    fi
                )
                ret=$?
                if [ $ret -ne 0 ]; then
                    # Initialize arrays to store VM IDs and names
                    ALL_INSTANCE_IDS=()
                    ALL_INSTANCE_NAMES=()
                    if [[ "$JOB_MANAGER" == "ssh" || "$JOB_MANAGER" == "slurm" ]]; then
                        # Query for tran-00 VM
                        TRAN_VM=$(az vm list -g "$RESOURCE_GROUP" \
                            --query "[?name=='$CUSTOMER_ID-$PROJECTNAME-tran-00'].{id:id, name:name}" \
                            --output json 2>/dev/null)

                        if [ -n "$TRAN_VM" ]; then
                            TRAN_INSTANCE_ID=$(echo "$TRAN_VM" | jq -r '.[]?.id')
                            TRAN_INSTANCE_NAME=$(echo "$TRAN_VM" | jq -r '.[]?.name')
                            [[ -n "$TRAN_INSTANCE_ID" ]] && ALL_INSTANCE_IDS+=("$TRAN_INSTANCE_ID")
                            [[ -n "$TRAN_INSTANCE_NAME" ]] && ALL_INSTANCE_NAMES+=("$TRAN_INSTANCE_NAME")
                        fi
                        # Query for splfs-00 VM
                        SPLFS_VM=$(az vm list -g "$RESOURCE_GROUP" \
                            --query "[?name=='$CUSTOMER_ID-$PROJECTNAME-splfs-00'].{id:id, name:name}" \
                            --output json 2>/dev/null)

                        if [ -n "$SPLFS_VM" ]; then
                            SPLFS_INSTANCE_ID=$(echo "$SPLFS_VM" | jq -r '.[]?.id')
                            SPLFS_INSTANCE_NAME=$(echo "$SPLFS_VM" | jq -r '.[]?.name')
                            [[ -n "$SPLFS_INSTANCE_ID" ]] && ALL_INSTANCE_IDS+=("$SPLFS_INSTANCE_ID")
                            [[ -n "$SPLFS_INSTANCE_NAME" ]] && ALL_INSTANCE_NAMES+=("$SPLFS_INSTANCE_NAME")
                        fi
                    elif [ "$JOB_MANAGER" == "null" ]; then
                        # Query only for splfs-00 VM
                        SPLFS_VM=$(az vm list -g "$RESOURCE_GROUP" \
                            --query "[?name=='$CUSTOMER_ID-$PROJECTNAME-splfs-00'].{id:id, name:name}" \
                            --output json 2>/dev/null)

                        if [ -n "$SPLFS_VM" ]; then
                            SPLFS_INSTANCE_ID=$(echo "$SPLFS_VM" | jq -r '.[]?.id')
                            SPLFS_INSTANCE_NAME=$(echo "$SPLFS_VM" | jq -r '.[]?.name')
                            [[ -n "$SPLFS_INSTANCE_ID" ]] && ALL_INSTANCE_IDS+=("$SPLFS_INSTANCE_ID")
                            [[ -n "$SPLFS_INSTANCE_NAME" ]] && ALL_INSTANCE_NAMES+=("$SPLFS_INSTANCE_NAME")
                        fi
                    fi
                    # Check if any VMs were found
                    if [ ${#ALL_INSTANCE_IDS[@]} -eq 0 ]; then
                        echo "No tran/splfs VMs found for $CUSTOMER_ID-$PROJECTNAME in resource group $RESOURCE_GROUP"
                    else
                        echo "Deleting Azure VMs: ${ALL_INSTANCE_NAMES[*]}"
                        for vm_id in "${ALL_INSTANCE_IDS[@]}"; do
                            vm_name=$(az vm show --ids "$vm_id" --query "name" --output tsv 2>/dev/null)
                            if az resource delete --ids "$vm_id" >/dev/null 2>&1; then
                                echo "Successfully deleted VM: $vm_name"
                            else
                                echo "Warning: Failed to delete VM with ID: $vm_id"
                            fi
                        done
                    fi
                fi
                # Function to delete Azure Storage Account if it exists
                delete_storage_account_if_exists() {
                    local STORAGE_ACCOUNT_NAME=$1
                    if az storage account show --name "$STORAGE_ACCOUNT_NAME" --resource-group "$RESOURCE_GROUP" --query "name" --output tsv 2>/dev/null; then
                        echo "Storage Account $STORAGE_ACCOUNT_NAME exists in resource group $RESOURCE_GROUP. Deleting..."
                        az storage account delete --name "$STORAGE_ACCOUNT_NAME" --resource-group "$RESOURCE_GROUP" --yes >/dev/null 2>&1
                        if [ $? -eq 0 ]; then
                            echo "Successfully deleted Storage Account: $STORAGE_ACCOUNT_NAME"
                        else
                            echo "Warning: Failed to delete Storage Account: $STORAGE_ACCOUNT_NAME"
                        fi
                    else
                        echo "Storage Account $STORAGE_ACCOUNT_NAME does not exist in resource group $RESOURCE_GROUP."
                    fi
                }

                STORAGE_ACCOUNTS=$(az storage account list --resource-group "$RESOURCE_GROUP" \
                    --query "[?contains(name, '$PROJECTNAME')].name" --output tsv 2>/dev/null)
                for sa in $STORAGE_ACCOUNTS; do
                    delete_storage_account_if_exists "$sa"
                done
                sleep 5
                # Function to delete Azure Resource Group if it exists
                delete_resource_group_if_exists() {
                    local RG_NAME=$1

                    if az group show --name "$RG_NAME" --query "name" --output tsv 2>/dev/null; then
                        echo "Resource Group $RG_NAME exists. Deleting..."
                        az group delete --name "$RG_NAME" --yes --no-wait >/dev/null 2>&1
                        if [ $? -eq 0 ]; then
                            echo "Successfully deleted Resource Group: $RG_NAME"
                        else
                            echo "Warning: Failed to delete Resource Group: $RG_NAME"
                        fi
                    else
                        echo "Resource Group $RG_NAME does not exist."
                    fi
                }
                # --- Delete Resource Group ---
                delete_resource_group_if_exists "$RESOURCE_GROUP"
                sleep 10
                # Move project directory to junk
                ret=$?
                if test $ret -eq 0 ; then
                    if [ -d "$PROJ_WDIR" ]; then
                        echo "Removing project directory $PROJ_WDIR"
                        rm -rf "$PROJ_WDIR" 2>/dev/null || echo "Warning: Failed to remove project directory $PROJ_WDIR"
                    fi
                else
                    echo "Failed to finish the terraform_command.sh"
                fi
                sleep 1
                # Remove SSH directory
                ret=$?
                if test $ret -eq 0 ; then
                    SSH_PROJECT_DIR="$SSH_DIR/$USER-$PROJECTNAME"
                    if [ -d "$SSH_PROJECT_DIR" ]; then
                            echo "Removing SSH directory $SSH_PROJECT_DIR"
                            rm -rf "$SSH_PROJECT_DIR" 2>/dev/null || {
                            echo "Warning: Failed to remove SSH directory $SSH_PROJECT_DIR"
                        }
                    else
                        echo "SSH directory $SSH_PROJECT_DIR does not exist"
                    fi
                else
                    echo "Failed to remove $PROJ_WDIR directory"
                fi
            else
                echo "Project directory $PROJ_WDIR does not exist"
                exit 1
            fi
        done
    elif [ "$1" == "google" ]; then
        CUSTOMER_ID=$(echo "$3" | tr -d '0-9')

        destroycluster() {
            local CUSTOMER_ID=$1
            local PROJECTNAME=$2
            local JOB_MANAGER=$3

            local MASTER_INSTANCES=()
            local WORKER_INSTANCES=()

            get_instance_ids() {
                local pattern=$1
                # Print both name and zone separated by a space
                gcloud compute instances list \
                    --filter="name~'$pattern'" \
                    --format="value(name,zone)" 2>/dev/null
            }

            case "$JOB_MANAGER" in
                ssh)
                    mapfile -t MASTER_INSTANCES < <(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-master-worker-.*")
                    mapfile -t WORKER_INSTANCES < <(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-slave-worker-.*")
                    if [ ${#WORKER_INSTANCES[@]} -eq 0 ]; then
                        mapfile -t WORKER_INSTANCES < <(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-worker-.*")
                    fi
                    ;;
                slurm)
                    mapfile -t MASTER_INSTANCES < <(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-master-.*")
                    mapfile -t WORKER_INSTANCES < <(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-worker-.*")
                    ;;
                null)
                    mapfile -t MASTER_INSTANCES < <(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-master-worker-.*")
                    mapfile -t WORKER_INSTANCES < <(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-worker-.*")
                    if [ ${#WORKER_INSTANCES[@]} -eq 0 ]; then
                        mapfile -t WORKER_INSTANCES < <(get_instance_ids "$CUSTOMER_ID-$PROJECTNAME-slave-worker-.*")
                    fi
                    ;;
                *)
                    echo "❌ Unknown JOB_MANAGER: $JOB_MANAGER" >&2
                    return 1
                    ;;
            esac

            # Merge all instances
            ALL_INSTANCES=("${MASTER_INSTANCES[@]}" "${WORKER_INSTANCES[@]}")

            if [ ${#ALL_INSTANCES[@]} -eq 0 ]; then
                echo "No instances found for $CUSTOMER_ID-$PROJECTNAME"
            else
                echo "Deleting instances:"
                for inst in "${ALL_INSTANCES[@]}"; do
                    NAME=$(echo "$inst" | awk '{print $1}')
                    ZONE=$(echo "$inst" | awk '{print $2}')
                    echo "  - $NAME ($ZONE)"
                    gcloud compute instances delete "$NAME" --zone "$ZONE" --quiet >/dev/null 2>&1
                done
            fi
        }

        # Process projects same as AWS
        for PROJECT in "${PROJECTS[@]}"; do
            USER=$(echo "$PROJECT" | awk -F- '{print $1}')
            PROJECTNAME=$(echo "$PROJECT" | cut -d- -f2-)
            PROJ_WDIR="$BASE_DIR/$USER/$PROJECTNAME"

            if [ -d "$PROJ_WDIR" ]; then
                echo "Processing GCP project $USER-$PROJECTNAME"
                (
                    cd "$PROJ_WDIR" || { echo "Failed to cd to $PROJ_WDIR"; exit 1; }
                    export TERRAFORM_ROOT="$PROJ_WDIR"

                    if [ -f "./terraform_command.sh" ]; then
                        JOB_MANAGER=$(awk -F'"' '/job_manager/ {print $4}' "$PROJ_WDIR/vars/google/terraform.u.auto.tfvars.json")
                        [ -z "$JOB_MANAGER" ] && JOB_MANAGER="null"

                        if [ "$2" = "destroyall" ]; then
                            destroycluster "$CUSTOMER_ID" "$PROJECTNAME" "$JOB_MANAGER"
                            echo "Processing destroyinfra for project $USER-$PROJECTNAME"
                            $PROJ_WDIR/terraform_command.sh "$1" "destroyinfra" "$USER" "$PROJECTNAME" || \
                                echo "Warning: terraform_command.sh destroyinfra failed for $USER-$PROJECTNAME"
                        elif [ "$2" = "destroycluster" ]; then
                            destroycluster "$CUSTOMER_ID" "$PROJECTNAME" "$JOB_MANAGER"
                        else
                            $PROJ_WDIR/terraform_command.sh "$1" "$2" "$USER" "$PROJECTNAME" || \
                                echo "Warning: terraform_command.sh failed for $USER-$PROJECTNAME"
                        fi
                    else
                        echo "Error: terraform_command.sh not found in $PROJ_WDIR"
                        exit 1
                    fi
                )

                # Delete GCP buckets if terraform fails
                ret=$?
                if [ $ret -ne 0 ]; then
                    BUCKET_NAME="ts-$CUSTOMER_ID-$PROJECTNAME"
                    BUCKET_LOGS="ts-$CUSTOMER_ID-$PROJECTNAME-logs"

                    delete_bucket_if_exists() {
                        local bucket=$1
                        if gsutil ls -b "gs://$bucket" >/dev/null 2>&1; then
                            echo "Deleting bucket $bucket..."
                            gsutil -m rm -r "gs://$bucket"
                        else
                            echo "Bucket $bucket does not exist"
                        fi
                    }

                    delete_bucket_if_exists "$BUCKET_NAME"
                    delete_bucket_if_exists "$BUCKET_LOGS"
                fi

                #Removing Project directory
                ret=$?
                if test $ret -eq 0 ; then
                    if [ -d "$PROJ_WDIR" ]; then
                        echo "Removing project directory $PROJ_WDIR"
                        rm -rf "$PROJ_WDIR" 2>/dev/null || echo "Warning: Failed to remove project directory $PROJ_WDIR"
                    fi
                else
                    echo "Failed to finish the terraform_command.sh"
                fi
                # Remove SSH directory
                ret=$?
                if test $ret -eq 0 ; then
                    SSH_PROJECT_DIR="$SSH_DIR/$USER-$PROJECTNAME"
                    if [ -d "$SSH_PROJECT_DIR" ]; then
                        echo "Removing SSH directory $SSH_PROJECT_DIR"
                        rm -rf "$SSH_PROJECT_DIR" 2>/dev/null || {
                            echo "Warning: Failed to remove SSH directory $SSH_PROJECT_DIR";
                        }
                    else
                        echo "SSH directory $SSH_PROJECT_DIR does not exist"
                    fi
                else
                    echo "Failed to remove $PROJ_WDIR directory"
                fi
            else
                echo "Project directory $PROJ_WDIR does not exist"
                exit 1
            fi
        done
    fi
    # If customer ID only, attempt to remove empty customer directory
    if [ "$#" -eq 3 ] && [[ "$3" != *"/"* ]]; then
        CUSTOMER_DIR="$BASE_DIR/$CUSTOMER_ID"
        if [ -d "$CUSTOMER_DIR" ] && [ -z "$(ls -A "$CUSTOMER_DIR")" ]; then
            echo "Customer directory $CUSTOMER_DIR is empty, removing it"
            rmdir "$CUSTOMER_DIR" 2>/dev/null || echo "Warning: Failed to remove empty customer directory $CUSTOMER_DIR"
        fi
    fi
fi
exit 0
