#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

usage()
{
    echo "Usage : $0 aws|google|azure <create|delete|prune>" >&2
}

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
if test "$#" -lt 2; then
    usage;
    exit 3
fi

if ! command -v juicefs &> /dev/null ; then
    echo "Installing Juicefs"
    sudo curl -sSL https://d.juicefs.com/install | sh - >&2
fi
APT="apt-get"
OSID=$(cat /etc/os-release | grep ^ID= | awk -F= '{print $2}' | sed 's/"//g')
if [[ $OSID == "centos" ]] || [[ "$OSID" == "rhel" ]] || [[ "$OSID" == "amzn" ]] || [[ "$OSID" == "rocky" ]]; then
    APT=yum
fi
if ! rpm -qa | grep -q "^lsof"; then
    echo "Installing lsof" >&2
    sudo $APT install lsof -y >&2
fi
if ! rpm -qa | grep -q "^fuse" || ! command -v fusermount &> /dev/null; then
    echo "FUSE is not installed. Please installing FUSE" >&2
    sudo $APT install fuse -y >&2
fi

if [ "$2" == "prune" ]; then
    if [ -f  $DIR/etc/spillbox_config ]; then
        redis_port=$(grep ^checkpoint_objport $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
        if [ "$redis_port" == "" ]; then
            echo "checkpoint_objport is not set in $DIR/etc/spillbox_config" >&2
            exit 1
        fi
    else
        echo "File missing : $DIR/etc/spillbox_config" >&2
        exit 3
    fi
    MOUNT_PATH="/tmp/juicefs_${redis_port}"
    sudo mkdir -p $MOUNT_PATH
    sudo umount -f $MOUNT_PATH
    tmp=$(which juicefs)
    sudo ${tmp} mount -d redis://`hostname`:${redis_port}/1 $MOUNT_PATH
    RETVAL=$?
    if test ${RETVAL} -ne 0 ; then
        echo "mount failed, status ${RETVAL}" >&2
        exit $RETVAL
    fi
    re='^[0-9]+$'
    for filename in `find "$MOUNT_PATH" -mindepth 1 -maxdepth 1 -type d  -mmin  +2880`; do
        if [ ! -d $filename ]; then
            continue
        fi
        tmp=$(echo $filename | awk -F '/' '{print $NF}')
        if ! [[ $tmp =~ $re ]] ; then
            continue
        fi
        sudo /bin/rm -rf $filename
    done
    sudo umount -f $MOUNT_PATH
    exit 0
fi
case "$1" in
    aws)
        if [ "$AWS_DEFAULT_REGION" == "" ]; then
            echo "AWS_DEFAULT_REGION is not set." >&2
            echo "Please export it manually, e.g.: export AWS_DEFAULT_REGION=us-west-2" >&2
            exit 1
        fi
        for var in AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY ; do
            if [ "${!var}" == "" ]; then
                echo "ENV $var is not set " >&2
                echo "Proceeding using EC2 instance profile (IAM role) if available..." >&2
            fi
        done
        ;;
    azure)
        for var in ARM_CLIENT_ID ARM_CLIENT_SECRET ARM_SUBSCRIPTION_ID ARM_TENANT_ID ARM_RESOURCE_GROUP ; do
            if [ "${!var}" == "" ]; then
                if [ "$var" == "ARM_RESOURCE_GROUP" ]; then
                    echo "Set ARM_RESOURCE_GROUP for your azure storage account" >&2
                else
                    echo "ENV $var is not set" >&2
                fi
                exit 1
            fi
        done
        ;;
    google)
        for var in GOOGLE_APPLICATION_CREDENTIALS GOOGLE_PROJECT GOOGLE_REGION; do
            if [ "${!var}" == "" ]; then
                if [ "$var" == "GOOGLE_REGION" ]; then
                    echo "Set GOOGLE_REGION for your google storage account" >&2
                else
                    echo "ENV $var is not set" >&2
                fi
                exit 1
            fi
        done
        ;;
    *)
        echo "Unknown cloud provider $1" >&2
        usage;
        exit 3
esac

if [ -f  $DIR/etc/spillbox_config ]; then
    bucket_name=$(grep ^checkpoint_bucket $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "$bucket_name" == "" ]; then
        echo "checkpoint_bucket is not set in $DIR/etc/spillbox_config" >&2
        exit 1
    fi
    cp_dir=$(grep ^checkpoint_image_dir $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "$cp_dir" == "" ]; then
        echo "checkpoint_image_dir is not set in $DIR/etc/spillbox_config" >&2
        exit 1
    fi
    redis_port=$(grep ^checkpoint_objport $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "$redis_port" == "" ]; then
        echo "checkpoint_objport is not set in $DIR/etc/spillbox_config" >&2
        exit 1
    fi
    # creating a redis database server in docker
    tmp=$(grep ^docker_path $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "${tmp}" == "" ]; then
        docker_path="docker"
    else
        docker_path="${tmp}"
    fi
else
    echo "File missing : $DIR/etc/spillbox_config" >&2
    exit 3
fi

mkdir -p $DIR/logs/
export TF_VAR_bucket_name=${bucket_name}

case "$2" in
    create)
        port_check=$(sudo lsof -i :$redis_port)
        if [ "$port_check" != "" ]; then
            echo "Port $redis_port is used by other process" >&2
            echo $port_check >&2
            exit 1
        fi
        $docker_path run --rm -p $redis_port:6379 -d --name sbox-$redis_port-jfs redis > $DIR/logs/log.image
        RET=$?
        if test $RET -ne 0 ; then
            echo "Redis docker database failed... Check $DIR/logs/log.image" >&2
            exit 3
        fi
        case "$1" in
            aws)
                export TF_VAR_region=${AWS_DEFAULT_REGION}
                echo "LOG FILE :: $DIR/logs/log.image" >&2
                # Creating a .terraformrc configuration file to define custom provider installation settings.
                cat << END > $DIR/stacks/provider/local/aws-objstore/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/stacks/provider/local/aws-objstore/plugins"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
                FILE_PATH="$DIR/stacks/provider/local/aws-objstore/plugins/registry.terraform.io/hashicorp/aws/5.62.0/linux_amd64/terraform-provider-aws_v5.84.0_x5"
                PLUGIN_DIR_PATH=$(dirname "$FILE_PATH")
                mkdir -p "$PLUGIN_DIR_PATH"
                # Check if the file exists
                if [ ! -f "$FILE_PATH" ]; then
                    curl -s -o "$FILE_PATH" \
                    https://s3.us-west-1.amazonaws.com/spillbox.io-internal/plugins/registry.terraform.io/hashicorp/aws/5.84.0/linux_amd64/terraform-provider-aws_v5.84.0_x5 > /dev/null 2>&1
                    chmod +x "$FILE_PATH"
                fi
                export TF_CLI_CONFIG_FILE="$DIR/stacks/provider/local/aws-objstore/.terraformrc"
                (cd $DIR/stacks/provider/local/aws-objstore; $DIR/bin/terraform init;
                $DIR/bin/terraform apply -auto-approve) > $DIR/logs/log.image
                RETVAL=$?
                if test ${RETVAL} -ne 0 ; then
                    echo "S3 bucket creation failed..." >&2
                    exit 3
                fi
                if [[ "$AWS_ACCESS_KEY_ID" == "" || "$AWS_SECRET_ACCESS_KEY" == "" ]]; then
                    echo "Please make sure that you have given current EC2 instance the AmazonS3FullAccess to the ${bucket_name} bucket" >&2
                    echo "Else follow below steps:" >&2
                    cat <<EOJ
1. IAM -> Roles -> createRole
2. Choose AWS Service , and then choose EC2 ,
3. Create an inline policy which has the permission to access s3 bucket and create an role
4. Then attach Instance profile (Selected instance -> click action -> under security -> Modify  IAM role and select the created IAM role)

example policy:
{
   "Version": "2012-10-17",
   "Statement": [
       {
           "Sid": "AmazonS3FullAccess",
           "Effect": "Allow",
           "Action": [
               "s3:*"
           ],
           "Resource": [
               "arn:aws:s3:::${bucket_name}",
               "arn:aws:s3:::${bucket_name}/*"
           ]
       }
   ]
}

once IAM role is attached to the instance, verify using : aws sts get-caller-identity

EOJ
                    unset AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY AWS_SESSION_TOKEN
                else
                    export ACCESS_KEY=${AWS_ACCESS_KEY_ID}
                    export SECRET_KEY=${AWS_SECRET_ACCESS_KEY}
                fi
                juicefs format --storage s3 --bucket https://${bucket_name}.s3.${AWS_DEFAULT_REGION}.amazonaws.com redis://`hostname`:${redis_port}/1 ${bucket_name}jfs &> $DIR/logs/log.image
                RET=$?
                if test $RET -ne 0 ; then
                    echo "Object store creation failed... Check $DIR/logs/log.image" >&2
                    exit 3
                fi
                ;;
            azure)
                export TF_VAR_resource_group=${ARM_RESOURCE_GROUP}
                echo "LOG FILE :: $DIR/logs/log.image" >&2
                # Creating a .terraformrc configuration file to define custom provider installation settings.
                cat << END > $DIR/stacks/provider/local/azure-objstore/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/stacks/provider/local/azure-objstore/plugins"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
                FILE_PATH="$DIR/stacks/provider/local/azure-objstore/plugins/registry.terraform.io/hashicorp/azurerm/4.16.0/linux_amd64/terraform-provider-azurerm_v4.16.0_x5"
                PLUGIN_DIR_PATH=$(dirname "$FILE_PATH")
                mkdir -p "$PLUGIN_DIR_PATH"
                if [ ! -f "$FILE_PATH" ]; then
                    curl -s -o "$FILE_PATH" \
                    https://s3.us-west-1.amazonaws.com/spillbox.io-internal/plugins/registry.terraform.io/hashicorp/azurerm/4.16.0/linux_amd64/terraform-provider-azurerm_v4.16.0_x5  > /dev/null 2>&1
                    chmod +x "$FILE_PATH"
                fi
                export TF_CLI_CONFIG_FILE="$DIR/stacks/provider/local/azure-objstore/.terraformrc"
                (cd $DIR/stacks/provider/local/azure-objstore; $DIR/bin/terraform init;
                $DIR/bin/terraform apply -auto-approve) > $DIR/logs/log.image
                RETVAL=$?
                if test ${RETVAL} -ne 0 ; then
                    echo "Azure storage account creation failed..." >&2
                    exit 3
                fi
                export SECRET_KEY=$(cd $DIR/stacks/provider/local/azure-objstore; $DIR/bin/terraform output -raw storage_account_primary_access_key)
                juicefs format --storage wasb --bucket https://mycontainer.blob.core.windows.net --access-key ${bucket_name} --secret-key ${SECRET_KEY} redis://`hostname`:${redis_port}/1 myjfs  &> $DIR/logs/log.image
                RET=$?
                if test $RET -ne 0 ; then
                    echo "Object store creation failed... Check $DIR/logs/log.image" >&2
                    exit 3
                fi
                ;;
            google)
                export TF_VAR_google_credentials_file=${GOOGLE_APPLICATION_CREDENTIALS}
                export TF_VAR_project_id=${GOOGLE_PROJECT}
                export TF_VAR_region=${GOOGLE_REGION}
                echo "LOG FILE :: $DIR/logs/log.image" >&2
                # Creating a .terraformrc configuration file to define custom provider installation settings.
                cat << END > $DIR/stacks/provider/local/google-objstore/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/stacks/provider/local/google-objstore/plugins"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
                FILE_PATH="$DIR/stacks/provider/local/google-objstore/plugins/registry.terraform.io/hashicorp/google/6.16.0/linux_amd64/terraform-provider-google_v6.16.0_x5"
                PLUGIN_DIR_PATH=$(dirname "$FILE_PATH")
                mkdir -p "$PLUGIN_DIR_PATH"
                if [ ! -f "$FILE_PATH" ]; then
                    curl -s -o "$FILE_PATH" \
                    https://s3.us-west-1.amazonaws.com/spillbox.io-internal/plugins/registry.terraform.io/hashicorp/google/6.16.0/linux_amd64/terraform-provider-google_v6.16.0_x5  > /dev/null 2>&1
                    chmod +x "$FILE_PATH"
                fi
                export TF_CLI_CONFIG_FILE="$DIR/stacks/provider/local/google-objstore/.terraformrc"
                (cd $DIR/stacks/provider/local/google-objstore; $DIR/bin/terraform init;
                $DIR/bin/terraform apply -auto-approve) > $DIR/logs/log.image
                RETVAL=$?
                if test ${RETVAL} -ne 0 ; then
                    echo "Google cloud storage creation failed..." >&2
                    exit 3
                fi
                juicefs format --storage gs --bucket ${bucket_name}.${region} redis://`hostname`:${redis_port}/1 myjfs  &> $DIR/logs/log.image
                RET=$?
                if test $RET -ne 0 ; then
                    echo "Object store creation failed... Check $DIR/logs/log.image" >&2
                    exit 3
                fi
                ;;
            *)
                echo "Unknown command $1" >&2
                usage;
                exit 3
        esac
        echo "Success..." >&2
        echo "For mounting juice fs on host, Run : juicefs mount -d redis://`hostname`:${redis_port}/1  <mount point>" >&2
        echo ""
        echo ""
        echo "Please run below command daily on this host to delete old checkpoint images" >&2
        echo "$DIR/objstore.sh $1 prune"
        echo ""
        echo ""
        echo "Suggested crontab entry \"0 0 * * * $DIR/objstore.sh $1 prune\""
        ;;
    delete)
        docker rm -f sbox-$redis_port-jfs  &>  $DIR/logs/log.image_destroy
        case "$1" in
            aws)
                export TF_VAR_region=${AWS_DEFAULT_REGION}
                echo "LOG FILE :: $DIR/logs/log.image_destroy" >&2
                (cd $DIR/stacks/provider/local/aws-objstore; $DIR/bin/terraform destroy -auto-approve) > $DIR/logs/log.image_destroy
                RET=$?
                if test $RET -ne 0 ; then
                    echo "s3 bucket deletion failed..." >&2
                    exit 3
                fi
                echo "Success..." >&2
                ;;
            azure)
                export TF_VAR_resource_group=${ARM_RESOURCE_GROUP}
                echo "LOG FILE :: $DIR/logs/log.image_destroy" >&2
                (cd $DIR/stacks/provider/local/azure-objstore; $DIR/bin/terraform destroy -auto-approve) > $DIR/logs/log.image_destroy
                RET=$?
                if test $RET -ne 0 ; then
                    echo "Azure storage account deletion failed..." >&2
                    exit 3
                fi
                echo "Success..." >&2
                ;;
            google)
                export TF_VAR_google_credentials_file=${GOOGLE_APPLICATION_CREDENTIALS}
                export TF_VAR_project_id=${GOOGLE_PROJECT}
                export TF_VAR_region=${GOOGLE_REGION}
                echo "LOG FILE :: $DIR/logs/log.image_destroy" >&2
                (cd $DIR/stacks/provider/local/google-objstore; $DIR/bin/terraform destroy -auto-approve) > $DIR/logs/log.image_destroy
                RET=$?
                if test $RET -ne 0 ; then
                    echo "Google cloud storage deletion failed..." >&2
                    exit 3
                fi
                echo "Success..." >&2
                ;;
            *)
                echo "Unknown command $1" >&2
                usage;
                exit 3
        esac
        ;;
    *)
        echo "Unknown command $2" >&2
        usage;
        exit 3
esac
