#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

# Wrapper Script for Executing Terraform
#

usage()
{
    echo "Usage : $0 [-kill] [-p <portnumber>] <tag>" >&2
    echo "	Default Port is 8080 " >&2
}

umask 022

export PATH="/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/usr/games:/usr/local/games:.:$PATH"
host=$(/bin/hostname -I | awk '{print $1}')
DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

if test "$#" -lt 1; then
    usage;
    exit 3
fi

timesynced=$(timedatectl |& grep "synchronized:" | awk -F":" '{print $2}' | sed 's/ //g')
if [ "$timesynced" == "no" ]; then
    echo "System Clock not synchronized." >&2
    exit 3
fi

maxproc=$(ulimit -Hu)
if [[ "${maxproc}" == "unlimited" ]] || [[ ${maxproc} -gt 65536 ]]; then
	maxproc=65536
fi
ulimit -u $maxproc
#maxfd=$(ulimit -Hn)
#if test $maxfd  -lt 65536  ; then
#    echo "Please increase user's max open file descriptor limit to 65536 or more for user $USER" >&2
#    if [ "${SPILLBOX_CONTAINER}" != "host" ] ; then
#        exit 1
#    fi
#fi
#ulimit -n $maxfd

SER_PORT=8080
if [ "$SPILLBOX_VNCPORT" != "" ]; then
    VNCPORT=$SPILLBOX_VNCPORT
else
    VNCPORT=8888
fi

case "$1" in 
-p)
	if test "$#" -ne 3; then
                usage;
                exit 3
        fi	
	SER_PORT=$2
	SPILLBOX_SITE=$3
	;;
-kill)
	if test "$#" -ne 2; then
                usage;
                exit 3
        fi
	SPILLBOX_SITE=$2
	;;
-*)
	usage;
	exit 0
	;;
*)	
	if test "$#" -ne 1; then
		usage;
		exit 3
	fi
	SPILLBOX_SITE=$1
	;;
esac

if ! [[ $SER_PORT =~ ^[0-9]+$ && $SPILLBOX_SITE == [A-Za-z0-9]* ]]; then
		usage;
                exit 3
fi
	
user=$(id -u -n)
export TF_VAR_vpn_container=true
export SPILLBOX_SITE
LOGDIR="$HOME"
WAN_THROUGHPUT_SIZE=10
OBJECT_STORE=false
if [ -f  $DIR/etc/spillbox_config ]; then
    tmp=$(grep ^workdir $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "$tmp" != "" ]; then
        LOGDIR="$tmp"
    else
        echo "workdir not set in $DIR/etc/spillbox_config" >&2
        exit 3
    fi
    tmp=$(grep wan_throughput_size $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "$tmp" != "" ]; then
        if test $tmp -gt 0; then
            if test $tmp -lt 4097; then
                WAN_THROUGHPUT_SIZE="$tmp"
            fi
        fi
    fi
    tmp=$(grep user_setup_file $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "$tmp" != "" ]; then
        if [ -f  $tmp ]; then
            . $tmp
        else
            echo "missing user_setup_file file $tmp" >&2
        fi
    fi
    tmp=$(grep job_manager $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "$tmp" == "uge" ]; then
        tmp=$(grep job_submit_command $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//')
	notify=$(echo "$tmp" | grep '\-notify')
	ret=$?
        if [ $ret -gt 0 ]; then
	    echo
            echo "WARN: Please specify -notify as qsub option in job_submit_command of $DIR/etc/spillbox_config file, for proper cleanup and signal handling."
	    echo
        fi
    elif [ "$tmp" == "slurm" ] ; then
        tmp=$(grep job_kill_command $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//')
        signal=$(echo "$tmp" | grep '\-f')
        ret=$?
        if [ $ret -gt 0 ]; then
            echo
                echo "WARN: Please specify -f as scancel option in job_kill_command of $DIR/etc/spillbox_config file, for proper cleanup and signal handling."
            echo
        fi
    fi
    tmp=$(grep ^checkpoint_objstore $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "$tmp" == "true" ]; then
        OBJECT_STORE=true
    fi
    tmp=$(grep ^spld_nfs_setup $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
    if [ "$tmp" == "true" ]; then
        export SPILLBOX_ULICENSE_CHECK=1
    fi
else
    echo "File missing : $DIR/etc/spillbox_config" >&2
    exit 3
fi

WDIR="$LOGDIR/.spillbox/sites/local/$SPILLBOX_SITE/"
# Note: if u remove the last slash in the WDIR variable, please make sure that u remove one (/) in -kill block and the following below lines
portlog="$WDIR/portlog"
log="$WDIR/log"
EXE="spld.server"

case "$1" in
-kill)
    if pgrep -f  "${SPILLBOX_SITE}//${EXE}.${user}.${SPILLBOX_SITE}" >/dev/null
    then
        PID=`pgrep -f  "${SPILLBOX_SITE}//${EXE}.${user}.${SPILLBOX_SITE}"`
        echo "killing ${EXE}.${user}.${SPILLBOX_SITE} pid $PID"
        pkill -9 -f  ${SPILLBOX_SITE}//${EXE}.${user}.${SPILLBOX_SITE}
    else
        host=$(hostname)
        echo "No server running on host $host. Please check if you are on correct host."
    fi
    /bin/rm -f $WDIR/new.crt  > /dev/null 
    /bin/rm -f $WDIR/new.key  > /dev/null 
    /bin/rm -f $WDIR/log      > /dev/null
    /bin/rm -f $WDIR/portlog  > /dev/null
    /bin/rm -f $WDIR/${EXE}.${user}.${SPILLBOX_SITE} > /dev/null
    /bin/rm -f $WDIR/spld.toml  > /dev/null
    exit 0
    ;;
 -p)
    ;;	
 -*)
    ;;
 *)
    ;;
esac 

if pgrep -f "${SPILLBOX_SITE}//${EXE}.${user}.${SPILLBOX_SITE}" >/dev/null
then
    PID=`pgrep -f   "${SPILLBOX_SITE}//${EXE}.${user}.${SPILLBOX_SITE}"`
    echo "killing ${EXE}.${user}.${SPILLBOX_SITE} pid $PID"
    pkill -9 -f  ${SPILLBOX_SITE}//${EXE}.${user}.${SPILLBOX_SITE}
    /bin/rm -f $WDIR/portlog  > /dev/null
    /bin/rm -f $WDIR/log  > /dev/null
fi
mkdir -p $WDIR
chmod 755 $WDIR

/bin/rm -f $WDIR/new.crt  > /dev/null 
/bin/rm -f $WDIR/new.key  > /dev/null 
if [ -x $DIR/stacks/modules/docker-nfs/install/splcert ]; then
    $DIR/stacks/modules/docker-nfs/install/splcert cert -cn "My CA" -c US -o Spillbox -d $((365 * 24))h -out $WDIR/new -ip $host  >& /dev/null
else
    echo $DIR/stacks/modules/docker-nfs/install/splcert not available >&2
    exit 3
fi

cat > $WDIR/spld.toml <<EOF

[server]

Url = "0.0.0.0" 
debugurl = "0.0.0.0" 
debugport = $VNCPORT

port = $SER_PORT 
cli  = false 
VPN = true
search = true
log = "$portlog" 
Max_ch_cap = $WAN_THROUGHPUT_SIZE

EOF

servermap=$(grep ^servermap $DIR/etc/spillbox_config | cut -d= -f2- | sed -e 's/"//g')
if [ "$servermap" != "" ]; then
    echo "[server.servermap]" >> $WDIR/spld.toml
    IFS=',' read -ra pairs <<< "$servermap"

    # Loop over each pair and convert to TOML key = "value"
    for pair in "${pairs[@]}"; do
        key="${pair%%=*}"       # before first '='
        value="${pair#*=}"      # after first '='
        echo "$key = \"$value\"" >> $WDIR/spld.toml
    done
fi

cat >> $WDIR/spld.toml <<EOF

[infra]

certdir = "$WDIR"
workload = true
control = true
data = true
root = "$WDIR/terraform"
cloud = "local" 
siteid = "$SPILLBOX_SITE" 

[clients]

EOF

for func in $(declare -Fx | awk '{print $3}'); do
  unset -f "$func"
done

sleep 5
mkdir -p  $WDIR/terraform
rsync -az --exclude=vars/ --exclude=logs/ --exclude=junk/ $DIR/ $WDIR/terraform/
INSTALL="$WDIR/terraform/stacks/provider/local/splserver/install"
export TF_CLI_CONFIG_FILE="$DIR/stacks/provider/local/image/.terraformrc"
( cd $WDIR/terraform/stacks/provider/local/splfs/ ; ../../../../bin/terraform init > /dev/null)
( cd $WDIR/terraform/stacks/provider/local/spillbox-cluster/ ; ../../../../bin/terraform init  > /dev/null) 
( cd $WDIR/terraform/stacks/provider/local/compute/ ; ../../../../bin/terraform init  > /dev/null)
( cd $WDIR/terraform/stacks/modules/docker-compute/ ; ../../../bin/terraform init  > /dev/null)
( cd $WDIR/terraform/stacks/modules/docker-nfs/ ; ../../../bin/terraform init  > /dev/null)
(cd $INSTALL; /bin/rm -f $WDIR/${EXE}.${user}.${SPILLBOX_SITE}; ln -s  ${INSTALL}/spld $WDIR/${EXE}.${user}.${SPILLBOX_SITE}; nohup $WDIR/${EXE}.${user}.${SPILLBOX_SITE} --config-file  $WDIR/spld.toml >& $WDIR/log &)
sleep 30

if [ ! -f "$portlog" ]; then
    echo "server failed" >&2 
    cat $log >&2
else
    port=$(cat $portlog)
    if [[ $1 == "-p" ]] && [[ "$port" -ne "$SER_PORT" ]]; then
	    	echo " Port number $SER_PORT is not available, Taking next available port... "
		echo ""
    fi
    echo "Please set followings environment variables for clients:"
    echo "export SPILLBOX_SERVER=$host:$port"
    echo "export SPILLBOX_VPN=1"
fi

if [ "$OBJECT_STORE" == "true" ]; then
    echo "- Run the script To create/delete objectstore fs : $DIR/objstore.sh <cloud> <create|delete|prune>"
fi
setup_bapp=$(grep ^setup_bapp $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
if [ " $setup_bapp" == " true" ] ; then
    export SPILLBOX_SERVER="$host:$port"
    ($DIR/bin/lsf_bapp_setup.sh)
    export RETVAL=$?
    if test ${RETVAL} -ne 0 ; then
        echo "LSF bapp setup failed" >&2
        exit $RETVAL
    fi
fi
nc_setup_bapp=$(grep ^nc_setup_bapp $DIR/etc/spillbox_config | grep -v ^# | awk -F= '{print $2}' | sed -e 's/^"//' -e 's/"$//' | sed 's/ //g')
if [ " $nc_setup_bapp" == " true" ] ; then
    ($DIR/bin/nc_docker_setup.sh)
    export RETVAL=$?
    if test ${RETVAL} -ne 0 ; then
        echo "NC docker setup failed" >&2
        exit $RETVAL
    fi
fi
