#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

# Wrapper Script for Executing Terraform
#

usage()
{
	echo "Usage : $0 build [update]|destroy|output" >&2
}

function docker_version_check () {
	docker_version=`docker --version 2>&1`
	export RETVAL=$?
	if test $RETVAL -gt 0 ; then
		echo "[ERROR] : docker setup not available, please make sure docker setup is available" >&2
		echo "$docker_version : $RETVAL" >&2
		exit 3
	fi
	tarrVer=(${docker_version// / })
	arrVer=(${tarrVer[2]//./ })
	if [[ ${arrVer[0]} -eq 20 && ${arrVer[1]} -ge 10 ]] || 
	   [[ ${arrVer[0]} -gt 20 ]] ; then
		echo "Recommended Docker version satisfied >=20.10"
	else
		echo "Recommended Docker version not satisfied"
		echo "Expected version : >=20.10"
		echo "Got : ${tarrVer[2]}"
		echo "Will continue with the image build, behavior might not be as desired"
	fi
}

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
export TERRAFORM_ROOT=$DIR

if test "$#" -lt 1; then
    usage >&2;
    exit 3
fi

spld_nfs_setup=false
spillbox_config_file=$(realpath $DIR/etc/spillbox_config)
if [ -f  $spillbox_config_file ]; then
    . $spillbox_config_file
else
    echo missing config file : $spillbox_config_file >&2
    exit 3
fi

if [ "$container" == "" ]; then 
    container="docker"
fi

if [[ "$spld_nfs_setup" == "true" ]]; then
    if [[ -z "$job_manager" || -z "$job_submit_command" || -z "$job_kill_command" ]]; then
        echo -e "\033[1;31mError: If spld_nfs_setup=true \"job_manager, job_submit_command, and job_kill_command\" must be set in \033[1;33m$spillbox_config_file\033[0m" >&2
        exit 1
    fi
fi


if [[ "$spld_nfs_setup" == "true" && "$container" == "docker" ]]; then
    echo "If spld_nfs_setup=true in $spillbox_config_file does not support container=$container"
    exit 1
fi
export TF_VAR_container=$container
export TF_VAR_runtime_container=$runtime_container
export TF_VAR_rootless_docker=$rootless_docker
export TF_VAR_exclude_mount_paths=$exclude_mount_paths
export TF_VAR_client_hosts=$client_hosts
export TF_VAR_spld_nfs_setup=$spld_nfs_setup
export TF_VAR_update_server_base_image=false

if [ "$runtime_container" == "" ]; then 
    if [ "$container" == "host" ]; then 
        runtime_container="singularity"
    else
        runtime_container=$container
    fi
fi

if [ "$runtime_container" == "host" ]; then 
    echo "runtime_container cannot be $runtime_container" >&2
    exit 3
fi

if [[ "$container" == "singularity" || $container == "host" ]]; then
    disable_server_image=true
    export TF_VAR_disable_server_image=true
    if [[ "$singularity_base_image" == "" && "$runtime_container" == "singularity" ]]; then
        echo "Please provide your singularity_base_image in $spillbox_config_file" >&2
        exit 1
    fi
    disable_registry_server=false
    export TF_VAR_disable_registry_server=false
fi

case "$1" in
    build|destroy)
        if [ "$container" == "docker" ]; then
            gc=$(id `whoami` | grep docker | wc -l)
            if [ "$gc" != "1" ] ; then
                user=$(whoami)
                if [ "$user" != "root" ] ; then
			echo "You($user) are not in docker unix group. Please request your sysadmin to add in docker unix group." >&2
                    exit 3
                fi
            fi
        fi
        ;;
    output|check_bapp)
        ;;
     *)
        usage;
        exit 3
        ;;
esac

if [ "$1" == "build" ] ; then
    if [[ "$container" == "docker" || "$runtime_container" = "docker" ]]; then
        if [ "$2" == "update" ]; then
            export TF_VAR_update_server_base_image=true
        fi
        #check for docker version
        docker_version_check
    fi
    mkdir -p $DIR/logs
    docker_mount_path=${DIR}/../sites/local/customers/mount_paths/
    mkdir -p $docker_mount_path
    chmod 777 $docker_mount_path
    mkdir -p $DIR/junk
    chmod 777 $DIR/junk
fi

if [ "$disable_server_image" == "" ]; then 
    export TF_VAR_disable_server_image=false
    if [ "$server_base_image" == "" ]; then 
        export TF_VAR_server_base_image=centos:7
    else
        #checking if server_base_image contains :
        if [[ "$server_base_image" == *:* ]]; then
            export TF_VAR_server_base_image_tag_name="${server_base_image##*:}"
        else
            export TF_VAR_server_base_image_tag_name=""
        fi
        export TF_VAR_server_base_image="${server_base_image%%:*}"
    fi
else
    export TF_VAR_disable_server_image=$disable_server_image
    if [[ "$server_base_image" == "" && "$container" == "docker" ]]; then
            echo "Please provide your server_base_image in $spillbox_config_file" >&2
            exit 1
    else
        #checking if server_base_image contains :
        if [[ "$server_base_image" == *:* ]]; then
            export TF_VAR_server_base_image_tag_name="${server_base_image##*:}"
        else
            export TF_VAR_server_base_image_tag_name=""
        fi
        export TF_VAR_server_base_image="${server_base_image%%:*}"
    fi
fi

if [ "$disable_registry_server" == "" ]; then 
    export TF_VAR_disable_registry_server=false
else
    export TF_VAR_disable_registry_server=$disable_registry_server
    if [ "$disable_registry_server" == "true" ]; then 
        if [ "$registry_server" == "" ]; then 
            echo "Please provide your registry_server in $spillbox_config_file" >&2
            exit 1
        else
            export TF_VAR_registry_server=$registry_server
        fi
    fi
fi

export TF_VAR_vm_user=$(id -u -n)

if [[ "$container" = "docker" || "$runtime_container" = "docker" ]]; then 
    if [ "$docker_registry_port" != "" ]; then
        export TF_VAR_docker_registry_port=$docker_registry_port
    else
        export TF_VAR_docker_registry_port=5000
    fi
else
        export TF_VAR_docker_registry_port=""
fi
if [ "$network_interface" == "" ]; then 
    network_interface=$(ip -o -4 addr show | awk '{print  $2}' | grep ^e[m-n])
fi
export TF_VAR_network_interface=$network_interface 
export TF_VAR_docker_rm=false
if [[ "$docker_registry_hostname" == "true" ]]; then
	 export TF_VAR_docker_host=$(hostname -f)
 else
	  export TF_VAR_docker_host=$(hostname -I | awk '{print $1}')
  fi
export TF_VAR_docker_port=2375            

if [[ "$container" = "docker" || "$runtime_container" = "docker" ]]; then 
    if [ "$docker_path" == "" ]; then 
        docker_path=$(which docker)        
    fi
    if [ "$docker_path" == "" ]; then 
        echo "Please provide docker_path in $spillbox_config_file file"  >&2
        exit 1
    fi
    export TF_VAR_docker_path=$docker_path
else
    export TF_VAR_docker_path==""
fi

if [[ "$container" = "singularity" || "$runtime_container" = "singularity" ]]; then 
    if [ "$singularity_path" == "" ]; then 
        singularity_path=$(which singularity)        
    fi
    if [ "$singularity_path" == "" ]; then 
        echo "Please provide singularity_path in $spillbox_config_file file"  >&2
        exit 1
    fi
    export TF_VAR_singularity_path=$singularity_path
else
    export TF_VAR_singularity_path=""
fi
if [ "$boostmode" == "true" ]; then
    export TF_VAR_boostmode=true
else
    export TF_VAR_boostmode=false
fi
if [[ "$1" == "build" && "$container" == "docker" ]]; then
    if [[ -n "$TF_VAR_server_base_image_tag_name" ]]; then
        #Inspecting image with tag, only if tag name is given
        image_local="$TF_VAR_server_base_image:$TF_VAR_server_base_image_tag_name"
        image_registry="${TF_VAR_docker_host}:${TF_VAR_docker_registry_port}/$TF_VAR_server_base_image:$TF_VAR_server_base_image_tag_name"
        if ! $docker_path image inspect "$image_local" 1>/dev/null 2>&1; then
            if ! $docker_path image inspect "$image_registry" 1>/dev/null 2>&1; then
                echo "Error: Could not find image locally or in registry." >&2
                exit 1
            fi
        fi
    else
        $docker_path image inspect "$TF_VAR_server_base_image" 1>/dev/null
        ret=$?
        if [ $ret -ne 0 ]; then
                exit 1
        fi
   fi
fi

case "$1" in
    build)
    # Creating a .terraformrc configuration file to define custom provider installation settings.
    cat << END > $DIR/stacks/provider/local/image/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/plugin"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
        export TF_CLI_CONFIG_FILE="$DIR/stacks/provider/local/image/.terraformrc"
        (cd $DIR/stacks/provider/local/image; $DIR/bin/terraform init; $DIR/bin/terraform init; \
            $DIR/bin/terraform init; $DIR/bin/terraform apply -auto-approve) 2>&1 > $DIR/logs/log.image | tee -a $DIR/logs/log.image
        RETVAL=$? 
        if test ${RETVAL} -ne 0 ; then
            echo "build failed. Please check logfile $DIR/logs/log.image." >&2
            exit $RETVAL
        fi
        (cd $DIR/stacks/provider/local/image; $DIR/bin/terraform output)
        RETVAL=$? 
        if test ${RETVAL} -ne 0 ; then
            echo "output failed" >&2
            exit $RETVAL
        fi
        if [[ "$container" == "docker" || "$runtime_container" = "docker" ]]; then
            registry_server=$((cd $DIR/stacks/provider/local/image; $DIR/bin/terraform output) | grep registry_server | awk -F= '{print $2}')
            if [ "$registry_server" != "" ]; then
                echo "Please add below entries to /etc/docker/daemon.json on all of clients:"
                echo "{"
                echo "  \"hosts\": [\"unix:///var/run/docker.sock\",\"tcp://0.0.0.0:2375\"],"
                echo "  \"insecure-registries\": [$registry_server]"
                echo "}"
            fi
        fi
        ;;
    destroy)
        (cd $DIR/stacks/provider/local/image; $DIR/bin/terraform destroy -auto-approve) 2>&1 > $DIR/logs/log.image_destroy | tee -a $DIR/logs/log.image_destroy
        RETVAL=$? 
        if test ${RETVAL} -ne 0 ; then
            echo "destroy failed. Please check logfile $DIR/logs/log.image_destroy." >&2
            exit $RETVAL
        fi
        ;;
    output)
        (cd $DIR/stacks/provider/local/image; $DIR/bin/terraform output -json)
        RETVAL=$? 
        if test ${RETVAL} -ne 0 ; then
            echo "output failed" >&2
            exit $RETVAL
        fi
        ;;
    *)
        usage;
        exit 3
        ;;
esac

exit 0
