# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

terraform {
  required_providers {
    docker = {
      source = "kreuzwerker/docker"
    }
  }
}

variable "customer" {
  description = "customer or cluster name"
}

variable "subnet" {
  description = "Local network CIDR"
}

variable "ip_range" {
  description = "Spillbox Docker network IP range"
}

variable "extra_address" {
  description = "Spillbox Docker network extra_address"
}

variable "gateway" {
  description = "Spillbox Docker network gateway"
}

variable "network_interface" {
  description = "network interface name"
}

variable "docker_host" {
  description = "Docker server hostname"
}

variable "docker_port" {
  description = "Network port used by docker server"
}

variable "docker_driver" {
  description = "Docker network driver type"
}

variable "jq_path" {
  description = "Path executable jq - JSON parser"
}

variable "docker_registry_port" {
  description = "Docker registry port"
}

resource "null_resource" "swarm" {
  count     = var.docker_driver == "macvlan" ? 1 : var.docker_driver == "overlay"? 1 :0
  provisioner "local-exec" {
    command = "sudo ${path.module}/install/swarm_init.sh"
  }
}

resource "null_resource" "registry" {
  provisioner "local-exec" {
    command = "${path.module}/install/install_registry.sh ${abspath(path.module)} ${var.docker_registry_port}"
  }
  provisioner "local-exec" {
    when    = destroy
    command = "docker container stop spillbox_registry && docker container rm -v spillbox_registry"
  }
}


data "external" "docker_worker_join_token" {
  depends_on = [null_resource.swarm]
  program = ["sudo", "${path.module}/install/get_token.sh", "${var.docker_host}", "${var.docker_port}", "${var.docker_driver}", "${var.jq_path}" ]
}

resource "docker_network" "private_network" {
  count           = var.docker_driver == "macvlan" ? 1 : var.docker_driver == "overlay"? 1 :0
  depends_on      = [ null_resource.swarm ]
  name            = var.customer
  check_duplicate = true

  ipam_config {
    subnet   = var.subnet
    ip_range = var.ip_range
    gateway  = var.gateway
    aux_address = { host = var.extra_address }
  }

  driver = var.docker_driver

  options = {
    parent = var.network_interface
    #scope = var.docker_driver == "overlay" ? "swarm" : var.docker_driver == "macvlan" ? "swarm" : "local" 
  }
  attachable = var.docker_driver == "overlay" ? true : var.docker_driver == "macvlan" ? true : false
}

output "network" {
  value = [docker_network.private_network.*.name][0]
}

output "docker_worker_join_token" {
  value = var.docker_driver == "macvlan" ? lookup(data.external.docker_worker_join_token.result, "token") : ""
}

