#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

#echo $@

#echo transport_server: $1 transport_port:$2 cache_dir:$3 customer:$4 mount_paths:$5 cloud_user:$6 cloud_group:$7 cloud_uid:$8 cloud_gid:$9 vm_user:${10} cloud_provider:${11} admin_host:${12} cache_server:${13} nameserver:${14} nfs_portmin:${15} registry_server:${16} container:${17} cache_option:${18} container_name: ${19} singularity_path:${20} server_image:${21} container_mount_path:${22} docker_path:${23} remote_repo_path: ${24} terraform_root: ${25} patchelf_path: ${26} max_ch_cap: ${27} rootless_docker: ${28} splfs_home: ${29} archive: ${30} disable_global_cache: ${31} global_cache_server: ${32} global_cache: ${33} nfs_model: ${34} hybrid_mount_paths: ${35} hybrid_mount_paths_exclude: ${36} boost_mode: ${37} boost_server_cache_size: ${38} boost_client_cache_dir: ${39} hybrid_option: ${40} wan_timeout: ${41} ws_pair: ${42} global_cache_pair: ${43} spld_nfs_setup: ${44} home: ${45} workdir: ${46} container_path_remap: ${47}

set -e

check_rpc() {
    IFS=',' read -ra SERVICE <<< "rpcbind,statd"
    for p in "${SERVICE[@]}"; do 
        if pgrep -f "$p" > /dev/null
        then
            true
        else
            echo "ERROR: $p is not running on this machine. Please contact your sysadmin."  >&2
            exit 10
        fi
    done
}

# environment variables
: ${SPILLBOX_TRANSPORT_SERVER:=$1}
: ${SPILLBOX_TRANSPORT_PORT:=$2}
: ${SPILLBOX_CACHE_DIR:=$3}
: ${SPILLBOX_CUSTOMER:=$4}
: ${SPILLBOX_MOUNT_PATHS:=$5}
: ${SPILLBOX_CLOUD_USER:=$6}
: ${SPILLBOX_CLOUD_GROUP:=$7}
: ${SPILLBOX_CLOUD_UID:=$8}
: ${SPILLBOX_CLOUD_GID:=$9}
: ${SPILLBOX_VM_USER:=${10}}
: ${SPILLBOX_CLOUD_PROVIDER:=${11}}
: ${SPILLBOX_ADMIN_HOST:=${12}}
: ${SPILLBOX_CACHE_SERVER:=${13}}
: ${SPILLBOX_NAMESERVER:=${14}}
: ${SPILLBOX_NFS_PORTMIN:=${15}}
: ${SPILLBOX_REGISTRY_SERVER:=${16}}
: ${SPILLBOX_CONTAINER:=${17}}
: ${SPILLBOX_CACHE_OPTION:=${18}}
: ${SPILLBOX_CONTAINER_NAME:=${19}}
: ${SPILLBOX_SINGULARITY_PATH:=${20}}
: ${SPILLBOX_SERVER_IMAGE:=${21}}
: ${SPILLBOX_CONTAINER_MOUNT_PATH:=${22}}
: ${SPILLBOX_DOCKER_PATH:=${23}}
: ${SPILLBOX_REMOTE_REPO_PATH:=${24}}
: ${SPILLBOX_TERRAFORM_ROOT:=${25}}
: ${SPILLBOX_PATCHELF_PATH:=${26}}
: ${SPILLBOX_MAX_CH_CAP:=${27}}
: ${SPILLBOX_ROOTLESS_DOCKER:=${28}}
: ${SPILLBOX_SPLFS_HOME:=${29}}
: ${SPILLBOX_ARCHIVE:=${30}}
: ${SPILLBOX_DISABLE_GLOBAL_CACHE:=${31}}
: ${SPILLBOX_GLOBAL_CACHE_SERVER:=${32}}
: ${SPILLBOX_GLOBAL_CACHE:=${33}}
: ${SPILLBOX_NFS_MODEL:=${34}}
: ${SPILLBOX_HYBRID_MOUNT_PATHS:=${35}}
: ${SPILLBOX_HYBRID_MOUNT_PATHS_EXCLUDE:=${36}}
: ${SPILLBOX_BOOSTMODE:=${37}}
: ${SPILLBOX_BOOST_SERVER_CACHE_SIZE:=${38}}
: ${SPILLBOX_BOOST_CLIENT_CACHE_DIR:=${39}}
: ${SPILLBOX_HYBRID_OPTION:=${40}}
: ${SPILLBOX_WAN_TIMEOUT:=${41}}
: ${SPILLBOX_WS_PAIR:=${42}}
: ${SPILLBOX_GLOBAL_CACHE_PAIR:=${43}}
: ${SPILLBOX_SPLD_NFS_SETUP:=${44}}
: ${SPILLBOX_HOME:=${45}}
: ${SPILLBOX_REMOTE_WORKDIR:=${46}}
: ${SPILLBOX_CONTAINER_PATH_REMAP:=${47}}

: ${EXPORT_ID:=1000}
: ${PROTOCOLS:=4}
: ${TRANSPORTS:="TCP"}
: ${ENABLE_UDP:=false}
: ${SEC_TYPE:="sys"}
: ${SQUASH_MODE:="No_Root_Squash"}
: ${GRACELESS:=true}

: ${WDIR:=$(realpath $(dirname $0))} 
: ${HOST_LOGDIR:="${WDIR}/spillbox_splfs"}
: ${GANESHA_CONFIG_HOST:=$HOST_LOGDIR/conf}
: ${GANESHA_LOGFILE:="/dev/stdout"}
: ${SPILLBOX_STORAGE:="PROXY"}
SETUP_SH="${HOST_LOGDIR}/setup.sh"

check_limit() {
    ulimit -s $(ulimit -Hs)
    CAN_I_RUN_SUDO=$(sudo -n uptime 2>&1|grep "load"|wc -l)
    nr_open=$(sysctl  fs.nr_open | awk -F"=" '{print $2}' | sed 's/ //g')
    if test $nr_open  -lt 1048576  ; then 
        if [[ $(($CAN_I_RUN_SUDO)) -gt 0 ]]; then
            sudo sysctl -w fs.nr_open=1048576
        else 
            echo "Please increase kernel fs.nr_open limit to 1048576 or more" >&2
            exit 20 
        fi
    fi
    file_max=$(sysctl  fs.file-max | awk -F"=" '{print $2}' | sed 's/ //g')
    if test $file_max  -lt 10240000  ; then 
        if [[ $(($CAN_I_RUN_SUDO)) -gt 0 ]]; then
            sudo sysctl -w fs.file-max=10240000
        else 
            echo "Please increase kernel fs.file-max limit to 10240000 or more. Please run:" >&2
            echo "sudo sysctl -w fs.file-max=10240000" >&2
            echo "Please add below entries to /etc/sysctl.conf" >&2
            echo "" >&2
            echo "" >&2
            fs.file-max=10240000
            echo "" >&2
            exit 30  
        fi
    fi
    #file
    maxfd=$(ulimit -Hn)
    if test $maxfd  -lt 1024000  ; then
        echo "" >&2
        echo "" >&2
        echo "Please increase user's max open file descriptor limit to 1024000 or more" >&2
        echo "Please add below entries to /etc/security/limits.conf" >&2
        echo "" >&2
        echo "" >&2
        echo "*     hard nofile 1024000" >&2
        echo "*     soft nofile 1024000" >&2
        echo "" >&2
    fi
    ulimit -n $maxfd
}

create_logdir() {
    mkdir -p $HOST_LOGDIR/LOG
    mkdir -p $HOST_LOGDIR/LOG/LSFLOG
    #chown ${SPILLBOX_VM_USER} $HOST_LOGDIR
    chmod -R 777 $HOST_LOGDIR/LOG || /bin/true
    /bin/rm -f $HOST_LOGDIR/LOG/conf.out;
    #GROUP=$(id -gn)
    #setfacl -dRm u:${SPILLBOX_VM_USER}:rwX,g:$GROUP:rwX $WDIR/LOG
}

check_nfs() {
    if [ "${SPILLBOX_CONTAINER}" == "singularity" ]; then
        SPILLBOX_STORAGE="VFS"
        return
    fi
    if [ "${SPILLBOX_CACHE_DIR}" == "/dev/shm" ]; then
        SPILLBOX_STORAGE="VFS"
        return
    fi
    if [ "${SPILLBOX_CACHE_DIR}" == "/spillbox_cachedir" ]; then
        SPILLBOX_STORAGE="VFS"
        return
    fi
    # this is only flow we support for now
    SPILLBOX_STORAGE="VFS"
}

bootstrap_config() {
    local global
    global=false
    if [ -f "${SPILLBOX_TERRAFORM_ROOT}/etc/spillbox_config_cache" ]; then
        customer=$(cat ${SPILLBOX_TERRAFORM_ROOT}/etc/spillbox_config_cache)
        if [ "$customer" != "$SPILLBOX_CUSTOMER" ] ; then
            global=true
            echo "Using cache from project $customer"
        fi
    fi
    
    echo "* Writing configuration"

    CONF_OUT=$SPILLBOX_CONTAINER_MOUNT_PATH/LOG/conf.out 
    LOGPATH=$SPILLBOX_CONTAINER_MOUNT_PATH/LOG/spillbox_file.log 
    if [ "$SPILLBOX_CONTAINER" == "singularity" ]; then
        CONF_OUT="/opt/spillbox/splfs/LOG/conf.out"
    fi

    Download_Enable=false
    Upload_Enable=false
    Hier_Enable=false
    Sync_Enable=false

    if [ "$SPILLBOX_GLOBAL_CACHE" = "false" ]; then
        if [ "${SPILLBOX_HYBRID_MOUNT_PATHS}" != "" ]; then
            if [ "${SPILLBOX_HYBRID_MOUNT_PATHS}" != "NONE" ]; then
                Hier_Enable=true
                if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-download"; then
                    Download_Enable=true
                fi
                if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-missing"; then
                    Upload_Enable=true
                fi
                if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-sync"; then
                    Sync_Enable=true
                fi
            fi
        fi
    else
        Hier_Enable=true
        Upload_Enable=true
        Sync_Enable=true
    fi

    cat <<END >${GANESHA_CONFIG_HOST}
NFS_CORE_PARAM {
    Enable_UDP = ${ENABLE_UDP};
    Cache_Autoport = $SPILLBOX_NFS_PORTMIN;
    Cache_dumppath = $CONF_OUT;
    #Dbus_Name_Prefix = $SPILLBOX_CUSTOMER;
    RPC_Max_Connections = 8000;
    RPC_Ioq_ThrdMax = 1600;
END
    if [[ "${SPILLBOX_CONTAINER}" == "host" && "${SPILLBOX_SPLFS_HOME}" == "" ]] ; then
        cat <<END >>${GANESHA_CONFIG_HOST}
    Plugins_Dir = ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/opt/spillbox/splfs/lib/ganesha;
END
    fi
        cat <<END >>${GANESHA_CONFIG_HOST}
}

MDCACHE
{
    Entries_HWMark = 800000;
}

NFSV4
{ 
    Graceless = ${GRACELESS};
    Only_Numeric_Owners = true;
END
    if [ "${SPILLBOX_CONTAINER}" == "host" ] ; then
        cat <<END >>${GANESHA_CONFIG_HOST}
    RecoveryRoot = ${HOST_LOGDIR}/LOG;
END
    fi
        cat <<END >>${GANESHA_CONFIG_HOST}
}

CACHE
{
    Cache_Dir = ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir;
    Sync_Time = latest;
    Local_UserId = ${SPILLBOX_CLOUD_UID};
    Local_GroupId = ${SPILLBOX_CLOUD_GID};
    Transport = WS;
    Bucket_Prefix = ${SPILLBOX_CUSTOMER};
    Cloud_Provider = ${SPILLBOX_CLOUD_PROVIDER};
    Archive = ${SPILLBOX_ARCHIVE};
    Download_Enable=${Download_Enable};
    Upload_Enable=${Upload_Enable};
    Hier_Enable=${Hier_Enable};
    Sync_Enable=${Sync_Enable};
    Splfs_Timeout=${SPILLBOX_WAN_TIMEOUT}; 
}

WS
{
    Srv_Addr = ${SPILLBOX_TRANSPORT_SERVER};
    WS_Port = ${SPILLBOX_TRANSPORT_PORT};
    WS_Pair = ${SPILLBOX_WS_PAIR};
END
    if [ "${global}" == "true" ] ; then
        if [ "${SPILLBOX_DISABLE_GLOBAL_CACHE}" == "false" ] ; then
            if [ "${SPILLBOX_GLOBAL_CACHE_SERVER}" == "" ] ; then
                echo "global_cache_server value not set" >&2
                exit 40
            fi 
            host=$(echo "$SPILLBOX_GLOBAL_CACHE_SERVER" | awk -F: '{print $1}') 
            port=$(echo "$SPILLBOX_GLOBAL_CACHE_SERVER" | awk -F: '{print $2}') 
            if [ "$port" == "" ] ; then
                echo "global_cache_server port value not set" >&2
                exit 50 
            fi 
            cat <<END >>${GANESHA_CONFIG_HOST}
    Srv_Global_Cache_Addr = $host;
    WS_Global_Cache_Port = $port;
    WS_Global_Cache_Pair = $SPILLBOX_GLOBAL_CACHE_PAIR;
END
        fi
    fi
            cat <<END >>${GANESHA_CONFIG_HOST}
    remote_repo_path = ${SPILLBOX_REMOTE_REPO_PATH};
    max_ch_cap = ${SPILLBOX_MAX_CH_CAP};
    VPN = true;
    logpath = ${LOGPATH};
}

CacheInode
{
    Entries_HWMark = 1000000;
    Chunks_HWMark = 1000000;
    Cache_Size = 326330;
    Use_Getattr_Directory_Invalidation = true; 
    Dir_Chunk = $dir_chunk;
}

LOG
{
    Components {
        #FSAL = FULL_DEBUG;
        FSAL = WARN;
    }
}

END

    mountpath=$(echo $SPILLBOX_MOUNT_PATHS | tr "," "\n")
    for apath in $mountpath
    do
        cat <<END >>${GANESHA_CONFIG_HOST}
EXPORT
{
    Export_Id = ${EXPORT_ID};
    Path = "${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/data$apath";
    Pseudo = "${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/data$apath";
    Access_type = RW;
    #Attr_Expiration_Time = 0;
    #Disable_ACL = true;
    Squash = ${SQUASH_MODE};
    Protocols = ${PROTOCOLS};
    FSAL {
        name = ${SPILLBOX_STORAGE};
END
        if [ "${SPILLBOX_STORAGE}" == "PROXY" ] ; then
            cat <<END >>${GANESHA_CONFIG_HOST}
            Srv_Addr = ${SPILLBOX_CACHE_SERVER};
            Use_Privileged_Client_Port = false;
            NFS_SendSize = 10485760;
            NFS_RecvSize = 10485760;
END
        fi

        cat <<END >>${GANESHA_CONFIG_HOST}
    }
}

END

        let EXPORT_ID=${EXPORT_ID}+1
    done
}

sleep 0.5

create_logdir
check_nfs

#echo "* Generated config"

#echo "* Starting Container"

yes | /bin/rm -f $SETUP_SH
touch $SETUP_SH
echo export SPILLBOX_CONTAINER_NAME=$SPILLBOX_CONTAINER_NAME >> $SETUP_SH
echo export SPILLBOX_CACHE_DIR=$SPILLBOX_CACHE_DIR >> $SETUP_SH
echo export SPILLBOX_CLOUD_UID=$SPILLBOX_CLOUD_UID >> $SETUP_SH
echo export SPILLBOX_CLOUD_GID=$SPILLBOX_CLOUD_GID >> $SETUP_SH
echo export SPILLBOX_CLOUD_USER=$SPILLBOX_CLOUD_USER >> $SETUP_SH
echo export SPILLBOX_CLOUD_GROUP=$SPILLBOX_CLOUD_GROUP >> $SETUP_SH
echo export SPILLBOX_NAMESERVER=$SPILLBOX_NAMESERVER >> $SETUP_SH
echo export SPILLBOX_CUSTOMER=$SPILLBOX_CUSTOMER >> $SETUP_SH
echo export SPILLBOX_CONTAINER_MOUNT_PATH=$SPILLBOX_CONTAINER_MOUNT_PATH >> $SETUP_SH
echo export SPILLBOX_REMOTE_REPO_PATH=$SPILLBOX_REMOTE_REPO_PATH >> $SETUP_SH
echo export SPILLBOX_CONTAINER=$SPILLBOX_CONTAINER >> $SETUP_SH
echo export SPILLBOX_MOUNT_PATHS=$SPILLBOX_MOUNT_PATHS >> $SETUP_SH
echo export SPILLBOX_TERRAFORM_ROOT=$SPILLBOX_TERRAFORM_ROOT >> $SETUP_SH
echo export SPILLBOX_STORAGE=$SPILLBOX_STORAGE >> $SETUP_SH
echo export SPILLBOX_PATCHELF_PATH=$SPILLBOX_PATCHELF_PATH >> $SETUP_SH
echo export SPILLBOX_MAX_CH_CAP=$SPILLBOX_MAX_CH_CAP >> $SETUP_SH
echo export SPILLBOX_ROOTLESS_DOCKER=$SPILLBOX_ROOTLESS_DOCKER >> $SETUP_SH
echo export SPILLBOX_SPLFS_HOME=$SPILLBOX_SPLFS_HOME >> $SETUP_SH
echo export SPILLBOX_HYBRID_MOUNT_PATHS=\"$SPILLBOX_HYBRID_MOUNT_PATHS\" >> $SETUP_SH
echo export SPILLBOX_HYBRID_MOUNT_PATHS_EXCLUDE=\"$SPILLBOX_HYBRID_MOUNT_PATHS_EXCLUDE\" >> $SETUP_SH
echo export SPILLBOX_GLOBAL_CACHE=\"$SPILLBOX_GLOBAL_CACHE\" >> $SETUP_SH
echo export SPILLBOX_GLOBAL_CACHE_SERVER=\"$SPILLBOX_GLOBAL_CACHE_SERVER\" >> $SETUP_SH
echo export SPILLBOX_DISABLE_GLOBAL_CACHE=\"$SPILLBOX_DISABLE_GLOBAL_CACHE\" >> $SETUP_SH
echo export SPILLBOX_BOOSTMODE=\"$SPILLBOX_BOOSTMODE\" >> $SETUP_SH
echo export SPILLBOX_BOOST_SERVER_CACHE_SIZE=\"$SPILLBOX_BOOST_SERVER_CACHE_SIZE\" >> $SETUP_SH
echo export SPILLBOX_BOOST_CLIENT_CACHE_DIR=\"$SPILLBOX_BOOST_CLIENT_CACHE_DIR\" >> $SETUP_SH
echo export SPILLBOX_HYBRID_OPTION=\"$SPILLBOX_HYBRID_OPTION\" >> $SETUP_SH
echo export SPILLBOX_HOME=\"$SPILLBOX_HOME\" >> $SETUP_SH
echo export SPILLBOX_SPLD_NFS_SETUP=\"$SPILLBOX_SPLD_NFS_SETUP\" >> $SETUP_SH
echo export SPILLBOX_REMOTE_WORKDIR=\"$SPILLBOX_REMOTE_WORKDIR\" >> $SETUP_SH
echo export SPILLBOX_CONTAINER_PATH_REMAP=\"$SPILLBOX_CONTAINER_PATH_REMAP\" >> $SETUP_SH

if [ "$SPILLBOX_CONTAINER" == "docker" ]; then
    (${SPILLBOX_DOCKER_PATH} rm -f $SPILLBOX_CONTAINER_NAME > /dev/null) || true
fi

BASE=$(realpath $(dirname $0))
spillbox_config_file=$(realpath $BASE/../../../../etc/spillbox_config)
spillbox_nfs_model_file=$(realpath $BASE/../../../../etc/spillbox_nfsModel_config)
export SPLAPP_BYPASS_EXE_WRAPPER=1
exe_wrapper=$(realpath $BASE/../../../../bin/exewrapper.sh)

if [ -f  $spillbox_config_file ]; then 
    . $spillbox_config_file
else
    echo missing config file : $spillbox_config_file >&2
    exit 50 
fi

if [ ! -f  $exe_wrapper ]; then 
    echo missing exe_wrapper file : $exe_wrapper >&2
    exit 60  
fi

if [ -f "$spillbox_nfs_model_file" ]; then
    . $spillbox_nfs_model_file
    if [ "$SPILLBOX_NFS_MODEL" != "" ]; then
	    get_model_config=$(grep "^$SPILLBOX_NFS_MODEL=" "$spillbox_nfs_model_file" | cut -d '=' -f 2-)
    else
	    get_model_config=$(grep "^$default=" "$spillbox_nfs_model_file" | cut -d '=' -f 2-)
    fi
    if [ -n "$get_model_config" ]; then
         get_model_config=$(echo "$get_model_config" | sed 's/^"//;s/"$//')
         echo "$get_model_config"
    else
         echo "Model not found: $SPILLBOX_NFS_MODEL"
    fi
else
    echo "File does not exist: $spillbox_nfs_model_file"
fi

if [ "$setup_ldap" == "true" ]; then
    LDAP_DIR="-v /var/lib/sss/pipes:/var/lib/sss/pipes:rw"
else
    LDAP_DIR=""
fi
if [ "$setup_nis" == "true" ]; then
    NIS_DIR="-v /etc/nsswitch.conf:/etc/nsswitch.conf:ro -v /etc/yp.conf:/etc/yp.conf:ro"
else
    NIS_DIR=""
fi

if [ "$dir_chunk" == "" ]; then
    dir_chunk=1024 
fi

bootstrap_config

if [ "$SPILLBOX_GLOBAL_CACHE" = "true" ]; then
    job_manager="" 
    job_submit_command=""
    job_kill_command=""
fi

if [ "$job_manager" != "" ]; then
    if [ "$job_submit_command" == "" ]; then
        echo "Please provide job_submit_command in $spillbox_config_file" >&2
        exit 70
    else
        #if [ "$SPILLBOX_NFS_MODEL" != "" ]; then
	if [ "$get_model_config" != "" ]; then
                echo "NFS Model map server config loaded with :: $get_model_config for the client config model:: $SPILLBOX_NFS_MODEL set"
                job_submit_command+=" $get_model_config"
                echo "Job submit command: $job_submit_command"
       fi
    fi
else
    if [ "$job_submit_command" != "" ]; then
        echo "Please provide job_manager in $spillbox_config_file" >&2
        exit 80  
    fi
fi
NFSJOB_NAME=""
case "$job_manager" in
    "lsf")
        NFSJOB_NAME="-J splfs-${SPILLBOX_CUSTOMER}"
        NFSJOB_OPTION=""
	;;
    "slurm")
        NFSJOB_NAME="-J splfs-${SPILLBOX_CUSTOMER}"
        NFSJOB_OPTION=""
	;;
    "nc")
        NFSJOB_NAME="-J splfs-${SPILLBOX_CUSTOMER}"
        NFSJOB_OPTION=""
	;;
    "uge")
        NFSJOB_NAME="-N splfs-${SPILLBOX_CUSTOMER}"
        NFSJOB_OPTION="-V"
	;;
esac
echo export SPILLBOX_JOB_MANAGER=$job_manager >> $SETUP_SH

/bin/rm -f $BASE/spillbox_splfs/job_output


###
#Creating a Setup file which stores all ENV spillbox environment which get uses on standalone setups
STANDALONE_SETUP=$(realpath $BASE/../../../../bin/standalone_setup.sh)
/bin/rm -rf $STANDALONE_SETUP
touch $STANDALONE_SETUP
echo export SPILLBOX_CLOUD_USER=\"$SPILLBOX_CLOUD_USER\" >> $STANDALONE_SETUP
echo export SPILLBOX_SITE=\"$SPILLBOX_SITE\" >> $STANDALONE_SETUP
#some variable are added install_volume.sh (ALL nfs mount info)
###

detach=""
restart=""
name=""

if [ "$SPILLBOX_CONTAINER" == "docker" ]; then
    nofile_unlimit="--ulimit nofile=1024000:1024000"
    sudocmd=""
    portmap="--net=host --init "
    RESOLV="/etc/resolv.conf"
    DO=""
    if [ "$SPILLBOX_ROOTLESS_DOCKER" == "true" ]; then
        nofile_unlimit=""
        check_limit
        sudocmd="sudo -E"
    fi
    if [ "$job_submit_command" == "" ]; then
        detach="-d"
        #restart="--restart=always"
        restart="--rm"
        name="--name $SPILLBOX_CONTAINER_NAME"
    else
        restart="--rm"
        name="--name $SPILLBOX_CONTAINER_NAME-$(date +%s)-$$ "
    fi
    CAPS="--cap-add=cap_chown --cap-add=cap_dac_override --cap-add=cap_dac_read_search --cap-add=cap_fowner --cap-add=cap_fsetid --cap-add=cap_kill --cap-add=cap_setgid --cap-add=cap_setuid --cap-add=cap_setpcap --cap-add=cap_linux_immutable --cap-add=cap_net_bind_service --cap-add=cap_net_broadcast --cap-add=cap_net_admin --cap-add=cap_net_raw --cap-add=cap_ipc_lock --cap-add=cap_ipc_owner --cap-add=cap_sys_module --cap-add=cap_sys_rawio --cap-add=cap_sys_chroot --cap-add=cap_sys_ptrace --cap-add=cap_sys_pacct --cap-add=cap_sys_admin --cap-add=cap_sys_boot --cap-add=cap_sys_nice --cap-add=cap_sys_resource --cap-add=cap_sys_time --cap-add=cap_sys_tty_config --cap-add=cap_mknod --cap-add=cap_lease --cap-add=cap_setfcap --cap-add=cap_mac_override --cap-add=cap_mac_admin"
    #CAPS="--cap-add=NET_ADMIN --cap-add=NET_RAW --cap-add=CAP_DAC_READ_SEARCH --cap-add=SYS_ADMIN --cap-add=cap_setgid"
    #SEC="--security-opt seccomp=unconfined --security-opt seccomp=$WDIR/seccomp.json"
    SEC=""
    if [ -f "$RESOLV" ]; then
        while IFS="" read -r p || [ -n "$p" ]
        do
            d=$(echo $p | grep ^nameserver | awk '{print $2}')
            if [ "$d" != "" ]; then
                DO+="--dns=$d "
            fi
            d=$(echo $p | grep ^search | awk '{$1=""; print $0}')
            if [ "$d" != "" ]; then
                for i in $d
                do
                    DO+="--dns-search=$i "
                done 
            fi
        done < $RESOLV
    fi
    MAINSERVER_WORKDIR=$(echo ${SPILLBOX_TERRAFORM_ROOT} | awk -F '//terraform' '{print $1}')
    mkdir -p ${MAINSERVER_WORKDIR}/sites/local/customers/conf
#   $job_submit_command $exe_wrapper $sudocmd stdbuf -e0 -o0 ${SPILLBOX_DOCKER_PATH} run $restart --rm $DO $portmap $nofile_unlimit --pids-limit 1024000 --privileged $name $CAPS $SEC $detach $SPILLBOX_CACHE_OPTION -v ${SPILLBOX_CACHE_DIR}:${SPILLBOX_CACHE_DIR} -v ${HOST_LOGDIR}:${SPILLBOX_CONTAINER_MOUNT_PATH} -v ${MAINSERVER_WORKDIR}/sites/local/customers/conf:${MAINSERVER_WORKDIR}/sites/local/customers/conf:ro --entrypoint ${SPILLBOX_CONTAINER_MOUNT_PATH}/install_nfs.sh $SPILLBOX_REGISTRY_SERVER/$SPILLBOX_SERVER_IMAGE ${SPILLBOX_CONTAINER_MOUNT_PATH}/start_nfs.sh > $BASE/spillbox_splfs/LOG/job_output
    set +e
    $job_submit_command $NFSJOB_OPTION $NFSJOB_NAME $exe_wrapper $sudocmd stdbuf -e0 -o0 ${SPILLBOX_DOCKER_PATH} run $restart --rm $DO $portmap $nofile_unlimit --pids-limit 1024000 --privileged $name $CAPS $SEC $detach $SPILLBOX_CACHE_OPTION --env=SPILLBOX_SITE=${SPILLBOX_SITE} -v ${SPILLBOX_CACHE_DIR}:${SPILLBOX_CACHE_DIR} -v /etc/passwd:/etc/passwd -v /etc/shadow:/etc/shadow -v /etc/group:/etc/group ${LDAP_DIR} ${NIS_DIR} -v $SPILLBOX_WORKDIR:$SPILLBOX_WORKDIR -v ${HOST_LOGDIR}:${SPILLBOX_CONTAINER_MOUNT_PATH} -v ${MAINSERVER_WORKDIR}/sites/local/customers/conf:${MAINSERVER_WORKDIR}/sites/local/customers/conf:ro --entrypoint ${SPILLBOX_CONTAINER_MOUNT_PATH}/install_nfs.sh $SPILLBOX_REGISTRY_SERVER/$SPILLBOX_SERVER_IMAGE ${SPILLBOX_CONTAINER_MOUNT_PATH}/start_nfs.sh > $BASE/spillbox_splfs/LOG/job_output
    RETVAL=$? 
    if test ${RETVAL} -ne 0 ; then
        if [ -f $BASE/spillbox_splfs/LOG/job_output ]; then
            tail -20 $BASE/spillbox_splfs/LOG/job_output | grep -v ^# >&2
        fi
        exit $RETVAL
    fi
else
    check_limit
    if [ "$SPILLBOX_CONTAINER" == "singularity" ]; then
        if [ "$job_submit_command" == "" ]; then
            detach="& disown"
            echo export SPILLBOX_CONTAINER_DETACH=true >> $SETUP_SH
        else
            echo export SPILLBOX_CONTAINER_DETACH=false >> $SETUP_SH
        fi
        set +e
        $job_submit_command $NFSJOB_OPTION $NFSJOB_NAME $exe_wrapper sudo stdbuf -e0 -o0 ${SPILLBOX_SINGULARITY_PATH} exec --writable-tmpfs $SPILLBOX_CACHE_OPTION -B ${HOST_LOGDIR}:${SPILLBOX_CONTAINER_MOUNT_PATH} $SPILLBOX_SERVER_IMAGE ${SPILLBOX_CONTAINER_MOUNT_PATH}/install_nfs.sh ${SPILLBOX_CONTAINER_MOUNT_PATH}/start_nfs.sh >& $BASE/spillbox_splfs/LOG/job_output $detach
        RETVAL=$? 
        if test ${RETVAL} -ne 0 ; then
            if [ -f $BASE/spillbox_splfs/LOG/job_output ]; then
                tail -20 $BASE/spillbox_splfs/LOG/job_output | grep -v ^# >&2
            fi
            exit $RETVAL
        fi
    else
        if [ "$job_submit_command" == "" ]; then
            detach="& disown"
            echo export SPILLBOX_CONTAINER_DETACH=true >> $SETUP_SH
        else
            echo export SPILLBOX_CONTAINER_DETACH=false >> $SETUP_SH
        fi
        check_rpc
        set +e
        $job_submit_command $NFSJOB_OPTION $NFSJOB_NAME $exe_wrapper stdbuf -e0 -o0  ${SPILLBOX_CONTAINER_MOUNT_PATH}/install_nfs.sh  >& $BASE/spillbox_splfs/LOG/job_output $detach
        RETVAL=$? 
        if test ${RETVAL} -ne 0 ; then
            if [ -f $BASE/spillbox_splfs/LOG/job_output ]; then
                tail -20 $BASE/spillbox_splfs/LOG/job_output | grep -v ^# >&2
            fi
            exit $RETVAL
        fi
    fi
fi

