#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

init_rpc() {
    echo "* Starting rpcbind"
    if [ ! -x /run/rpcbind ] ; then
        install -m755 -g 32 -o 32 -d /run/rpcbind
    fi
    rpcbind || true
    rpc.statd -L || true
    rpc.idmapd || true
    sleep 1
}

check_rpc() {
    IFS=',' read -ra SERVICE <<< "rpcbind,statd"
    for p in "${SERVICE[@]}"; do 
        if pgrep -f "$p" > /dev/null
        then
            true
        else
            echo $p is not running on this machine. Please contact your sysadmin.  >&2
            exit 3
        fi
    done
}


DIR=$(realpath $(dirname $0)) 
spillbox_setup_file=$(realpath $DIR/setup.sh)
if [ -f  $spillbox_setup_file ]; then 
    . $spillbox_setup_file
else
    echo missing setup file : $spillbox_setup_file >&2
    exit 3
fi
if [ "$SPILLBOX_CONTAINER" == "docker" ] ; then
    init_rpc
elif [ "$SPILLBOX_CONTAINER" == "host" ] ; then
    check_rpc
fi
CAN_I_RUN_SUDO=$(sudo -n uptime 2>&1|grep "load"|wc -l)
sudocmd=""
if [ $(($CAN_I_RUN_SUDO)) -gt 0 ]; then
    sudocmd="sudo"
fi

if [ "$SPILLBOX_SPLD_NFS_SETUP" == "true" ]; then
    SPILLBOX_HUB_PATH="$SPILLBOX_TERRAFORM_ROOT"
    echo "export SPILLBOX_HUB_PATH=$SPILLBOX_HUB_PATH" >> $spillbox_setup_file
    export TERRAFORM_ROOT=$SPILLBOX_TERRAFORM_ROOT
    if [ "$SPILLBOX_CONTAINER" == "host" ] ; then
        ${SPILLBOX_HUB_PATH}/hub.sh $SPILLBOX_CUSTOMER start $SPILLBOX_GLOBAL_CACHE $SPILLBOX_DISABLE_GLOBAL_CACHE > $DIR/LOG/hub.out
    fi
    SPILLBOX_HUB_SERVER=$(cat $DIR/LOG/hub.out | sed -n 's/.*"server": {[^}]*"value":"\([^"]*\)".*/\1/p')
    SPILLBOX_HUB_PAIR=$(cat $DIR/LOG/hub.out | sed -n 's/.*"pair": {[^}]*"value":"\([^"]*\)".*/\1/p')
    SPILLBOX_HUB_GLOBAL_CACHE_SERVER=$(cat $DIR/LOG/hub.out | sed -n 's/.*"global_cache_server": {[^}]*"value":"\([^"]*\)".*/\1/p')
    SPILLBOX_HUB_GLOBAL_CACHE_PAIR=$(cat $DIR/LOG/hub.out | sed -n 's/.*"global_cache_pair": {[^}]*"value":"\([^"]*\)".*/\1/p')
    SPILLBOX_HUB_IP=$(echo $SPILLBOX_HUB_SERVER | awk -F: '{print $1}')
    SPILLBOX_HUB_PORT=$(echo $SPILLBOX_HUB_SERVER | awk -F: '{print $2}')
    GANESHA_CONFIG_HOST="$DIR/conf"
    sed -i "s|Srv_Addr = .*;|Srv_Addr = ${SPILLBOX_HUB_IP};|" $GANESHA_CONFIG_HOST
    sed -i "s|WS_Port = .*;|WS_Port = ${SPILLBOX_HUB_PORT};|" $GANESHA_CONFIG_HOST
    sed -i "s|WS_Pair = .*;|WS_Pair = ${SPILLBOX_HUB_PAIR};|" $GANESHA_CONFIG_HOST
    global=false
    if [ -f "${SPILLBOX_TERRAFORM_ROOT}/etc/spillbox_config_cache" ]; then
        customer=$(cat ${SPILLBOX_TERRAFORM_ROOT}/etc/spillbox_config_cache)
        if [ "$customer" != "$SPILLBOX_CUSTOMER" ] ; then
            global=true
            echo "Using cache from project $customer"
        fi
    fi
    if [ "${global}" == "true" ] ; then
        if [ "${SPILLBOX_DISABLE_GLOBAL_CACHE}" == "false" ] ; then
            if [ "${SPILLBOX_GLOBAL_CACHE_SERVER}" == "" ] ; then
                echo "global_cache_server value not set" >&2
                exit 1  
            fi 
            host=$(echo "$SPILLBOX_HUB_GLOBAL_CACHE_SERVER" | awk -F: '{print $1}')
            port=$(echo "$SPILLBOX_HUB_GLOBAL_CACHE_SERVER" | awk -F: '{print $2}')
            if [ "$port" == "" ] ; then
                echo "global_cache_server port value not set" >&2
                exit 1
            fi
            sed -i "s|Srv_Global_Cache_Addr = .*;|Srv_Global_Cache_Addr = ${host};|" $GANESHA_CONFIG_HOST
            sed -i "s|WS_Global_Cache_Port = .*;|WS_Global_Cache_Port = ${port};|" $GANESHA_CONFIG_HOST
            sed -i "s|WS_Global_Cache_Pair = .*;|WS_Global_Cache_Pair = ${SPILLBOX_HUB_GLOBAL_CACHE_PAIR};|" $GANESHA_CONFIG_HOST
        fi
    fi
fi



sudo chmod 777 $SPILLBOX_CACHE_DIR || /bin/true
#sudo systemctl stop firewalld || /bin/true
#sudo systemctl disable firewalld || /bin/true

if [ "$SPILLBOX_CONTAINER" == "docker" ] ; then
    $sudocmd mkdir -p /opt/spillbox
    $sudocmd chmod 777 /opt/spillbox
    (cd / ; $sudocmd tar xfz $DIR/splfs.tar)
    yes | $sudocmd cp /opt/spillbox/splfs/lib/libganesha_nfsd.so.2.8 /lib64/.
    yes | $sudocmd cp /opt/spillbox/splfs/lib/libntirpc.so.1.8.0 /lib64/libntirpc.so.1.8
    if [ "$SPILLBOX_ROOTLESS_DOCKER" == "true" ] ; then
        $sudocmd setcap cap_dac_read_search,cap_setgid,CAP_NET_RAW,CAP_NET_ADMIN,CAP_NET_BROADCAST,CAP_NET_BIND_SERVICE=+eip /opt/spillbox/splfs/bin/ganesha.nfsd
    else
        #setcap cap_dac_read_search,cap_setgid=+eip /opt/spillbox/splfs/bin/ganesha.nfsd
        $sudocmd setcap cap_chown,cap_dac_override,cap_dac_read_search,cap_fowner,cap_fsetid,cap_kill,cap_setgid,cap_setuid,cap_setpcap,cap_linux_immutable,cap_net_bind_service,cap_net_broadcast,cap_net_admin,cap_net_raw,cap_ipc_lock,cap_ipc_owner,cap_sys_module,cap_sys_rawio,cap_sys_chroot,cap_sys_ptrace,cap_sys_pacct,cap_sys_admin,cap_sys_boot,cap_sys_nice,cap_sys_resource,cap_sys_time,cap_sys_tty_config,cap_mknod,cap_lease,cap_setfcap,cap_mac_override,cap_mac_admin,cap_syslog+eip /opt/spillbox/splfs/bin/ganesha.nfsd
        #setcap cap_chown,cap_dac_override,cap_dac_read_search,cap_fowner,cap_setgid,cap_setuid,cap_mknod,cap_setfcap,cap_syslog,cap_sys_admin+eip /opt/spillbox/splfs/bin/ganesha.nfsd
    fi
    if [ "$SPILLBOX_BOOSTMODE" == "true" ]; then
        if [ "$SPILLBOX_BOOST_SERVER_CACHE_SIZE" = "" ]; then
            echo "SPILLBOX_BOOST_SERVER_CACHE_SIZE not set" >&2 
        else
            RAND="/spillbox_server_cache$$"
            $sudocmd mkdir  $RAND $SPILLBOX_CACHE_DIR
            $sudocmd mount -t ramfs ramfs $RAND
            $sudocmd truncate -s $SPILLBOX_BOOST_SERVER_CACHE_SIZE $RAND/ramdisk.img
            yes| $sudocmd  mkfs.ext4 $RAND/ramdisk.img
            $sudocmd mount $RAND/ramdisk.img $SPILLBOX_CACHE_DIR
            $sudocmd chmod 777 $SPILLBOX_CACHE_DIR
        fi
    fi
    $sudocmd getcap /opt/spillbox/splfs/bin/ganesha.nfsd
    $sudocmd userdel -r --force --remove ${SPILLBOX_CLOUD_USER} || true
    $sudocmd groupdel ${SPILLBOX_CLOUD_GROUP}  || true
    $sudocmd groupadd -r ${SPILLBOX_CLOUD_GROUP} -g ${SPILLBOX_CLOUD_GID}
    RETVAL=$?
    if test ${RETVAL} -ne 0 ; then
        $sudocmd useradd  --uid ${SPILLBOX_CLOUD_UID}  ${SPILLBOX_CLOUD_USER}
    else
        $sudocmd useradd  --uid ${SPILLBOX_CLOUD_UID}  -g ${SPILLBOX_CLOUD_GROUP} ${SPILLBOX_CLOUD_USER}
    fi
    #exec sudo -E -u  $SPILLBOX_CLOUD_USER --  "$@"
    exec "$@"
elif [ "$SPILLBOX_CONTAINER" == "singularity" ] ; then
    sudo mkdir -p /dev/shm/opt/spillbox /opt
    sudo ln -s /dev/shm/opt/spillbox /opt/spillbox
    (cd / ; tar xfz $DIR/splfs.tar)
    yes | cp /opt/spillbox/splfs/lib/libganesha_nfsd.so.2.8 /lib64/.
    yes | cp /opt/spillbox/splfs/lib/libntirpc.so.1.8.0 /lib64/libntirpc.so.1.8
    #setcap cap_dac_read_search,cap_setgid,CAP_NET_RAW,CAP_NET_ADMIN,CAP_NET_BROADCAST,CAP_NET_BIND_SERVICE=+eip /opt/spillbox/splfs/bin/ganesha.nfsd
    #setcap CAP_NET_RAW,CAP_SETPCAP,CAP_IPC_LOCK,CAP_SYS_TTY_CONFIG,CAP_MAC_OVERRIDE,CAP_DAC_READ_SEARCH,CAP_NET_BROADCAST,CAP_LEASE,CAP_SYS_PTRACE,CAP_MAC_ADMIN,CAP_CHOWN,CAP_KILL,CAP_LINUX_IMMUTABLE,CAP_SYS_MODULE,CAP_FOWNER,CAP_SYS_NICE,CAP_SYSLOG,CAP_SYS_RESOURCE,CAP_NET_BIND_SERVICE,CAP_SYS_CHROOT,CAP_SYS_PACCT,CAP_SYS_ADMIN,CAP_SYS_BOOT,CAP_IPC_OWNER,CAP_SYS_RAWIO,CAP_DAC_OVERRIDE,CAP_FSETID,CAP_SETGID,CAP_SETUID,CAP_NET_ADMIN,CAP_SETFCAP,CAP_SYS_TIME,CAP_MKNOD=+eip /opt/spillbox/splfs/bin/ganesha.nfsd
    getcap /opt/spillbox/splfs/bin/ganesha.nfsd
    groupadd -r ${SPILLBOX_CLOUD_GROUP} -g ${SPILLBOX_CLOUD_GID}
    useradd  --uid ${SPILLBOX_CLOUD_UID}  -g ${SPILLBOX_CLOUD_GROUP} ${SPILLBOX_CLOUD_USER}
    chown -R $SPILLBOX_CLOUD_USER /opt/spillbox/splfs 
    if [ ! -d "/spillbox_cachedir" ]; then 
        MEMTOTAL=$(expr $(vmstat -s | grep "total memory" | awk '{print $1}'))
        MEMTOTAL=$(($MEMTOTAL / 1048576 ))
        LIMIT="8"
        if [ "$MEMTOTAL" -lt "$LIMIT" ] ; then
            echo Not enough memory : $MEMTOTAL is less than $LIMIT >&2
            exit 1
        fi
        TMPFS_SIZE=$(expr $MEMTOTAL - $LIMIT)
        mkdir -p /spillbox_cachedir
        mount -t tmpfs -o size=${TMPFS_SIZE}G tmpfs /spillbox_cachedir
        chmod 777 /spillbox_cachedir
    fi
    #exec sudo -E -u  $SPILLBOX_CLOUD_USER --  "$@"
    exec "$@"
elif [ "$SPILLBOX_CONTAINER" == "host" ] ; then
    if [ "$SPILLBOX_SPLFS_HOME" == "" ]; then
        OPTDIR="${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/"
        mkdir -p $OPTDIR
        chmod 777 $OPTDIR
        (cd ${OPTDIR} ; tar xfz $DIR/splfs.tar)
        mv $OPTDIR/opt/spillbox/splfs/bin/ganesha.nfsd $OPTDIR/opt/spillbox/splfs/bin/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME}
        if [ "$SPILLBOX_STORAGE" == "VFS" ] ; then
            if [ "${SPILLBOX_PATCHELF_PATH}" == "" ]; then
                echo spillbox_patchelf_path is empty >&2
                exit $3
            fi
            patchelf --set-rpath ${OPTDIR}/opt/spillbox/splfs/lib ${OPTDIR}/opt/spillbox/splfs/bin/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME}
            RETVAL=$?
            if test ${RETVAL} -ne 0 ; then
                echo patchelf failed. >&2
                exit $RETVAL
            fi
            #sudo setcap cap_dac_read_search,cap_setgid,CAP_NET_RAW,CAP_NET_ADMIN,CAP_NET_BROADCAST,CAP_NET_BIND_SERVICE=+eip ${OPTDIR}/opt/spillbox/splfs/bin/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME}
            sudo setcap cap_chown,cap_dac_override,cap_dac_read_search,cap_fowner,cap_fsetid,cap_kill,cap_setgid,cap_setuid,cap_setpcap,cap_linux_immutable,cap_net_bind_service,cap_net_broadcast,cap_net_admin,cap_net_raw,cap_ipc_lock,cap_ipc_owner,cap_sys_module,cap_sys_rawio,cap_sys_chroot,cap_sys_ptrace,cap_sys_pacct,cap_sys_admin,cap_sys_boot,cap_sys_nice,cap_sys_resource,cap_sys_time,cap_sys_tty_config,cap_mknod,cap_lease,cap_setfcap,cap_mac_override,cap_mac_admin,cap_syslog+eip ${OPTDIR}/opt/spillbox/splfs/bin/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME}
            RETVAL=$?
            if test ${RETVAL} -ne 0 ; then
                echo setcap failed. Please check if you have right permission >&2
                exit $RETVAL
            fi
        fi
    else
        if [ ! -f "$SPILLBOX_SPLFS_HOME/opt/spillbox/splfs/bin/ganesha.nfsd" ] ; then
            echo $SPILLBOX_SPLFS_HOME/opt/spillbox/splfs/bin/ganesha.nfsd not found >&2
            exit 1
        fi
    fi
    getcap ${OPTDIR}/opt/spillbox/splfs/bin/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME}

    exec ${DIR}/start_nfs.sh
fi
