#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

function copy_as_user {
    local tmplog="$1"
    local SPILLBOX_CONTAINER_MOUNT_PATH="$2"
    local SPILLBOX_CLOUD_USER="$3"
    timeout 120 /bin/bash -c "until stat $tmplog/conf.out  2>/dev/null; do echo waiting inside container ...; sleep 2; done"
    sleep 10
    if [ -f $tmplog/conf.out ]; then
        sudo -E -u  $SPILLBOX_CLOUD_USER -- cp -f $tmplog/conf.out ${SPILLBOX_CONTAINER_MOUNT_PATH}/LOG/conf.out 
    else
        echo file server conf.out not produced >&2
    fi
}

check_limit() {
    ulimit -s unlimited
    CAN_I_RUN_SUDO=$(sudo -n uptime 2>&1|grep "load"|wc -l)
    nr_open=$(sysctl  fs.nr_open | awk -F"=" '{print $2}' | sed 's/ //g')
    if test $nr_open  -lt 1048576  ; then 
        if [[ $(($CAN_I_RUN_SUDO)) -gt 0 ]]; then
            sudo sysctl -w fs.nr_open=1048576
        else 
            echo "Please increase kernel fs.nr_open limit to 1048576 or more" >&2
            exit 1  
        fi
    fi
    file_max=$(sysctl  fs.file-max | awk -F"=" '{print $2}' | sed 's/ //g')
    if test $file_max  -lt 10240000  ; then 
        if [[ $(($CAN_I_RUN_SUDO)) -gt 0 ]]; then
            sudo sysctl -w fs.file-max=10240000
        else 
            echo "Please increase kernel fs.file-max limit to 10240000 or more. Please run:" >&2
            echo "sudo sysctl -w fs.file-max=10240000" >&2
            echo "Please add below entries to /etc/sysctl.conf" >&2
            echo "" >&2
            echo "" >&2
            fs.file-max=10240000
            echo "" >&2
            exit 1  
        fi
    fi
    #file
    maxfd=$(ulimit -Hn)
    if test $maxfd  -lt 1024000  ; then
        echo "" >&2
        echo "" >&2
        echo "Please increase user's max open file descriptor limit to 1024000 or more" >&2
        echo "Please add below entries to /etc/security/limits.conf" >&2
        echo "" >&2
        echo "" >&2
        echo "*     hard nofile 1024000" >&2
        echo "*     soft nofile 1024000" >&2
        echo "" >&2
    fi
    ulimit -n $maxfd
}

DIR=$(realpath $(dirname $0)) 
spillbox_setup_file=$(realpath $DIR/setup.sh)
if [ -f  $spillbox_setup_file ]; then 
    . $spillbox_setup_file
else
    echo missing setup file : $spillbox_setup_file >&2
    exit 3
fi

LOGDIR=$DIR/LOG
if [ ! -d "$LOGDIR" ]; then
    echo "directory $LOGDIR is missing" 2>&1 | tee /dev/stderr
    exit 1 
fi

CAN_I_RUN_SUDO=$(sudo -n uptime 2>&1|grep "load"|wc -l)
sudocmd=""
if [ $(($CAN_I_RUN_SUDO)) -gt 0 ]; then
    sudocmd="sudo -u  $SPILLBOX_CLOUD_USER --"
fi

mountpath=$(echo $SPILLBOX_MOUNT_PATHS | tr "," "\n")
for apath in $mountpath
do
    $sudocmd mkdir -p ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/data/$apath
    $sudocmd mkdir -p ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$apath
#    $sudocmd chown  ${SPILLBOX_CLOUD_USER} ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/data/$apath
    $sudocmd chgrp  ${SPILLBOX_CLOUD_GROUP} ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/data/$apath
    $sudocmd chmod 0777  ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/data/$apath
    $sudocmd chmod 0777  ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}
done

if [ "${SPILLBOX_GLOBAL_CACHE}" == "false" ]; then
    IFS=' ' read -ra HYBRID <<< "${SPILLBOX_HYBRID_MOUNT_PATHS}"
    for i in "${HYBRID[@]}"; do
        if [ "$i" == "NONE" ]; then
             break 
        fi
        if [ "$i" == "ALL" ]; then
            for apath in $mountpath
            do
                $sudocmd mkdir -p ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$apath
                if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-missing"; then
                    echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$apath/spillbox.mstat
                fi
                if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-download"; then
                    echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$apath/spillbox.dstat
                fi
                if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-stream"; then
                    echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$apath/spillbox.tstat
                fi
                if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-sync"; then
                    echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$apath/spillbox.sstat
                fi
            done
            break
        fi
        $sudocmd mkdir -p ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$i
        if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-missing"; then
            echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$i/spillbox.mstat
        fi
        if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-download"; then
            echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$i/spillbox.dstat
        fi
        if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-stream"; then
            echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$i/spillbox.tstat
        fi
        if echo "$SPILLBOX_HYBRID_OPTION" | grep -q "\-sync"; then
            echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$i/spillbox.sstat
        fi
    done
    
    IFS=' ' read -ra HYBRID <<< "${SPILLBOX_HYBRID_MOUNT_PATHS_EXCLUDE}"
    for i in "${HYBRID[@]}"; do
        $sudocmd mkdir -p ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$i
        $sudocmd touch ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$i/spillbox.nstat
    done

    pattern='^[^[:space:]:]+:[^[:space:]:]+$'
    IFS=' ' read -ra array <<<  ${SPILLBOX_CONTAINER_PATH_REMAP}
    for i in "${array[@]}"; do
        if [[ $i =~ $pattern ]]; then
            IFS=':' read -ra j <<< $i
            $sudocmd mkdir -p ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/${j[0]}
            echo "${j[1]}" |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/${j[0]}/spillbox.nstat
        else
            echo "Invalid entry in container_path_remap: $i"
        fi
    done
else
    for apath in $mountpath
    do
        $sudocmd mkdir -p ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$apath
        echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$apath/spillbox.mstat
        echo 'h' |& $sudocmd tee ${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/meta/$apath/spillbox.sstat
    done
fi

kill_descendant_processes() {
    local pid="$1"
    local and_self="${2:-false}"
    local initial_pid="${3:-false}"
    if children="$(pgrep -P "$pid")"; then
        for child in $children; do
            kill_descendant_processes "$child" true 
        done
    fi
    MAINSERVER_WORKDIR=$(echo ${SPILLBOX_TERRAFORM_ROOT} | awk -F '/terraform' '{print $1}' )
    spillbox_setup_file=$(realpath ${MAINSERVER_WORKDIR}/sites/local/customers/conf/${SPILLBOX_CUSTOMER})
    if [ -f  $spillbox_setup_file ]; then
	. $spillbox_setup_file
    else
        echo missing setup file : $spillbox_setup_file >&2
        exit 3
    fi
    if [ "$SPILLBOX_SPLD_NFS_SETUP" == "true" ]; then
        export TERRAFORM_ROOT=$SPILLBOX_TERRAFORM_ROOT
        ${SPILLBOX_HUB_PATH}/hub.sh  $SPILLBOX_CUSTOMER stop $SPILLBOX_GLOBAL_CACHE $SPILLBOX_DISABLE_GLOBAL_CACHE

    fi
    if [[ "$pid" == "$initial_pid" ]] ; then
        if [ "$SPILLBOX_CONTAINER" == "docker" ]; then
            if [[ "$SPILLBOX_JOB_MANAGER" == "uge" ]] && [[ "$sig" == "12" || "$sig" == "10" ]] ; then
                ret=$( /spillbox_splfs/uninstall_splfs.sh ${SPILLBOX_CACHE_DIR} ${SPILLBOX_CUSTOMER} ${SPILLBOX_RM_CACHE})
            elif [[ "$SPILLBOX_JOB_MANAGER" == "lsf" ]] && [[ "$sig" == "2" ]]; then
                ret=$( /spillbox_splfs/uninstall_splfs.sh ${SPILLBOX_CACHE_DIR} ${SPILLBOX_CUSTOMER} ${SPILLBOX_RM_CACHE})
            elif [[ "$SPILLBOX_JOB_MANAGER" == "nc" ]] && [[ "$sig" == "1" || "$sig" == "2" || "$sig" == "15"  ]]; then
                ret=$( /spillbox_splfs/uninstall_splfs.sh ${SPILLBOX_CACHE_DIR} ${SPILLBOX_CUSTOMER} ${SPILLBOX_RM_CACHE})
            elif [[ "$SPILLBOX_JOB_MANAGER" == "slurm" ]] && [[ "$sig" == "15" ]]; then
                ## Slurm initially sends SIGCONT to wake up the process if it is sleeping
                ## And then send the SIGTERM, so handling only SIGTERM
                ret=$( /spillbox_splfs/uninstall_splfs.sh ${SPILLBOX_CACHE_DIR} ${SPILLBOX_CUSTOMER} ${SPILLBOX_RM_CACHE})
       		fi
	    else
		    setsid $SPILLBOX_TERRAFORM_ROOT/stacks/modules/docker-nfs/install/spillbox_splfs/uninstall_splfs.sh ${SPILLBOX_CACHE_DIR} ${SPILLBOX_CUSTOMER} ${SPILLBOX_RM_CACHE}
	    fi	
    fi
    if [[ "$and_self" == true ]]; then
        kill -9 "$pid"
    fi
}

func_handler() {
    raw_sig=$?
    sig=$(($raw_sig - 128))
    echo "killing nfs pid $NFSD_PID children on signal $raw_sig ($sig)"
    kill_descendant_processes $$ true $$
    exit 0
}

if [[ "$SPILLBOX_JOB_MANAGER" == "uge" ]] || [[ "$SPILLBOX_JOB_MANAGER" == "lsf" ]] || [[ "$SPILLBOX_JOB_MANAGER" == "nc" ]] || [[ "$SPILLBOX_JOB_MANAGER" == "slurm" ]] ; then
        for signal in HUP INT QUIT ABRT SEGV USR1 USR2 TERM ; do
            echo "trapping signal ${signal}"
            trap "func_handler" ${signal}
        done
fi
echo "* Starting NFS"
export LD_LIBRARY_PATH=/opt/spillbox/splfs/lib
if [ "$SPILLBOX_CONTAINER" == "docker" ]; then
    maxfd=$(ulimit -Hn)
    if [ "${maxfd}"  == "unlimited" ]; then
        ulimit -Sn $maxfd
    else
        if test $maxfd  -lt 1024000  ; then
            echo "Please increase user's max open file descriptor limit to 1024000 or more" >&2
            exit 1
        fi
        if test $maxfd  -gt 1024000  ; then
            maxfd=1024000
        fi
        ulimit -n $maxfd
    fi
    maxproc=$(ulimit -Hu)
    if [ "${maxproc}"  == "unlimited" ]; then
        ulimit -Su 4096 
    else 
        if [ ${maxproc} -gt 4096 ]; then
            maxproc=4096
        fi
        if test $maxproc  -lt 4096 ; then
            echo "Please increase user's max processes limit to 4096 or more" >&2
            exit 1
        fi
        ulimit -u $maxproc
    fi 
    sudo /opt/spillbox/splfs/bin/ganesha.nfsd  -F  -L $LOGDIR/LOG -f $DIR/conf -N NIV_EVENT -p $LOGDIR/pid >& $LOGDIR/LOG.STDOUT &
    NFSD_PID=$!
    wait $NFSD_PID
elif [ "$SPILLBOX_CONTAINER" == "singularity" ]; then
    tmplog="/opt/spillbox/splfs/LOG" 
    /bin/rm -rf $tmplog
    mkdir -p $tmplog
    /bin/rm -f ${SPILLBOX_CONTAINER_MOUNT_PATH}/LOG/conf.out
    ln -s /opt/spillbox/splfs/bin/ganesha.nfsd  $tmplog/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME}
    detach=""
    if [ "$SPILLBOX_CONTAINER_DETACH" == "false" ]; then
        detach="-F"
    fi
    copy_as_user $tmplog $SPILLBOX_CONTAINER_MOUNT_PATH $SPILLBOX_CLOUD_USER &
    $tmplog/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME} $detach  -L STDOUT -f $DIR/conf -N NIV_EVENT -p $tmplog/pid 
elif [ "$SPILLBOX_CONTAINER" == "host" ]; then
    check_limit
    detach=""
    if [ "$SPILLBOX_CONTAINER_DETACH" == "false" ]; then
        detach="-F"
    fi
    OPTDIR="${SPILLBOX_CACHE_DIR}/${SPILLBOX_CUSTOMER}/cache_dir/"
    /bin/rm -f ${SPILLBOX_CONTAINER_MOUNT_PATH}/conf.out
    if [ "$SPILLBOX_SPLFS_HOME" == "" ]; then
        pkill -9 -f "ganesha.nfsd.${SPILLBOX_CONTAINER_NAME}"
        LD_LIBRARY_PATH=${OPTDIR}/opt/spillbox/splfs/lib sudo ${OPTDIR}/opt/spillbox/splfs/bin/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME} $detach -f $DIR/conf -N NIV_EVENT -L $LOGDIR/LOG -p $OPTDIR/pid >& $LOGDIR/LOG.STDOUT &
	    NFSD_PID=$!
        wait $NFSD_PID
    else
        ln -s ${SPILLBOX_SPLFS_HOME}/opt/spillbox/splfs/bin/ganesha.nfsd  $OPTDIR/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME}
        LD_LIBRARY_PATH=${SPILLBOX_SPLFS_HOME}/opt/spillbox/splfs/lib $OPTDIR/ganesha.nfsd.${SPILLBOX_CONTAINER_NAME} $detach -f $DIR/conf -N NIV_EVENT -L $LOGDIR/LOG -p $OPTDIR/pid  >& $LOGDIR/LOG.STDOUT &
	    NFSD_PID=$!
        wait $NFSD_PID
    fi
    timeout 120 /bin/bash -c "until stat $SPILLBOX_CONTAINER_MOUNT_PATH/conf.out  2>/dev/null; do echo -n \#; sleep 10; done"
    sleep 10
fi
