# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

terraform {
  required_version = ">= 1.10.4, < 1.10.5" 
  required_providers {
    external = {
      version = "2.3.4"
    }
    local = {
      version = "2.5.2"
    }
    null = {
      version = "3.2.3"
    }
  }
}

variable "container" {
  description = "Container type"
}

variable "client_hosts" {
  default = ""
  description = "list of client_hosts"
}

variable "runtime_container" {
  description = "Runtime Container type"
}

variable "server_base_image" {
  description = "Server base image"
}

variable "exclude_mount_paths" {
  description = "Exclude on-premise mount paths"
}

variable "disable_server_image" {
  type = bool
  description = "Disable building of server image"
}

variable "disable_registry_server" {
  type = bool
  description = "Disable local machine as registry server"
}

variable "registry_server" {
  default = ""
  description = "Customer existing registry server"
}

variable "vm_user" {
  description = "Unix username under which Spillbox server runs"
}


variable "docker_registry_port" {
  description = "Docker registry port"
}

variable "docker_path" {
  description = "Unix path of docker binary "
}

variable "singularity_path" {
  description = "Unix path of singularity container image"
}

variable "rootless_docker" {}

provider "local" {}

variable "boostmode" {
  type = bool
  description = "Boost mode"
}

variable "spld_nfs_setup" {
  type = bool
  description = "Spld Nfs Setup"
}
variable "server_base_image_tag_name" {}
variable "update_server_base_image" {}
locals {
    need_registry = var.container == "docker" ? 1 : var.runtime_container == "docker" ? 1 : 0
    create_registry = local.need_registry == 1 ? var.disable_registry_server == true ? 0 : 1 : 0 
    registry_server = var.docker_registry_port == "" ? "" : "${var.docker_host}:${var.docker_registry_port}"
    image_name = var.container == "singularity" ? var.server_base_image :  var.disable_server_image == true? var.server_base_image : "spillbox_common" 
}

variable "network_interface" {
  description = "network interface name"
}

variable "docker_host" {
  description = "Docker server hostname"
}

variable "docker_port" {
  description = "Network port used by docker server"
}

resource "null_resource" "registry" {
  count = var.container == "singularity" ? 0 : local.create_registry
  provisioner "local-exec" {
    command = "${path.module}/install/install_registry.sh ${abspath(path.module)} ${var.docker_registry_port} ${var.docker_path} ${var.rootless_docker}"
  }
}

resource "null_resource" "registry_destroy" {
  count = var.container == "singularity" ? 0 : local.create_registry
  triggers = {
    docker_path = var.docker_path
  }
  provisioner "local-exec" {
    when    = destroy 
    command = "${self.triggers.docker_path} container stop spillbox_registry && ${self.triggers.docker_path} container rm -v spillbox_registry || true"
  }
}

resource "null_resource" "image" {
  count = var.container == "singularity" ? 0 : local.create_registry
  depends_on   = [null_resource.registry]
  provisioner "local-exec" {
   command = "${path.module}/install/build_server.sh '${local.registry_server}' '${var.server_base_image}' '${local.image_name}' '${var.disable_server_image}' '${var.disable_registry_server}' '${var.docker_path}' '${var.rootless_docker}' '${var.server_base_image_tag_name}' '${var.update_server_base_image}' '${path.module}'"
  }
}

output "container" {
  value = var.container
}

output "runtime_container" {
  value = var.runtime_container
}

output "registry_server" {
  value = local.registry_server
}

output "exclude_mount_paths" {
  value = var.exclude_mount_paths 
}
output "client_hosts" {
  value = var.client_hosts
}
output "boostmode" {
  value = var.boostmode
}
output "spld_nfs_setup" {
  value = var.spld_nfs_setup
}
data "external" "image_data" {
  depends_on = [null_resource.image]

  program = ["bash", "-c", "echo '{\"image_tag\": \"'$(tr -d '\n' < ${path.module}/image_output.txt)'\"}'"]
}

output "server_image" {
  value = data.external.image_data.result["image_tag"]
}
