#!/bin/bash -f

#Copyright © 2025, Spillbox. All rights reserved.
#Property of Spillbox Inc.
#Any kind of copying or producing derivative work is prohibited without
#explicit written consent from Spillbox Inc.

#-----------------------------------------------------------------------
#Description:  This script is used to create a ami image to docker image in the current machine.
#-----------------------------------------------------------------------

show_usage() {
    echo "Usage: $0 <provider> <tagname> <create|[delete]>"
    exit 1
}

if [[ $# -lt 3 ]]; then
    show_usage
fi

CLOUD_PROVIDER="${1}"
TAGNAME="${2}"
NODESTAGE="${3}"
WEBSERVER_IP=$(hostname -I | awk '{print $1}')

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
export SPILLBOX_USER=$(whoami)
TERRAFORM_DIR="$DIR/../../bin"
LOCK_TIMEOUT="20m"
centos=0
OSID=$(cat /etc/os-release | grep ^ID= | awk -F= '{print $2}' | sed 's/"//g')
APT="apt-get"
CONFIG=""

#checking for the os
if [[ $OSID == "centos" ]] || [[ "$OSID" == "rhel" ]]; then
    APT="yum"
    centos=1
    CHRONYCONF="/etc/chrony.conf"
    centos_version=$(rpm -E %{rhel})
    if [[ -f /etc/centos-release && ( "$centos_version" == "7" || "$centos_version" == "8" ) ]]; then
        sudo find /etc/yum.repos.d/ -type f -name '*.repo' -exec sed -i 's/mirror.centos.org/vault.centos.org/g' {} +
        sudo find /etc/yum.repos.d/ -type f -name '*.repo' -exec sed -i 's/^#.*baseurl=http/baseurl=http/g' {} +
        sudo find /etc/yum.repos.d/ -type f -name '*.repo' -exec sed -i 's/^mirrorlist=http/#mirrorlist=http/g' {} +
    fi
fi
if [[ $OSID == "amzn" ]]; then
    alinux=1
    APT="yum"
fi

#Checking for the spillbox_image_config
if [ -f $DIR/spillbox_image.config ]; then
    CONFIG=$DIR/spillbox_image.config
else
    echo "ERROR :: No config file detected!!!"
    exit 1
fi

#sourcing the config file
. $CONFIG

ami_create() {
LOGDIR="$DIR/${TAGNAME}/logs"
mkdir -p $LOGDIR

echo "Log File: "
echo "   $LOGDIR/log.apply"

#Installing the jq package
which jq > /dev/null
ret=$?
if test $ret -ne 0 ; then
    echo "Installing jq package : " | tee -a $LOGDIR/log.apply
    sudo $APT install -y wget   >> $LOGDIR/log.apply
    if [ "$OSID" == "rhel" ]; then
        rhel_version=$(rpm -E %{rhel})
        wget -P /tmp https://dl.fedoraproject.org/pub/epel/epel-release-latest-${rhel_version}.noarch.rpm  >> $LOGDIR/log.apply
        sudo $APT install -y /tmp/epel-release-latest-${rhel_version}.noarch.rpm  >> $LOGDIR/log.apply
    else
        sudo $APT install -y epel-release  >> $LOGDIR/log.apply
    fi
    sudo $APT install -y jq 2>&1 >> $LOGDIR/log.apply | tee -a $LOGDIR/log.apply
fi

# Check if rsync is installed
if ! command -v rsync >/dev/null 2>&1; then
    echo "rsync is not installed. Installing..."
    sudo $APT install -y rsync  >> $LOGDIR/log.apply
fi

# Check if unzip is installed
if ! command -v unzip &> /dev/null; then
    echo "unzip is not installed. Installing..."
    sudo $APT install -y unzip  >> $LOGDIR/log.apply
fi

#check if wget packages is installed
if ! command -v wget &> /dev/null; then
    echo "wget is not installed. Installing..."
    sudo $APT install -y wget  >> $LOGDIR/log.apply
fi

#check if yum-utils packages is installed
if ! command -v yum-config-manager &> /dev/null; then
    echo "yum-utils is not installed. Installing..."
    sudo $APT install -y yum-utils >> $LOGDIR/log.apply
fi

#checking for the credentials and cli
case "$CLOUD_PROVIDER" in
    aws)
        if [[ -z "$AWS_ACCESS_KEY_ID" || -z "$AWS_SECRET_ACCESS_KEY" || -z "$AWS_DEFAULT_REGION" ]]; then
            echo "Error: One or more required AWS environment variables are not set."
            echo "Ensure the following variables are set:"
            echo "  - AWS_ACCESS_KEY_ID"
            echo "  - AWS_SECRET_ACCESS_KEY"
            echo "  - AWS_DEFAULT_REGION"
            exit 1
        fi
        # Check if aws cli is installed, if not, install it
        if ! command -v aws &> /dev/null; then
            echo  "AWS CLI is not installed. Installing..."
            curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip" >> $LOGDIR/log.apply
            unzip awscliv2.zip >> $LOGDIR/log.apply
            sudo ./aws/install >> $LOGDIR/log.apply
        fi
        ;;
    azure)
        if [[ -z "$ARM_CLIENT_ID" || -z "$ARM_CLIENT_SECRET" || -z "$ARM_SUBSCRIPTION_ID" || -z "$ARM_TENANT_ID"  ]]; then
            echo "Cloud Crendentials missing" 2>&1 | tee /dev/stderr
            echo "Check if these varaiables are set : ARM_CLIENT_ID  ARM_CLIENT_SECRET ARM_SUBSCRIPTION_ID ARM_TENANT_ID" 2>&1 | tee /dev/stderr
            echo "Do az login " 2>&1 | tee /dev/stderr
            exit 3
        fi
        # Check if azure cli is installed, if not, install it
        if ! command -v az &> /dev/null; then
            if [ -f /etc/os-release ]; then
                . /etc/os-release
                if [[ "$ID" == "rhel" && ( "$VERSION_ID" == 8* ) ]]; then
                        sudo yum -y install dnf
                        sudo rpm --import https://packages.microsoft.com/keys/microsoft.asc >> $LOGDIR/log.apply
                        sudo dnf install -y https://packages.microsoft.com/config/rhel/8/packages-microsoft-prod.rpm >> $LOGDIR/log.apply
                        sudo dnf install -y azure-cli >> $LOGDIR/log.apply
                elif  [[ "$ID" == "rhel" && ( "$VERSION_ID" == 9* ) ]] ; then
                        sudo yum -y install dnf  >> $LOGDIR/log.apply
                        sudo rpm --import https://packages.microsoft.com/keys/microsoft.asc >> $LOGDIR/log.apply
                        sudo dnf install -y https://packages.microsoft.com/config/rhel/9.0/packages-microsoft-prod.rpm >> $LOGDIR/log.apply
                        sudo dnf install -y azure-cli >> $LOGDIR/log.apply
                elif [[ "$ID" == "ubuntu" ]]; then
                        curl -sL https://aka.ms/InstallAzureCLIDeb | sudo bash >> $LOGDIR/log.apply
                fi
                if [[ "$centos" == 1 ]]; then
                     sudo $APT install -y dnf  >> $LOGDIR/log.apply
    sudo rpm --import https://packages.microsoft.com/keys/microsoft.asc
    echo -e "[azure-cli]
name=Azure CLI
baseurl=https://packages.microsoft.com/yumrepos/azure-cli
enabled=1
gpgcheck=1
gpgkey=https://packages.microsoft.com/keys/microsoft.asc" | sudo tee /etc/yum.repos.d/azure-cli.repo
    sudo dnf install -y azure-cli  >> $LOGDIR/log.apply
                fi
             fi
        fi
        ;;
    google)
        if [[ -z "$GOOGLE_PROJECT" || -z "$GOOGLE_APPLICATION_CREDENTIALS" ]]; then
            echo "Cloud Crendentials missing" 2>&1 | tee /dev/stderr
            echo "Check if these varaiables are set : GOOGLE_PROJECT GOOGLE_APPLICATION_CREDENTIALS" 2>&1 | tee /dev/stderr
            echo "Do gcloud login : gcloud auth activate-service-account  <service account name> --key-file=/path/to/credfile" 2>&1 | tee /dev/stderr
            exit 3
        fi
        # Check if google cli is installed, if not, install it
        if ! command -v gcloud &> /dev/null; then
            if [ -f /etc/os-release ]; then
                . /etc/os-release
                if [[ "$ID" == "rhel" && ( "$VERSION_ID" == 8* || "$VERSION_ID" == 9* ) ]]; then
                    sudo tee -a /etc/yum.repos.d/google-cloud-sdk.repo > /dev/null <<EOF
[google-cloud-sdk]
name=Google Cloud SDK
baseurl=https://packages.cloud.google.com/yum/repos/cloud-sdk-el$VERSION_ID
enabled=1
gpgcheck=1
repo_gpgcheck=0
gpgkey=https://packages.cloud.google.com/yum/doc/rpm-package-key.gpg
EOF
                    sudo yum install -y google-cloud-cli >> $LOGDIR/log.apply
                elif [[ "$ID" == "centos" ]]; then
                    sudo tee -a /etc/yum.repos.d/google-cloud-sdk.repo > /dev/null <<EOF
[google-cloud-sdk]
name=Google Cloud SDK
baseurl=https://packages.cloud.google.com/yum/repos/cloud-sdk-el7
enabled=1
gpgcheck=1
repo_gpgcheck=0
gpgkey=https://packages.cloud.google.com/yum/doc/rpm-package-key.gpg
EOF
                    sudo yum install -y google-cloud-cli >> $LOGDIR/log.apply
                elif [[ "$ID" == "ubuntu" ]]; then
                export CLOUD_SDK_REPO="cloud-sdk-$(lsb_release -c -s)"
                echo "deb http://packages.cloud.google.com/apt $CLOUD_SDK_REPO main" | sudo tee -a /etc/apt/sources.list.d/google-cloud-sdk.list
                curl https://packages.cloud.google.com/apt/doc/apt-key.gpg | sudo apt-key add -
                sudo apt update >> $LOGDIR/log.apply
                sudo apt install -y google-cloud-cli >> $LOGDIR/log.apply
                fi
            fi
        fi
        ;;
    *)
        echo "Error: Unknown option $1"
        show_usage
        ;;
esac

which docker > /dev/null
ret=$?
if test $ret -ne 0 ; then
    if [[ $centos -eq 1 ]]; then
            sudo yum-config-manager --add-repo https://download.docker.com/linux/centos/docker-ce.repo
            sudo $APT install -y docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin
    elif [[ $rhel -eq 1 ]]; then
         sudo yum-config-manager --add-repo https://download.docker.com/linux/rhel/docker-ce.repo
         sudo $APT install -y docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin
    elif [[ $alinux -eq 1 ]]; then
        sudo amazon-linux-extras install docker -y
        sudo $APT install -y docker
    else
        sudo $APT install -y ca-certificates curl
        sudo install -m 0755 -d /etc/apt/keyrings
        sudo curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /etc/apt/keyrings/docker.gpg
        sudo chmod a+r /etc/apt/keyrings/docker.gpg
        echo \
  "deb [arch=$(dpkg --print-architecture) signed-by=/etc/apt/keyrings/docker.gpg] https://download.docker.com/linux/ubuntu \
  $(. /etc/os-release && echo "$VERSION_CODENAME") stable" | \
        sudo tee /etc/apt/sources.list.d/docker.list > /dev/null
        sudo $APT $UPDATE
        sudo $APT install -y docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin
    fi
fi
sudo systemctl enable docker
sudo systemctl start docker
which docker > /dev/null
ret=$?
if test $ret -ne 0 ; then
    echo "Docker installation Failed..."
    exit 1
fi
sudo usermod -aG docker ${ADMIN_USER}

file="/etc/docker/daemon.json"
tmpfile="/tmp/daemon.json"
registry="$WEBSERVER_IP:5000"
market_place=0
created=0
changed=0
if [[ -f $file && "$(sudo cat $file 2> /dev/null)" != "" ]]; then
    # Check if the registry already exists in the file
    tmp=$(sudo jq '."insecure-registries"' "$file" 2> /dev/null)
    if [ "$tmp" != "null" ]; then
        if sudo jq --arg registry "$registry" 'any(."insecure-registries"[] == $registry; .)' "$file" | grep -q true; then
            echo "The registry $registry is already present in $file" >> $LOGDIR/log.apply
        else
            changed=1
            sudo jq --arg registry "$registry" '."insecure-registries" += [$registry]' "$file" > "$tmpfile"
            sudo mv "$tmpfile" "$file"
            echo "Registry value added successfully." >> $LOGDIR/log.apply
        fi
    else
        changed=1
        sudo jq --arg registry "$registry" '."insecure-registries" = [$registry]' "$file" > "$tmpfile"
        sudo mv "$tmpfile" "$file"
        echo "Registry value added successfully." >> $LOGDIR/log.apply
    fi
else
    changed=1
    # Create a new daemon.json file
    sudo bash -c "cat << 'END' > $file
{
    \"hosts\": [\"unix:///var/run/docker.sock\",\"tcp://0.0.0.0:2375\"],
    \"insecure-registries\": [\"$registry\"]
}
END"
    echo "Daemon.json file created successfully." >> $LOGDIR/log.apply
fi

#internal debug will optimize it before production
if [ "$changed" == "1" ]; then
    sudo cat /etc/docker/daemon.json
    sudo cp /lib/systemd/system/docker.service /etc/systemd/system/
    sudo bash -c "sed -i 's/\ -H\ fd:\/\///g' /etc/systemd/system/docker.service"
    sudo systemctl daemon-reload
    sudo service docker restart
fi

sudo su - "${SPILLBOX_USER}" -c "docker pull registry" >> $LOGDIR/log.apply
sudo su - "${SPILLBOX_USER}" -c "docker run -d -p 5000:5000 --restart=always --name spillbox-registry registry" >> $LOGDIR/log.apply

rsync -az $DIR/stacks/ $DIR/${TAGNAME}/
mkdir -p $DIR/${TAGNAME}/vars
mkdir -p $DIR/.ssh
touch ${DIR}/.ssh/${TAGNAME}-auto-key.pub
touch ${DIR}/.ssh/${TAGNAME}-auto-key.pem

PROJECT_PUBLIC_KEY=${DIR}/.ssh/${TAGNAME}-auto-key.pub
PROJECT_PRIVATE_KEY=${DIR}/.ssh/${TAGNAME}-auto-key.pem

if [ "${CLOUD_PROVIDER}" == "aws" ]; then
    # Extracting the DOCKER BASE AMI ID
    if [ "${DOCKER_BASEAMI_ID}" != "" ]; then
        SNAPSHOT_ID=$(aws ec2 describe-images --image-ids "$DOCKER_BASEAMI_ID" \
            --query "Images[0].BlockDeviceMappings[0].Ebs.SnapshotId" --output text)
        if [[ -z "$SNAPSHOT_ID" || "$SNAPSHOT_ID" == "None" ]]; then
            echo "Error: Unable to retrieve snapshot ID from AMI $DOCKER_BASEAMI_ID."
            exit 1
        fi
        IS_MARKETPLACE=$(aws ec2 describe-images --image-ids "$DOCKER_BASEAMI_ID" --query "Images[?ProductCodes].ProductCodes[].ProductCodeType" --output text)
        if [[ "$IS_MARKETPLACE" == "marketplace" ]]; then
            market_place=1
        else
            market_place=0
        fi
    fi
fi

#extracting values from yaml file
TFVAR_FILE="${DIR}/${TAGNAME}/vars/terraform.tfvars"
rm -rf $TFVAR_FILE
echo "region                        = \"${REGION}\""     >> ${TFVAR_FILE}
echo "cloud_ssh_port                = \"22\""                        >> ${TFVAR_FILE}
echo "secrets                       = \"${DIR}/.ssh\""               >> ${TFVAR_FILE}
echo "vpc_id                        = \"${VPC_ID}\""                 >> ${TFVAR_FILE}
echo "subnet_id                     = \"${SUBNET_ID}\""              >> ${TFVAR_FILE}
echo "security_group_id             = \"${SECURITY_GROUP_ID}\""      >> ${TFVAR_FILE}
echo "snapshot_id                   = \"${SNAPSHOT_ID}\""            >> ${TFVAR_FILE}
echo "market_place                  = \"${market_place}\""           >> ${TFVAR_FILE}
echo "kms_key_id                    = \"${KMS_KEY_ID}\""             >> ${TFVAR_FILE}
echo "docker_ami_id                 = \"${DOCKER_BASEAMI_ID}\""      >> ${TFVAR_FILE}
echo "ami_user                      = \"${ADMIN_USER}\""             >> ${TFVAR_FILE}
echo "imagetag                      = \"${TAGNAME}\""                >> ${TFVAR_FILE}
echo "image_instance_type           = \"${IMAGE_INSTANCE_TYPE}\""    >> ${TFVAR_FILE}
echo "image_root_volume_size        = \"${IMAGE_ROOT_VOLUME_SIZE}\"" >> ${TFVAR_FILE}
echo "service_userid                = \"$SERVICE_USERID\""           >> ${TFVAR_FILE}
echo "webserverip                   = \"$WEBSERVER_IP\""             >> ${TFVAR_FILE}
case "$CLOUD_PROVIDER" in
    azure)
        echo "resource_group        = \"${RESOURCE_GROUP}\""         >> ${TFVAR_FILE}
        echo "docker_base_ami_plan  = \"\""                          >> ${TFVAR_FILE}
        ;;
    google)
        echo "zone                  = \"${ZONE}\""                   >> ${TFVAR_FILE}
        echo "project               = \"${GOOGLE_PROJECT}\""         >> ${TFVAR_FILE}
        echo "google_credentials    = \"${GOOGLE_APPLICATION_CREDENTIALS}\""         >> ${TFVAR_FILE}
        ;;
esac

case "$CLOUD_PROVIDER" in
    aws)
        cat << END > $DIR/${TAGNAME}/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/../../plugin"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
        FILE_PATH="$DIR/../../plugin/registry.terraform.io/hashicorp/aws/5.89.0/linux_amd64/terraform-provider-aws_v5.89.0_x5"
        PLUGIN_DIR_PATH=$(dirname "$FILE_PATH")
        mkdir -p "$PLUGIN_DIR_PATH"
        # Check if the file exists
        if [ ! -f "$FILE_PATH" ]; then
            echo "Downloading AWS Plugin..."
            curl -s -o "$FILE_PATH" "https://s3.us-west-1.amazonaws.com/spillbox.io-internal/plugins/registry.terraform.io/hashicorp/aws/5.89.0/linux_amd64/terraform-provider-aws_v5.89.0_x5"
        fi
        chmod +x "$FILE_PATH"
        export TF_CLI_CONFIG_FILE="$DIR/${TAGNAME}/.terraformrc"
        (cd $DIR/${TAGNAME}/provider/aws/ssh_key_pair; $TERRAFORM_DIR/terraform init ;\
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        fi
        keyname=$(cd $DIR/${TAGNAME}/provider/aws/ssh_key_pair ; $TERRAFORM_DIR/terraform output keyname | sed 's/"//g')
        echo "key_name=\"${keyname}\"" >> ${TFVAR_FILE}
        echo "Private SSH Key saved to: "
        echo "   ${DIR}/.ssh/${TAGNAME}-auto-key.pem"
        (cd $DIR/${TAGNAME}/provider/aws/ami_to_docker; $TERRAFORM_DIR/terraform init  -backend-config="region=${REGION}" ; \
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        else
            created=1
        fi
    ;;
    azure)
        # Creating a .terraformrc configuration file to define custom provider installation settings.
                cat << END > $DIR/${TAGNAME}/provider/azure/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/../../plugin"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
        FILE_PATH="$DIR/../../plugin/registry.terraform.io/hashicorp/azurerm/4.17.0/linux_amd64/terraform-provider-azurerm_v4.17.0_x5"
        PLUGIN_DIR_PATH=$(dirname "$FILE_PATH")
        mkdir -p "$PLUGIN_DIR_PATH"
        # Check if the file exists
        if [ ! -f "$FILE_PATH" ]; then
            curl -s -o "$FILE_PATH" "https://s3.us-west-1.amazonaws.com/spillbox.io-internal/plugins/registry.terraform.io/hashicorp/azurerm/4.17.0/linux_amd64/terraform-provider-azurerm_v4.17.0_x5"
        fi
        chmod +x "$FILE_PATH"
        export TF_CLI_CONFIG_FILE="$DIR/${TAGNAME}/provider/azure/.terraformrc"
        (cd $DIR/${TAGNAME}/provider/azure/ssh_key_pair; $TERRAFORM_DIR/terraform init ;\
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        fi
        keyname=$(cd $DIR/${TAGNAME}/provider/azure/ssh_key_pair ; $TERRAFORM_DIR/terraform output keyname | sed 's/"//g')
        echo "key_name                      =\"${keyname}\""                   >> ${TFVAR_FILE}
        echo "project_public_key            = \"${PROJECT_PUBLIC_KEY}\""       >> ${TFVAR_FILE}
        echo "project_private_key           = \"${PROJECT_PRIVATE_KEY}\""      >> ${TFVAR_FILE}
        echo "Private SSH Key saved to: "
        echo "   ${DIR}/.ssh/${TAGNAME}-auto-key.pem"
        (cd $DIR/${TAGNAME}/provider/azure/ami_to_docker; $TERRAFORM_DIR/terraform init ; \
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        else
            created=1
        fi
    ;;
    google)
        cat << END > $DIR/${TAGNAME}/provider/google/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/../../plugin"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
        FILE_PATH="$DIR/../../plugin/registry.terraform.io/hashicorp/google/6.19.0/linux_amd64/terraform-provider-google_v6.19.0_x5"
        PLUGIN_DIR_PATH=$(dirname "$FILE_PATH")
        mkdir -p "$PLUGIN_DIR_PATH"
        # Check if the file exists
        if [ ! -f "$FILE_PATH" ]; then
            curl -s -o "$FILE_PATH" "https://s3.us-west-1.amazonaws.com/spillbox.io-internal/plugins/registry.terraform.io/hashicorp/google/6.19.0/linux_amd64/terraform-provider-google_v6.19.0_x5"
        fi
        chmod +x "$FILE_PATH"
        export TF_CLI_CONFIG_FILE="$DIR/${TAGNAME}/provider/google/.terraformrc"
        (cd $DIR/${TAGNAME}/provider/google/ssh_key_pair; $TERRAFORM_DIR/terraform init ;\
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        fi
        keyname=$(cd $DIR/${TAGNAME}/provider/google/ssh_key_pair ; $TERRAFORM_DIR/terraform output keyname | sed 's/"//g')
        echo "key_name                      =\"${keyname}\""                 >> ${TFVAR_FILE}
        echo "project_public_key            = \"${PROJECT_PUBLIC_KEY}\""       >> ${TFVAR_FILE}
        echo "project_private_key           = \"${PROJECT_PRIVATE_KEY}\""      >> ${TFVAR_FILE}
        echo "Private SSH Key saved to: "
        echo "   ${DIR}/.ssh/${TAGNAME}-auto-key.pem"
        (cd $DIR/${TAGNAME}/provider/google/ami_to_docker; $TERRAFORM_DIR/terraform init ; \
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        else
            created=1
        fi
    ;;
esac
}

ami_delete() {
    case "$CLOUD_PROVIDER" in
        aws)
            (cd $DIR/${TAGNAME}/provider/aws/ami_to_docker; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "Apply failed" >&2
                exit ${PIPESTATUS[0]}
            fi
            (cd $DIR/${TAGNAME}/provider/aws/ssh_key_pair; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "SSH-KEY pair Deletion Failed ..." >&2
                exit ${PIPESTATUS[0]}
            fi
            rm -rf $DIR/${TAGNAME}
            echo "Success"
        ;;
        azure)
            (cd $DIR/${TAGNAME}/provider/azure/ami_to_docker; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "Apply failed" >&2
                exit ${PIPESTATUS[0]}
            fi
            (cd $DIR/${TAGNAME}/provider/azure/ssh_key_pair; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "SSH-KEY pair Deletion Failed ..." >&2
                exit ${PIPESTATUS[0]}
            fi
            rm -rf $DIR/${TAGNAME}
            echo "Success"
        ;;
        google)
           (cd $DIR/${TAGNAME}/provider/google/ami_to_docker; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "Apply failed" >&2
                exit ${PIPESTATUS[0]}
            fi
            (cd $DIR/${TAGNAME}/provider/google/ssh_key_pair; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "SSH-KEY pair Deletion Failed ..." >&2
                exit ${PIPESTATUS[0]}
            fi
            rm -rf $DIR/${TAGNAME}
            echo "Success"
        ;;
    esac

}

if [[ "$NODESTAGE" == "create" ]]; then
    ami_create
elif [[ "$NODESTAGE" == "delete" ]]; then
    ami_delete
else
    show_usage
fi

if [ "$created" == "1" ]; then
    DOCKER_IMAGE_ID=$(sudo docker inspect --format='{{.Id}}' $WEBSERVER_IP:5000/spillbox)
    DOCKER_IMAGE_INSPECT=$(sudo docker images --no-trunc | grep ${DOCKER_IMAGE_ID})
    if [[ ! -z "$DOCKER_IMAGE_ID" && ! -z "$DOCKER_IMAGE_INSPECT" ]]; then
        echo "Destroying the Image Machine"
        ami_delete
        echo "Update this Docker image:: $WEBSERVER_IP:5000/spillbox"
    else
        echo "Docker Image Failed"
    fi
fi
