#!/bin/bash -f

show_usage() {
    echo "Usage: $0 <provider> <tagname> <create/delete>"
    exit 1
}

if [[ $# -lt 3 ]]; then
    show_usage
fi

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
CLOUD_PROVIDER="$1"
TAGNAME="$2"
NODESTAGE="$3"
TERRAFORM_DIR="$DIR/../../bin"
LOCK_TIMEOUT="20m"
CONFIG=""

if [ -f $DIR/spillbox_image.config ]; then
    CONFIG=$DIR/spillbox_image.config
else
    echo "ERROR : $DIR/spillbox_image.config : No config file detected!!!"  >&2
    exit 1
fi

. $CONFIG

LOGDIR="$DIR/${TAGNAME}/logs"
mkdir -p $LOGDIR

case "$CLOUD_PROVIDER" in
    aws)
        if [[ -z "$AWS_ACCESS_KEY_ID" || -z "$AWS_SECRET_ACCESS_KEY" ]]; then
            echo "Cloud Crendentials missing"  >&2
            echo "Check if these varaiables are set : AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY "  >&2
            exit 3
         fi
         ;;
    azure)
        if [[ -z "$ARM_CLIENT_ID" || -z "$ARM_CLIENT_SECRET" || -z "$ARM_SUBSCRIPTION_ID" || -z "$ARM_TENANT_ID"  ]]; then
            echo "Cloud Crendentials missing" >&2
            echo "Check if these varaiables are set : ARM_CLIENT_ID  ARM_CLIENT_SECRET ARM_SUBSCRIPTION_ID ARM_TENANT_ID" >&2
            echo "Do az login " >&2
            exit 3
        fi
        ;;
    google)
        if [[ -z "$GOOGLE_PROJECT" || -z "$GOOGLE_APPLICATION_CREDENTIALS" ]]; then
            echo "Cloud Crendentials missing" >&2
            echo "Check if these varaiables are set : GOOGLE_PROJECT GOOGLE_APPLICATION_CREDENTIALS" >&2
            exit 3
        fi
        ;;
    *)
        usage;
        exit 3
        ;;
esac

if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS_NAME=$ID
    OS_VERSION=$VERSION_ID
else
    echo "Unsupported OS"
    exit 1
fi

if [[ "$OS_NAME" == "centos" ]] || [[ "$OS_NAME" == "rhel" ]] || [[ "$OS_NAME" == "amzn" ]] || [[ "$OS_NAME" == "rocky" ]]; then
        APT="yum"
elif [[ "$OS_NAME" == "ubuntu" ]]; then
        APT="apt-get"
fi

#Installing the jq package
which jq > /dev/null
ret=$?
if test $ret -ne 0 ; then
    echo "Installing jq package : " | tee -a $LOGDIR/log.apply
    sudo $APT install -y wget   >> $LOGDIR/log.apply
    if [ "$OS_NAME" == "rhel" ]; then
        rhel_version=$(rpm -E %{rhel})
        wget -P /tmp https://dl.fedoraproject.org/pub/epel/epel-release-latest-${rhel_version}.noarch.rpm  >> $LOGDIR/log.apply
        sudo $APT install -y /tmp/epel-release-latest-${rhel_version}.noarch.rpm  >> $LOGDIR/log.apply
    else
        sudo $APT install -y epel-release  >> $LOGDIR/log.apply
    fi
    sudo $APT install -y jq 2>&1 >> $LOGDIR/log.apply | tee -a $LOGDIR/log.apply
fi

# Installing rsync
which rsync > /dev/null
ret=$?
if [ $ret -ne 0 ]; then
    echo "Installing rsync package..." | tee -a "$LOGDIR/log.apply"
    sudo $APT install -y rsync >> "$LOGDIR/log.apply" 2>&1 | tee -a "$LOGDIR/log.apply"
fi

# Check if unzip is installed
if ! command -v unzip &> /dev/null; then
    echo "unzip is not installed. Installing..."
    sudo $APT install -y unzip  >> $LOGDIR/log.apply
fi

#Installing Cloud cli which will be used to converting mach to image
case "$CLOUD_PROVIDER" in
    aws)
        # Check if AWS CLI is installed, if not, install it
        if ! command -v aws &> /dev/null; then
            echo  "AWS CLI is not installed. Installing..."
            curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip" >> $LOGDIR/log.apply
            unzip awscliv2.zip >> $LOGDIR/log.apply
            sudo ./aws/install >> $LOGDIR/log.apply
        fi
        ;;
    google)
        if ! command -v gcloud &> /dev/null; then
            if [[ "$OS_NAME" == "rhel" && ( "$OS_VERSION" == 8* || "$OS_VERSION" == 9* ) ]]; then
                sudo tee -a /etc/yum.repos.d/google-cloud-sdk.repo > /dev/null <<EOF
[google-cloud-sdk]
name=Google Cloud SDK
baseurl=https://packages.cloud.google.com/yum/repos/cloud-sdk-el$OS_VERSION
enabled=1
gpgcheck=1
repo_gpgcheck=0
gpgkey=https://packages.cloud.google.com/yum/doc/rpm-package-key.gpg
EOF
                sudo yum install -y google-cloud-cli >> $LOGDIR/log.apply
            elif [[ "$OS_NAME" == "centos" ]]; then
                sudo tee -a /etc/yum.repos.d/google-cloud-sdk.repo > /dev/null <<EOF
[google-cloud-sdk]
name=Google Cloud SDK
baseurl=https://packages.cloud.google.com/yum/repos/cloud-sdk-el7
enabled=1
gpgcheck=1
repo_gpgcheck=0
gpgkey=https://packages.cloud.google.com/yum/doc/rpm-package-key.gpg
EOF
                sudo yum install -y google-cloud-cli >> $LOGDIR/log.apply
            elif [[ "$OS_NAME" == "ubuntu" ]]; then
                export CLOUD_SDK_REPO="cloud-sdk-$(lsb_release -c -s)"
                echo "deb http://packages.cloud.google.com/apt $CLOUD_SDK_REPO main" | sudo tee -a /etc/apt/sources.list.d/google-cloud-sdk.list
                curl https://packages.cloud.google.com/apt/doc/apt-key.gpg | sudo apt-key add -
                sudo apt update >> $LOGDIR/log.apply
                sudo apt install -y google-cloud-cli >> $LOGDIR/log.apply
            fi
        fi
        ;;
    azure)
        if ! command -v az &> /dev/null; then
            if [[ "$OS_NAME" == "rhel" && ( "$OS_VERSION" == 8* ) ]]; then
                sudo yum -y install dnf
                sudo rpm --import https://packages.microsoft.com/keys/microsoft.asc >> $LOGDIR/log.apply
                sudo dnf install -y https://packages.microsoft.com/config/rhel/8/packages-microsoft-prod.rpm >> $LOGDIR/log.apply
                sudo dnf install -y azure-cli >> $LOGDIR/log.apply
            elif  [[ "$OS_NAME" == "rhel" && ( "$OS_VERSION" == 9* ) ]] ; then
                sudo yum -y install dnf  >> $LOGDIR/log.apply
                sudo rpm --import https://packages.microsoft.com/keys/microsoft.asc >> $LOGDIR/log.apply
                sudo dnf install -y https://packages.microsoft.com/config/rhel/9.0/packages-microsoft-prod.rpm >> $LOGDIR/log.apply
                sudo dnf install -y azure-cli >> $LOGDIR/log.apply
            elif [[ "$OS_NAME" == "ubuntu" ]]; then
                curl -sL https://aka.ms/InstallAzureCLIDeb | sudo bash >> $LOGDIR/log.apply
            fi
            if [[ "$OS_NAME" == "centos" ]]; then
                sudo $APT install -y dnf  >> $LOGDIR/log.apply
                sudo rpm --import https://packages.microsoft.com/keys/microsoft.asc
    echo -e "[azure-cli]
name=Azure CLI
baseurl=https://packages.microsoft.com/yumrepos/azure-cli
enabled=1
gpgcheck=1
gpgkey=https://packages.microsoft.com/keys/microsoft.asc" | sudo tee /etc/yum.repos.d/azure-cli.repo
                sudo dnf install -y azure-cli  >> $LOGDIR/log.apply
            fi
        fi
        ;;
esac

ami_create() {
        echo "Detected OS: $OS_NAME"
        echo "Detected OS Version: $OS_VERSION"
        PKG_SCRIPT=$DIR/${TAGNAME}/install/pkg_script.sh
        mkdir -p $DIR/${TAGNAME}/install
        rm -rf $DIR/${TAGNAME}/install/pkg_script.sh
        echo "Log File: "
        echo "  - $DIR/${TAGNAME}/logs/log.apply"
        mkdir -p $DIR/${TAGNAME}
        rsync -az $DIR/stacks/ $DIR/${TAGNAME}/
        mkdir -p $DIR/${TAGNAME}/vars
        mkdir -p $DIR/.ssh
        touch ${DIR}/.ssh/${TAGNAME}-auto-key.pub
        touch ${DIR}/.ssh/${TAGNAME}-auto-key.pem
        PROJECT_PUBLIC_KEY=${DIR}/.ssh/${TAGNAME}-auto-key.pub
        PROJECT_PRIVATE_KEY=${DIR}/.ssh/${TAGNAME}-auto-key.pem

        case "$OS_NAME" in
            ubuntu)
                dpkg-query -W -f='${Package}=${Version}\n' > "$DIR/${TAGNAME}/install/installed_packages.txt"
                echo "sudo apt install -y $(tr '\n' ' ' < "$DIR/${TAGNAME}/install/installed_packages.txt")" >> "$PKG_SCRIPT"

                REPO_BACKUP_DIR="$DIR/${TAGNAME}/install"
                GPG_BACKUP_DIR="$DIR/${TAGNAME}/install"
                REPO_DIR="/etc/apt/sources.list.d"
                GPG_DIR="/etc/apt/trusted.gpg.d"

                mkdir -p "$REPO_BACKUP_DIR" "$GPG_BACKUP_DIR"
                sudo cp -r "$REPO_DIR" "$REPO_BACKUP_DIR/"
                sudo cp -r "$GPG_DIR" "$GPG_BACKUP_DIR/"
                ;;
            centos|rhel|rocky|amzn)
                rpm -qa --qf "%{NAME}-%{VERSION}\n" | grep -v '^gpg-pubkey' > "$DIR/${TAGNAME}/install/installed_packages.txt"
                echo "sudo yum install -y $(tr '\n' ' ' < $DIR/${TAGNAME}/install/installed_packages.txt)" >> "$PKG_SCRIPT"

                REPO_BACKUP_DIR="$DIR/${TAGNAME}/install"
                GPG_BACKUP_DIR="$DIR/${TAGNAME}/install"
                REPO_DIR="/etc/yum.repos.d"
                GPG_DIR="/etc/pki/rpm-gpg"

                mkdir -p "$REPO_BACKUP_DIR" "$GPG_BACKUP_DIR"
                sudo cp -r "$REPO_DIR" "$REPO_BACKUP_DIR/"
                sudo cp -r "$GPG_DIR" "$GPG_BACKUP_DIR/"
                ;;

            *)
                echo "Unsupported OS"
                exit 1
                ;;
        esac

        #extracting values from yaml file
        TFVAR_FILE="${DIR}/${TAGNAME}/vars/terraform.tfvars"
        rm -rf $TFVAR_FILE
        echo "region                        =\"${REGION}\""                 >> ${TFVAR_FILE}
        echo "cloud_ssh_port                =\"22\""                        >> ${TFVAR_FILE}
        echo "vpc_id                        =\"${VPC_ID}\""                 >> ${TFVAR_FILE}
        echo "subnet_id                     =\"${SUBNET_ID}\""              >> ${TFVAR_FILE}
        echo "securitygroup_id             =\"${SECURITY_GROUP_ID}\""       >> ${TFVAR_FILE}
        echo "kms_key_id                    =\"${KMS_KEY_ID}\""             >> ${TFVAR_FILE}
        echo "docker_ami_id                 =\"${DOCKER_BASEAMI_ID}\""      >> ${TFVAR_FILE}
        echo "ami_user                      =\"${ADMIN_USER}\""             >> ${TFVAR_FILE}
        echo "image_instance_type           =\"${IMAGE_INSTANCE_TYPE}\""    >> ${TFVAR_FILE}
        echo "image_root_volume_size        =\"${IMAGE_ROOT_VOLUME_SIZE}\"" >> ${TFVAR_FILE}
        echo "service_userid                =\"$SERVICE_USERID\""           >> ${TFVAR_FILE}
        echo "imagetag                      =\"${TAGNAME}\""                >> ${TFVAR_FILE}
        echo "secrets                       = \"${DIR}/.ssh\""              >> ${TFVAR_FILE}
        echo "repo_dir                     =\"${REPO_DIR}\""                >> ${TFVAR_FILE}
        echo "gpg_dir                      =\"${GPG_DIR}\""                 >> ${TFVAR_FILE}

        case "$CLOUD_PROVIDER" in
                azure)
                    echo "resource_group        = \"${RESOURCE_GROUP}\""    >> ${TFVAR_FILE}
                    echo "docker_base_ami_plan  = \"\""                     >> ${TFVAR_FILE}
                    ;;
                google)
                    echo "zone                  = \"${ZONE}\""                                  >> ${TFVAR_FILE}
                    echo "project               = \"${GOOGLE_PROJECT}\""                        >> ${TFVAR_FILE}
                    echo "google_credentials    = \"${GOOGLE_APPLICATION_CREDENTIALS}\""        >> ${TFVAR_FILE}
                    ;;
        esac

case "$CLOUD_PROVIDER" in
        aws)
            cat << END > $DIR/${TAGNAME}/provider/aws/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/../../plugin"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
        FILE_PATH="$DIR/../../plugin/registry.terraform.io/hashicorp/aws/5.89.0/linux_amd64/terraform-provider-aws_v5.89.0_x5"
        PLUGIN_DIR_PATH=$(dirname "$FILE_PATH")
        mkdir -p "$PLUGIN_DIR_PATH"
        # Check if the file exists
        if [ ! -f "$FILE_PATH" ]; then
            echo "Downloading AWS Plugin..."
            curl -s -o "$FILE_PATH" "https://s3.us-west-1.amazonaws.com/spillbox.io-internal/plugins/registry.terraform.io/hashicorp/aws/5.89.0/linux_amd64/terraform-provider-aws_v5.89.0_x5" >> $LOGDIR/log.apply
        fi
        chmod +x "$FILE_PATH"
        export TF_CLI_CONFIG_FILE="$DIR/${TAGNAME}/provider/aws/.terraformrc"
        (cd $DIR/${TAGNAME}/provider/aws/ssh_key_pair; $TERRAFORM_DIR/terraform init ;\
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        fi
        key_name=$(cd $DIR/${TAGNAME}/provider/aws/ssh_key_pair ; $TERRAFORM_DIR/terraform output keyname | sed 's/"//g')
        echo "key_name=\"${key_name}\"" >> ${TFVAR_FILE}
        echo "Private SSH Key saved to: "
        echo "   ${DIR}/.ssh/${TAGNAME}-auto-key.pem"
        (cd $DIR/${TAGNAME}/provider/aws/host_to_ami; $TERRAFORM_DIR/terraform init -backend-config="region=${REGION}" ; \
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
             echo "apply failed" >&2
             exit ${PIPESTATUS[0]}
        fi
        jsondump=$(cd $DIR/${TAGNAME}/provider/aws/host_to_ami ; $TERRAFORM_DIR/terraform output -json)
        HOST_IMAGE_MACH_IP=$(echo $jsondump | jq -r '.priv_ip.value')
        HOST_IMAGE_ID=$(echo $jsondump | jq -r '.mach_id.value')
        echo "AMI MACH INFORMATION:"
        echo "   ADMIN_USER   : $ADMIN_USER"
        echo "   PRIVATE IP   : $HOST_IMAGE_MACH_IP"
        echo "   INSTANCE ID  : $HOST_IMAGE_ID"
        echo "  Log in to the machine and check if any packages failed to install. A list of failed packages will be available at /tmp/install/fail.log. You can manually install any additional packages on the AMI if needed."
        echo "Once everything is complete, make sure to delete the resource."
        ;;
        azure)
        # Creating a .terraformrc configuration file to define custom provider installation settings.
                cat << END > $DIR/${TAGNAME}/provider/azure/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/../../plugin"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
        FILE_PATH="$DIR/../../plugin/registry.terraform.io/hashicorp/azurerm/4.17.0/linux_amd64/terraform-provider-azurerm_v4.17.0_x5"
        PLUGIN_DIR_PATH=$(dirname "$FILE_PATH")
        mkdir -p "$PLUGIN_DIR_PATH"
        # Check if the file exists
        if [ ! -f "$FILE_PATH" ]; then
            curl -s -o "$FILE_PATH" "https://s3.us-west-1.amazonaws.com/spillbox.io-internal/plugins/registry.terraform.io/hashicorp/azurerm/4.17.0/linux_amd64/terraform-provider-azurerm_v4.17.0_x5"
        fi
        chmod +x "$FILE_PATH"
        export TF_CLI_CONFIG_FILE="$DIR/${TAGNAME}/provider/azure/.terraformrc"
        (cd $DIR/${TAGNAME}/provider/azure/ssh_key_pair; $TERRAFORM_DIR/terraform init ;\
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        fi
        keyname=$(cd $DIR/${TAGNAME}/provider/azure/ssh_key_pair ; $TERRAFORM_DIR/terraform output keyname | sed 's/"//g')
        echo "key_name                      =\"${keyname}\""                   >> ${TFVAR_FILE}
        echo "project_public_key            = \"${PROJECT_PUBLIC_KEY}\""       >> ${TFVAR_FILE}
        echo "project_private_key           = \"${PROJECT_PRIVATE_KEY}\""      >> ${TFVAR_FILE}
        echo "Private SSH Key saved to: "
        echo "   ${DIR}/.ssh/${TAGNAME}-auto-key.pem"
        (cd $DIR/${TAGNAME}/provider/azure/host_to_ami ; $TERRAFORM_DIR/terraform init ; \
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        fi
        jsondump=$(cd $DIR/${TAGNAME}/provider/azure/host_to_ami ; $TERRAFORM_DIR/terraform output -json)
        HOST_IMAGE_MACH_IP=$(echo $jsondump | jq -r '.priv_ip.value[][]')
        HOST_IMAGE_ID=$(echo $jsondump | jq -r '.mach_id.value[][]')
        echo "AMI MACH INFORMATION:"
        echo "   ADMIN_USER   : $ADMIN_USER"
        echo "   PRIVATE IP   : $HOST_IMAGE_MACH_IP"
        echo "   INSTANCE ID  : $HOST_IMAGE_ID"
        echo "Log in to the machine and check if any packages failed to install. A list of failed packages will be available at /tmp/install/fail.log. You can manually install any additional packages on the image if needed."
        echo "Once everything is complete, make sure to run 'az login' (if not already done) before deleting the resources."
    ;;
        google)
            cat << END > $DIR/${TAGNAME}/provider/google/.terraformrc
provider_installation {
  filesystem_mirror {
    path    = "$DIR/../../plugin"
    include = ["registry.terraform.io/hashicorp/*"]
  }
  direct {
    exclude = ["*/*"]
  }
}
END
        FILE_PATH="$DIR/../../plugin/registry.terraform.io/hashicorp/google/6.19.0/linux_amd64/terraform-provider-google_v6.19.0_x5"
        PLUGIN_DIR_PATH=$(dirname "$FILE_PATH")
        mkdir -p "$PLUGIN_DIR_PATH"
        # Check if the file exists
        if [ ! -f "$FILE_PATH" ]; then
            curl -s -o "$FILE_PATH" "https://s3.us-west-1.amazonaws.com/spillbox.io-internal/plugins/registry.terraform.io/hashicorp/google/6.19.0/linux_amd64/terraform-provider-google_v6.19.0_x5"
        fi
        chmod +x "$FILE_PATH"
        export TF_CLI_CONFIG_FILE="$DIR/${TAGNAME}/provider/google/.terraformrc"
        (cd $DIR/${TAGNAME}/provider/google/ssh_key_pair; $TERRAFORM_DIR/terraform init ;\
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
            exit ${PIPESTATUS[0]}
        fi
        keyname=$(cd $DIR/${TAGNAME}/provider/google/ssh_key_pair ; $TERRAFORM_DIR/terraform output keyname | sed 's/"//g')
        echo "key_name                      =\"${keyname}\""                 >> ${TFVAR_FILE}
        echo "project_public_key            = \"${PROJECT_PUBLIC_KEY}\""       >> ${TFVAR_FILE}
        echo "project_private_key           = \"${PROJECT_PRIVATE_KEY}\""      >> ${TFVAR_FILE}
        echo "Private SSH Key saved to: "
        echo "   ${DIR}/.ssh/${TAGNAME}-auto-key.pem"
        (cd $DIR/${TAGNAME}/provider/google/host_to_ami; $TERRAFORM_DIR/terraform init ; \
        $TERRAFORM_DIR/terraform apply -auto-approve -refresh=true) 2>&1 >> $DIR/${TAGNAME}/logs/log.apply | tee -a $DIR/${TAGNAME}/logs/log.apply
        if test ${PIPESTATUS[0]} -ne 0 ; then
            echo "apply failed" >&2
           exit ${PIPESTATUS[0]}
        fi
        jsondump=$(cd $DIR/${TAGNAME}/provider/google/host_to_ami ; $TERRAFORM_DIR/terraform output -json)
        HOST_IMAGE_MACH_IP=$(echo $jsondump | jq -r '.priv_ip.value[][]')
        HOST_IMAGE_ID=$(echo $jsondump | jq -r '.mach_id.value')
        echo "AMI MACH INFORMATION:"
        echo "   ADMIN_USER   : $ADMIN_USER"
        echo "   PRIVATE IP   : $HOST_IMAGE_MACH_IP"
        echo "   INSTANCE ID  : $HOST_IMAGE_ID"
        echo "  Log in to the machine and check if any packages failed to install. A list of failed packages will be available at /tmp/install/fail.log. You can manually install any additional packages on the AMI if needed."
        echo "  Once everything is complete, make sure to delete the resource."
    ;;
esac
}

ami_delete() {
    case "$CLOUD_PROVIDER" in
        aws)
            echo "Converting machine to image"
            jsondump=$(cd $DIR/${TAGNAME}/provider/aws/host_to_ami ; $TERRAFORM_DIR/terraform output -json)
            HOST_IMAGE_ID=$(echo $jsondump | jq -r '.mach_id.value')
            tmp=$(aws ec2 describe-images --filters "Name=name,Values=spillbox-$TAGNAME-hostimage" --query "Images[*].[ImageId]" --output text --region $REGION)
            if [ "$tmp" != "" ]; then
                echo "spillbox-${TAGNAME}-hostimage Image already exist (AMI ID : ${tmp})"
                exit 3
            fi
            aws ec2 create-image --instance-id "${HOST_IMAGE_ID}" --name "spillbox-$TAGNAME-hostimage" --output text --region $REGION \
            --description "Image with host"
            RET=$?
            sleep 20
            if [ $RET -eq 0 ]; then
                echo "Image creation processing..."
                SPILLBOX_HOSTAMI_ID=$(aws ec2 describe-images --filters "Name=name,Values=spillbox-$TAGNAME-hostimage" --output text --query 'Images[*].[ImageId]' --region $REGION)
            else
                echo "Error: Image creation failed. Exit status: $aws_exit_status" 2>&1 | tee /dev/stderr
                exit 1
            fi
            while true; do
                STATE=$(aws ec2 describe-images --image-ids $SPILLBOX_HOSTAMI_ID --query 'Images[0].State' --output text --region $REGION ) 
                if [ "$STATE" == "available" ]; then
                    break
                elif [ "$STATE" == "failed" ]; then
                    echo "AWS build image failed " 2>&1 | tee /dev/stderr
                    exit 1
                else
                    echo -n ".."
                    sleep 60
                fi
            done
            echo ""
            echo "Image created : spillbox-$TAGNAME-hostimage"
            echo ""
            echo "Destroying Host Machine"
            echo "Log File: "
            echo "  -  $DIR/${TAGNAME}/logs/log.destroy"
            export TF_CLI_CONFIG_FILE="$DIR/${TAGNAME}/provider/aws/.terraformrc"
            (cd $DIR/${TAGNAME}/provider/aws/host_to_ami; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "Apply failed" >&2
                exit ${PIPESTATUS[0]}
            fi
            (cd $DIR/${TAGNAME}/provider/aws/ssh_key_pair; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "SSH-KEY pair Deletion Failed ..." >&2
                exit ${PIPESTATUS[0]}
            fi
            sudo rm -rf $DIR/${TAGNAME}
            rm -rf ${DIR}/.ssh/${TAGNAME}-auto-key.pem
            rm -rf ${DIR}/.ssh/${TAGNAME}-auto-key.pub
            echo "Success"
            ;;
        azure)
            echo "Converting machine to image"
            tmp=$(az image show --name spillbox-$TAGNAME-hostimage --resource-group $RESOURCE_GROUP 2>> $LOGDIR/log.apply |  jq -r '.id')
            if [[ "$tmp" != "" ]] ; then
                echo "Image spillbox-$TAGNAME-hostimage already exist"
                exit 3
            fi
            AZ_OS_DISK=$(az vm show -g "$RESOURCE_GROUP" -n spillbox-${TAGNAME}-hostimage --query "storageProfile.osDisk.name" -o tsv)
            HYPER_GEN=$(az disk show -g "$RESOURCE_GROUP" -n $AZ_OS_DISK --query "hyperVGeneration" -o tsv)
            az vm deallocate -g "$RESOURCE_GROUP" -n "spillbox-${TAGNAME}-hostimage"
            az vm generalize -g "$RESOURCE_GROUP" -n "spillbox-${TAGNAME}-hostimage"
            az image create -n "spillbox-${TAGNAME}-hostimage" -g "$RESOURCE_GROUP" --os-type Linux --source "spillbox-${TAGNAME}-hostimage" --hyper-v-generation $HYPER_GEN
            RET=$?
            sleep 20
            if [ $RET -eq 0 ]; then
                echo "Image creation successful."
            else
                echo "Error: Image creation failed. Exit status: $RET" 2>&1 | tee /dev/stderr
                exit 1
            fi
            sleep 60
            SPILLBOX_HOSTAMI_ID=$(az image show --name spillbox-${TAGNAME}-hostimage --resource-group $RESOURCE_GROUP |  jq -r '.id')
            if [[ "$(echo ${SPILLBOX_HOSTAMI_ID} | egrep '^/subscriptions')" == "" ]]; then
                echo "Can't fetch Image Id" 2>&1 | tee /dev/stderr
                exit 3
            fi
            echo "Image created : $SPILLBOX_HOSTAMI_ID"
            echo ""
            echo "Destroying Host Machine"
            echo "Log File: "
            echo "  -  $DIR/${TAGNAME}/logs/log.destroy"
            export TF_CLI_CONFIG_FILE="$DIR/${TAGNAME}/provider/azure/.terraformrc"
            (cd $DIR/${TAGNAME}/provider/azure/host_to_ami; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "Apply failed" >&2
                exit ${PIPESTATUS[0]}
            fi
            (cd $DIR/${TAGNAME}/provider/azure/ssh_key_pair; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "SSH-KEY pair Deletion Failed ..." >&2
                exit ${PIPESTATUS[0]}
            fi
            sudo rm -rf $DIR/${TAGNAME}
            rm -rf ${DIR}/.ssh/${TAGNAME}-auto-key.pem
            rm -rf ${DIR}/.ssh/${TAGNAME}-auto-key.pub
            echo "Success"
            ;;
        google)
            echo "Converting machine to image"
            jsondump=$(cd $DIR/${TAGNAME}/provider/google/host_to_ami ; $TERRAFORM_DIR/terraform output -json)
            image_machzone=$(echo $jsondump | jq -r '.imagezone.value[]')
            tmp=$(gcloud compute images describe spillbox-$TAGNAME-hostimage --project=$GOOGLE_PROJECT  --format=json 2>> $LOGDIR/log.apply | jq -r '.status') 
            if [[ "$tmp" != "" ]] ; then
                echo "Image spillbox-$TAGNAME-hostimage already exist"
                exit 3
            fi
            gcloud compute images create spillbox-$TAGNAME-hostimage --source-disk=spillbox-$TAGNAME-hostimage --source-disk-zone=$image_machzone --project=$GOOGLE_PROJECT ${KMS_KEY_ID:+--kms-key=$KMS_KEY_ID} --force
            RET=$?
            if test $RET -ne 0 ; then
                echo "Image creation Failed..." 2>&1 | tee /dev/stderr
                exit 3
            fi
            echo "Image created : spillbox-$TAGNAME-hostimage"
            echo ""
            echo "Destroying Host Machine"
            echo "Log File: "
            echo "  -  $DIR/${TAGNAME}/logs/log.destroy"
            export TF_CLI_CONFIG_FILE="$DIR/${TAGNAME}/provider/google/.terraformrc"
            (cd $DIR/${TAGNAME}/provider/google/host_to_ami; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "Apply failed" >&2
                exit ${PIPESTATUS[0]}
            fi
            (cd $DIR/${TAGNAME}/provider/google/ssh_key_pair; $TERRAFORM_DIR/terraform destroy -lock-timeout=$LOCK_TIMEOUT -auto-approve) 2>&1 >> $DIR/${TAGNAME}/logs/log.destroy | tee -a $DIR/${TAGNAME}/logs/log.destroy
            if test ${PIPESTATUS[0]} -ne 0 ; then
                echo "SSH-KEY pair Deletion Failed ..." >&2
                exit ${PIPESTATUS[0]}
            fi
            sudo rm -rf $DIR/${TAGNAME}
            rm -rf ${DIR}/.ssh/${TAGNAME}-auto-key.pem
            rm -rf ${DIR}/.ssh/${TAGNAME}-auto-key.pub
            echo "Success"
            ;;
        esac
}

if [[ "$NODESTAGE" == "create" ]]; then
    ami_create
elif [[ "$NODESTAGE" == "delete" ]]; then
    ami_delete
else
    show_usage
fi
