provider "aws" {
  region = var.region
}

locals {
  key_name = "${var.imagetag}-auto-key"
  private_key_filename = format("%s/%s.pem", var.secrets, local.key_name)
}

resource "aws_instance" "docker_image" {
  ami                    = var.docker_ami_id
  instance_type          = var.image_instance_type
  subnet_id              = var.subnet_id
  vpc_security_group_ids = [var.security_group_id]
  key_name               = var.key_name
  root_block_device {
    volume_size           = var.image_root_volume_size
    kms_key_id            = var.kms_key_id == "" ? null : var.kms_key_id
    encrypted             = var.kms_key_id == "" ? false : true
    delete_on_termination = true
  }
  tags = {
    Name = "spillbox-dockerimage"
  }
}

resource "aws_ebs_volume" "new_volume" {
  depends_on = [aws_instance.docker_image]
  snapshot_id        = var.snapshot_id
  size               = var.image_root_volume_size
  availability_zone  = aws_instance.docker_image.availability_zone
  kms_key_id         = var.kms_key_id == "" ? null : var.kms_key_id
  encrypted          = var.kms_key_id == "" ? false : true
  tags = {
    Name = "spillbox-docker"
  }
}

resource "null_resource" "manage_instance_marketplace" {
  depends_on = [aws_instance.docker_image, aws_ebs_volume.new_volume]
  provisioner "local-exec" {
    command = <<EOT
      if [ ${var.market_place} -eq 1 ]; then
        aws ec2 stop-instances --instance-ids ${aws_instance.docker_image.id}
        aws ec2 wait instance-stopped --instance-ids ${aws_instance.docker_image.id}
        aws ec2 attach-volume --volume-id ${aws_ebs_volume.new_volume.id} --instance-id ${aws_instance.docker_image.id} --device /dev/sdg
        aws ec2 modify-instance-attribute --instance-id ${aws_instance.docker_image.id}  --block-device-mappings "[{\"DeviceName\": \"/dev/sdg\", \"Ebs\":{\"DeleteOnTermination\":true}}]"
        aws ec2 start-instances --instance-ids ${aws_instance.docker_image.id}
      else
        aws ec2 attach-volume --volume-id ${aws_ebs_volume.new_volume.id} --instance-id ${aws_instance.docker_image.id} --device /dev/sdg
        aws ec2 modify-instance-attribute --instance-id ${aws_instance.docker_image.id}  --block-device-mappings "[{\"DeviceName\": \"/dev/sdg\", \"Ebs\":{\"DeleteOnTermination\":true}}]"
        aws ec2 start-instances --instance-ids ${aws_instance.docker_image.id}
      fi
    EOT
  }
}

resource "null_resource" "detach_volume"{
  triggers = {
    instance_id = aws_instance.docker_image.id
    volume_id   = aws_ebs_volume.new_volume.id
  }
  provisioner "local-exec" {
  when    = destroy
    command = <<EOT
      aws ec2 stop-instances --instance-ids ${self.triggers.instance_id}
      aws ec2 wait instance-stopped --instance-ids ${self.triggers.instance_id}
      aws ec2 detach-volume --volume-id ${self.triggers.volume_id}
      aws ec2 wait volume-available --volume-id ${self.triggers.volume_id}
      aws ec2 delete-volume --volume-id ${self.triggers.volume_id}
    EOT
  }
}

resource "null_resource" "build_docker_image" {
  depends_on    = [null_resource.manage_instance_marketplace]
  connection {
    user        = var.ami_user
    private_key = file(local.private_key_filename)
    host        = element(flatten(aws_instance.docker_image.*.private_ip), 0)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }

  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_build.sh ; /tmp/install/docker_build.sh '${var.webserverip}' '${var.service_userid}' "]
  }
}

resource "null_resource" "setup_docker_image_webserver_local" {
  depends_on = [null_resource.build_docker_image]
  provisioner "local-exec" {
     command = "chmod 755 ${path.module}/install/docker_setup.sh ; ${path.module}/install/docker_setup.sh '${var.webserverip}' '${var.service_userid}'"
  }
}
