#!/bin/bash -f

#Exit automatically when something goes wrong
set -euo pipefail

LOG_DIR="/tmp/install"
REPO_DIR="$1"
GPG_DIR="$2"
REPO_BACK_DIR=$(basename "$REPO_DIR")
GPG_BACK_DIR=$(basename "$GPG_DIR")

PKG_LIST="$LOG_DIR/installed_packages.txt"
PKG_SCRIPT="$LOG_DIR/pkg_script.sh"
SUCCESS_LOG="$LOG_DIR/success.log"
FAIL_LOG="$LOG_DIR/fail.log"

# Detect OS
if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS_NAME=$ID
else
    echo "Unable to detect OS."
    exit 1
fi

# Restore repos
echo "Restoring yum repos..."
if [ -d "$LOG_DIR/$REPO_BACK_DIR" ]; then
    sudo cp -r "$LOG_DIR/$REPO_BACK_DIR/." "$REPO_DIR"
else
    echo "Warning: Repo backup directory is empty or doesn't exist: $LOG_DIR/$REPO_BACK_DIR"
fi

# Restore GPG keys
echo "Restoring GPG keys..."
if [[ "$OS_NAME" == "amzn" || "$OS_NAME" == "rhel" || "$OS_NAME" == "centos" ]]; then
    if [ -d "$LOG_DIR/$GPG_BACK_DIR" ]; then
        sudo cp -r "$LOG_DIR/$GPG_BACK_DIR/." "$GPG_DIR"

        # Import each GPG key file found
        for key in "$GPG_DIR"/.; do
            if [ -f "$key" ]; then
                echo "Importing: $key"
                sudo rpm --import "$key" || echo "Warning: Failed to import $key"
            fi
        done
    else
        echo "Warning: GPG backup directory not found: $LOG_DIR/$GPG_BACK_DIR"
    fi
elif [[ "$OS_NAME" == "ubuntu" || "$OS_NAME" == "debian" ]]; then
    if [ -d "$LOG_DIR/$GPG_BACK_DIR" ]; then
        sudo cp -r "$LOG_DIR/$GPG_BACK_DIR/." /etc/apt/trusted.gpg.d/
        echo "GPG keys restored to /etc/apt/trusted.gpg.d/"
    else
        echo "Warning: GPG backup directory not found: $LOG_DIR/$GPG_BACK_DIR"
    fi
fi

# Ensure the package script is executable
chmod +x "$PKG_SCRIPT"

# Clean previous logs
rm -f "$SUCCESS_LOG" "$FAIL_LOG"

echo "Starting package installation..."

# Read each package and install individually
while read -r package; do
    if [[ "$OS_NAME" == "ubuntu" ]]; then
        if sudo apt-get install -y "$package"; then
            echo "$package" >> "$SUCCESS_LOG"
        else
            echo "$package" >> "$FAIL_LOG"
        fi
    elif [[ "$OS_NAME" =~ (centos|rhel|rocky|amzn) ]]; then
        if sudo yum install -y "$package"; then
            echo "$package" >> "$SUCCESS_LOG"
        else
            echo "$package" >> "$FAIL_LOG"
        fi
    else
        echo "Unsupported OS: $OS_NAME"
        exit 1
    fi
done < "$PKG_LIST"

echo " Installation finished!"
echo " Successfully installed packages: $SUCCESS_LOG"

# Optionally, list pending packages
if [[ -s "$FAIL_LOG" ]]; then
    echo ""
    echo " ERROR: The following packages failed to install:"
    cat "$FAIL_LOG"
fi
