terraform {
  required_version = ">= 1.10.4, < 1.10.5"
  required_providers {
    azurerm = {
      version = "4.17.0"
    }
    null = {
      version = "3.2.3"
    }
    template = {
      version = "2.2.0"
    }
  }
}

provider "azurerm" {
  features {
    resource_group {
      prevent_deletion_if_contains_resources = false
    }
  }
}

locals {
  docker_ami_name   = var.docker_ami_id == "" ? null : var.docker_ami_id
  docker_is_image_gallery  = can(regex("Microsoft.Compute/galleries/([^/]+)/images/([^/]+)", var.docker_ami_id)) ? true : false
  docker_is_image_custom   = can(regex("Microsoft.Compute/images/([^/]+)", var.docker_ami_id)) ? true : false
  docker_is_image_market   = can(regex(":", var.docker_ami_id)) ? true : false
  docker_plan_given        = var.docker_base_ami_plan == "" ? "" : var.docker_base_ami_plan
}

#For docker image
data "azurerm_shared_image" "docker_shared_image" {
  count               = local.docker_is_image_gallery ? 1 : 0
  name                = element(split("/", var.docker_ami_id), 10)
  gallery_name        = element(split("/", var.docker_ami_id), 8)
  resource_group_name = element(split("/", var.docker_ami_id), 4)
}

resource "azurerm_virtual_machine" "docker_image" {
  name                          = "${var.imagetag}-dockerimage"
  resource_group_name           = var.resource_group
  location                      = var.region
  vm_size                       = var.image_instance_type == "" ? "Standard_DS1_v2" : var.instance_type[var.model]
  delete_os_disk_on_termination = true

  network_interface_ids = [azurerm_network_interface.docker_image_public.id]

  os_profile {
    computer_name  = "${var.imagetag}-dockerimage"
    admin_username = var.ami_user
    custom_data = data.template_cloudinit_config.config.rendered
  }


  dynamic "storage_image_reference" {
    for_each = local.docker_ami_name != null ? [1] : []

    content {
      id        = !local.docker_is_image_market ? local.docker_ami_name : null
      publisher = local.docker_is_image_market ? element(split(":", local.docker_ami_name), 0) : null
      offer     = local.docker_is_image_market ? element(split(":", local.docker_ami_name), 1) : null
      sku       = local.docker_is_image_market ? element(split(":", local.docker_ami_name), 2) : null
      version   = local.docker_is_image_market ? element(split(":", local.docker_ami_name), 3) : null
    }
  }
  plan {
    name      = local.docker_plan_given == "" ? local.docker_is_image_gallery ? data.azurerm_shared_image.docker_shared_image[0].purchase_plan[0].name : element(split(":", var.docker_ami_id), 2) : element(split(":", local.docker_plan_given), 0)
    product   = local.docker_plan_given == "" ? local.docker_is_image_gallery ? data.azurerm_shared_image.docker_shared_image[0].purchase_plan[0].product : element(split(":", var.docker_ami_id), 1) : element(split(":", local.docker_plan_given), 1)
    publisher = local.docker_plan_given == "" ? local.docker_is_image_gallery ? data.azurerm_shared_image.docker_shared_image[0].purchase_plan[0].publisher : element(split(":", var.docker_ami_id), 0) : element(split(":", local.docker_plan_given), 2)
  }

  os_profile_linux_config {
    disable_password_authentication = true
    ssh_keys {
      path     = "/home/${var.ami_user}/.ssh/authorized_keys"
      key_data = file("${var.project_public_key}")
    }
  }

  storage_os_disk {
    name              = "${var.imagetag}-osdisk-dockerimage"
    caching           = "ReadWrite"
    create_option     = "FromImage"
    managed_disk_type = "Standard_LRS"
    disk_size_gb      = var.image_root_volume_size
  }
}

data "azurerm_managed_disk" "os_disk" {
  depends_on          = [azurerm_virtual_machine.docker_image]
  name                = "${var.imagetag}-osdisk-dockerimage"
  resource_group_name = var.resource_group
}

resource "azurerm_managed_disk" "docker_data_disk" {
  depends_on           = [azurerm_virtual_machine.docker_image]
  name                 = "${var.imagetag}-spillbox-datadisk"
  resource_group_name  = var.resource_group
  location             = var.region
  storage_account_type = "Standard_LRS"
  create_option        = "Copy"
  source_resource_id   = data.azurerm_managed_disk.os_disk.id
  disk_size_gb         = var.image_root_volume_size
}

resource "azurerm_virtual_machine_data_disk_attachment" "docker_data_attach" {
  depends_on         = [azurerm_managed_disk.docker_data_disk]
  managed_disk_id    = azurerm_managed_disk.docker_data_disk.id
  virtual_machine_id = azurerm_virtual_machine.docker_image.id
  lun               = 1  # Logical Unit Number (should be unique per VM)
  caching           = "ReadWrite"
}

resource "azurerm_public_ip" "docker_image_public" {
  name                = "${var.imagetag}-dockerimage-public"
  location            = var.region
  resource_group_name = var.resource_group
  allocation_method   = "Static"
}

resource "azurerm_network_interface" "docker_image_public" {
  name                = "${var.imagetag}-dockerimage-public"
  location            = var.region
  resource_group_name = var.resource_group

  ip_configuration {
    name                          = "${var.imagetag}-dockerimage-pubconf"
    subnet_id                     = var.subnet_id
    private_ip_address_allocation = "Dynamic"
  }
}

resource "null_resource" "build_docker_image" {
  depends_on    = [azurerm_virtual_machine_data_disk_attachment.docker_data_attach, azurerm_virtual_machine.docker_image]
  connection {
    user        = var.ami_user
    private_key = file(var.project_private_key)
    host        = element(flatten(azurerm_network_interface.docker_image_public.*.private_ip_address), 0)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }
  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }
  provisioner "remote-exec" {
    inline = ["chmod 775 /tmp/install/docker_build.sh ; /tmp/install/docker_build.sh '${var.webserverip}' '${var.service_userid}'"]
  }
}

resource "null_resource" "setup_docker_image_webserver_local" {
  depends_on = [null_resource.build_docker_image]
  provisioner "local-exec" {
     command = "chmod 755 ${path.module}/install/docker_setup.sh ; ${path.module}/install/docker_setup.sh '${var.webserverip}' '${var.service_userid}'"
  }
}
