# Copyright © 2018-2025, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

terraform {
  required_version = ">= 1.10.4, < 1.10.5"
  required_providers {
    google = {
        version = "6.19.0"
    }
    null = {
        version = "3.2.3"
    }
    template = {
        version = "2.2.0"
    }
  }
}

provider "google" {
  region      = var.region
  project     = var.project
  credentials = var.google_credentials
}

resource "google_compute_instance" "docker_image" {
  project      = var.project
  zone         = var.zone
  machine_type = var.image_instance_type == "" ? "n1-standard-2" : var.instance_type[var.model]

  boot_disk {
    kms_key_self_link = var.kms_key_id == "" ? null : var.kms_key_id
    initialize_params {
      image = var.docker_ami_id == "" ? "centos-7-v20240213" : var.docker_ami_id
      size  = var.image_root_volume_size
    }
  }
  metadata = {
    serial-port-enable      = "true"
    block-project-ssh-keys  = "true"
    enable-guest-attributes = "TRUE"
    ssh-keys                = <<EOF
${var.ami_user}:${replace(file(var.project_public_key), "\n", "")} ${var.ami_user}
EOF

  }

  network_interface {
    subnetwork         = var.subnet_id
    subnetwork_project = var.project
    dynamic "access_config" {
      for_each = var.disable_compute_public_ips == true ? [] : [1]
      content {}
    }
  }
  name                    = "spillbox-docker"
#  metadata_startup_script = data.template_cloudinit_config.config.rendered
}

resource "google_compute_disk" "spillbox_disk" {
  name  = "spillbox-disk"
  type  = "pd-standard"  # Use "pd-standard", "pd-balanced", or "pd-extreme" as needed
  zone  = var.zone
  size  = var.image_root_volume_size
  image = var.docker_ami_id
  disk_encryption_key {
    kms_key_self_link = var.kms_key_id == "" ? null : var.kms_key_id
  }
}
resource "google_compute_attached_disk" "spillbox_disk_attachment" {
  disk     = google_compute_disk.spillbox_disk.id
  instance = google_compute_instance.docker_image.id
  zone     = var.zone

  lifecycle {
    prevent_destroy = false  # Allows Terraform to detach the disk when needed
  }
}

resource "null_resource" "build_docker_image" {
  depends_on    = [google_compute_attached_disk.spillbox_disk_attachment]
  connection {
    user        = var.ami_user
    private_key = file(var.project_private_key)
    host        = element(flatten(google_compute_instance.docker_image.*.network_interface.0.network_ip), 0)
    port        = var.cloud_ssh_port
    timeout     = "600s"
  }

  provisioner "file" {
    source      = "${path.module}/install"
    destination = "/tmp"
  }

  provisioner "remote-exec" {
    inline = [ "chmod 755 /tmp/install/docker_build.sh ; /tmp/install/docker_build.sh '${var.webserverip}' '${var.service_userid}' "]
  }
}

resource "null_resource" "setup_docker_image_webserver_local" {
  depends_on = [null_resource.build_docker_image]
  provisioner "local-exec" {
     command = "chmod 755 ${path.module}/install/docker_setup.sh ; ${path.module}/install/docker_setup.sh '${var.webserverip}' '${var.service_userid}'"
  }
}
