#!/bin/bash -f

# Copyright © 2018-2023, Spillbox. All rights reserved.
# Property of Spillbox Inc.
# Any kind of copying or producing derivative work is prohibited without
# explicit written consent from Spillbox Inc.

usage()
{
    echo "Usage : $0 [project|-restart]" >&2
}

umask 022

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
cd $DIR
RETVAL=$? 
if test ${RETVAL} -ne 0 ; then
    echo "cannot change directory $DIR" >&2
    exit $RETVAL
fi

project=$(hostname)
hostip=$(hostname -i)
restart=0
user=$(whoami)
if [ "$user" == "root" ] ; then
    echo "Please do not run as root" >&2
    exit 3
fi

if test "$#" -eq 1; then
    if [ "$1" == "-restart" ]; then
        restart=1
    else
        project=$1
    fi
fi
if test "$#" -gt 1; then
    usage;
    exit 3
fi

if test $restart -eq 0; then
    curl -o spill-dc.tar.gz https://s3-us-west-1.amazonaws.com/spillbox.io-public/spill-dc/v1.3/spill-dc.tar.gz
    tar -xzf spill-dc.tar.gz
    find "$DIR/etc" -name "*.py" -exec chmod 700 {} \;
fi

if [ ! -f terraform/etc/spillbox_config ]; then
    echo "Please copy terraform/etc/spillbox_config.example to terraform/etc/spillbox_config and edit" >&2
    exit 0
fi

terraform/run_server.sh -kill $project
terraform/server_image.sh destroy
terraform/server_image.sh build
terraform/run_server.sh $project

spillbox_config_file=terraform/etc/spillbox_config
if [ -f  $spillbox_config_file ]; then
    . $spillbox_config_file 
else
    echo missing config file : $spillbox_config_file >&2
    exit 3
fi

if [ "$docker_registry_port" == "" ]; then
    docker_registry_port=5000
fi

if [ "$disable_registry_server" == "false" ]; then
    if [ "$disable_server_image" == "true" ]; then
        if [ "$docker_base_image" != "" ]; then
            docker tag $docker_base_image ${hostip}:${docker_registry_port}/$docker_base_image
            docker push ${hostip}:${docker_registry_port}/$docker_base_image
        fi
    fi
fi

if test $restart -eq 0; then
    cat > splserver.service <<EOF

[Unit]
Description=Spillbox webserver
After=network.target auditd.service

[Service]
ExecStart=$DIR/upgrade.sh -restart
ExecReload=/bin/kill -HUP $MAINPID
KillMode=process
Restart=on-failure
RestartPreventExitStatus=255
Type=simple
User=$user
TimeoutSec=0
RestartSec=2
StartLimitBurst=0
StartLimitInterval=60s
WorkingDirectory=$DIR

[Install]
WantedBy=multi-user.target
Alias=splserver.service

EOF

echo "Please add $DIR/splserver.service to system service at OS restart:" 
echo "sudo cp $DIR/splserver.service /etc/systemd/system/."
echo "sudo systemctl daemon-reload"
echo "sudo systemctl restart   splserver"

fi

